<?php

namespace App\Services\Store;

use App\Models\Statistic;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Morilog\Jalali\Jalalian;

class DashboardService
{
    /** get sales amount */
    public function getSalesAmount()
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        $todayItems = $store->consignmentItems()->where(function ($query) {
            $query->orWhere('status', 'sent')->orWhere('status', 'delivered');
        })->whereDate('created_at', '>=', Carbon::now()->subDay(1))->get()->sum('get_shares.seller');
        $lastMonthItems = $store->consignmentItems()->where(function ($query) {
            $query->orWhere('status', 'sent')->orWhere('status', 'delivered');
        })->whereDate('created_at', '>=', Carbon::now()->subDay(30))->get()->sum('get_shares.seller');
        $totalItems = $store->consignmentItems()->where(function ($query) {
            $query->orWhere('status', 'sent')->orWhere('status', 'delivered');
        })->get()->sum('get_shares.seller');

        $sales = [
            'today_sale' => $todayItems,
            'month_sales' => $lastMonthItems,
            'total_sales' => $totalItems,
        ];

        return $sales;
    }

    /** get sales amount */
    public function getSalesChart($subDays)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        // past {$subDays} days period
        $period = CarbonPeriod::create(Carbon::now()->subDays($subDays), Carbon::now());

        // chart date labels
        $dates = [];

        // chart data
        $counts = [];
        $amounts = [];

        // Iterate over the period
        foreach ($period as $date) {
            // date labels
            $dates[] = Jalalian::forge($date)->format('m-d');

            // order data
            $counts[] = $store->consignmentItems()->where(function ($query) {
                $query->orWhere('status', 'sent')->orWhere('status', 'delivered');
            })->whereDate('created_at', $date)->count();
            $amounts[] = $store->consignmentItems()->where(function ($query) {
                $query->orWhere('status', 'sent')->orWhere('status', 'delivered');
            })->whereDate('created_at', $date)->get()->sum('get_shares.seller');
        }

        return [
            'dates' => $dates,
            'counts' => $counts,
            'amounts' => $amounts,
        ];
    }

    /** get sales amount */
    public function getStatisticsChart($subDays)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        // past {$subDays} days period
        $period = CarbonPeriod::create(Carbon::now()->subDays($subDays), Carbon::now());

        // chart date labels
        $dates = [];

        // chart data
        $views = [];

        // Iterate over the period
        foreach ($period as $date) {
            // date labels
            $dates[] = Jalalian::forge($date)->format('m-d');

            // order data
            $views[] = Statistic::where('url', 'like', "%store/{$store->uuid}%")->whereDate('created_at', $date)->count();
        }

        return [
            'total' => $store->view_count,
            'dates' => $dates,
            'views' => $views,
        ];
    }

    /** get latest consignments */
    public function getLatestConsignments($count)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        $consignments =  $store->consignments()->where('status', '!=', 'created')
            ->orderByRaw("FIELD(status , 'awaiting_seller') desc")
            ->orderBy('created_at', 'desc')
            ->take($count)
            ->get()
            ->map(fn($consignment) => [
                'id' => $consignment->id,
                'uuid' => $consignment->uuid,
                'status' => $consignment->status,
                'jalali_created_at' => $consignment->jalali_created_at,
            ]);

        return $consignments;
    }
}
