<?php

namespace App\Services\Main\Products;

use App\Models\Brand;
use App\Models\Color;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductTag;
use App\Models\Search;
use App\Models\Setting;
use App\Models\Store;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class ProductsArchiveService
{
    /** seo meta data */
    public array $seo = [
        'schema' => null,
        'site_name' => null,
        'title' => null,
        'store_name' => null,
        'description' => null,
        'keywords' => null,
        'canonical' => null,
        'author' => null,
        'image' => null,
        'type' => null,
    ];

    /** url query keywords */
    public array $keywords = [
        'page' => null,
        'search' => [
            'query' => null,
            'title' => null,
        ],
        'cat' => [
            'query' => null,
            'title' => null,
            'content' => null,
        ],
        'tag' => [
            'query' => null,
            'title' => null,
        ],
        'color' => [
            'query' => null,
            'title' => null,
        ],
        'brand' => [
            'query' => null,
            'title' => null,
        ],
        'price' => null,
        'available' => null,
        'discount' => null,
    ];

    public function __construct()
    {
        $shopTitle = Cache::remember('setting_shop_title', now()->addMonth(), function () {
            return Setting::where('key', 'shop_title')->first() != null ? Setting::where('key', 'shop_title')->first()->value : null;
        });

        // set site name
        $this->seo['site_name'] = $shopTitle;

        // set default title
        $this->seo['title'] = __('messages.word.products');
    }

    /** get product list */
    public function getProducts(Request $request, Store $store = null)
    {
        /** @var LengthAwarePaginator */
        if ($store != null) {
            $this->seo['description'] = $store->bio != null ? $store->bio : '';
            $products = $store->products()->where('status', 'publish');

            $this->seo['store_name'] = $store->name;
            $store->update([
                'view_count' => $store->view_count + 1
            ]);
        } else {
            $products = Product::query()->where('status', 'publish');
        }

        // filters
        $products = $this->productFilters($request, $products);

        // search
        $products = $this->productSearch($request, $products);

        // lists
        $products = $this->productLists($request, $products);

        // categories
        $products = $this->productCategories($request, $products);

        // tags
        $products = $this->productTags($request, $products);

        // color
        $products = $this->productColors($request, $products);

        // brand
        $products = $this->productBrands($request, $products);

        // order by with inventory
        $products = $products
            ->with(['inventories' => function ($q) {
                $q->select('product_id', 'count', 'status');
            }])
            ->orderByRaw("
            EXISTS (
                SELECT 1 FROM inventories
                WHERE inventories.product_id = products.id
                AND inventories.status = 'publish'
                AND inventories.count > 0
            ) DESC
            ")->orderBy('created_at', 'desc');

        // paginate products
        $products = $products->paginate(32)->through(fn($product) => [
            'id' => $product->id,
            'title' => $product->title,
            'slug' => $product->slug,
            'get_images' => $product->get_images,
            'comments_info' => $product->comments_info,
            'in_stock_status' => $product->in_stock_status,
            'inventory' => $product->inventory,
            'best_normal_price' => $product->best_normal_price,
            'unique_color' => $product->unique_color,
            'best_price' => $product->best_price,
            'is_vip' => $product->is_vip,
            'brand' => $product->get_simple_brand,
            'tags' => $product->productTags->map(fn($tag) => [
                'id' => $tag->id,
                'title' => $tag->title,
                'slug' => $tag->slug,
            ])
        ]);

        // get best prices
        $bestPrice = $this->getBestPrice($products);

        // get best prices
        $allTags = $this->getAllTags($products);

        // sorts
        $products = $this->productSorts($request, $products);

        // handle pagination with query strings
        $products->withQueryString();

        // set seo product archive schema
        $this->seo['schema'] = $this->productListSchema($products);

        // set page request
        $this->setCommonRequestKeywords($request);

        // update seo title
        if ($store == null) {
            $shopTitle = Cache::remember('setting_shop_title', now()->addMonth(), function () {
                return Setting::where('key', 'shop_title')->first() != null ? Setting::where('key', 'shop_title')->first()->value : null;
            });
            $this->seo['title'] = $this->seo['title'] . ' - ' . $shopTitle;
        }

        // return results
        return [
            'products' => $products,
            'brands' => $this->getProductBrands($products),
            'keywords' => $this->keywords,
            'seo' => $this->seo,
            'price' => $bestPrice,
            'tags' => $allTags,
        ];
    }

    /** product filters */
    public function productFilters(Request $request, $products)
    {
        $products->when($request->input('latest'), function ($query, $latest) {
            if ($latest) {
                $query->orderBy('created_at', 'desc');

                // seo
                $this->seo['title'] = __('messages.sentence.latest_products');
            }
        })->when($request->input('most_viewed'), function ($query, $most_viewed) {
            if ($most_viewed) {
                $query->orderBy('view_count', 'desc');

                // seo
                $this->seo['title'] = __('messages.sentence.most_viewed_products');
            }
        })->when($request->input('popular'), function ($query, $popular) {
            if ($popular) {
                $query->withCount(['productComments as comment_rating_average' => function ($query) {
                    $query->select(DB::raw('avg(rating)'));
                }])->orderBy('comment_rating_average', 'desc');

                // seo
                $this->seo['title'] = __('messages.sentence.popular_products');
            }
        })->when($request->input('most_sale'), function ($query, $most_sale) {
            if ($most_sale) {
                $query->withCount(['consignmentItems' => function ($query) {
                    $query->where(function ($query) {
                        $query->where('status', 'sent')->orWhere('status', 'delivered');
                    })->select(DB::raw('SUM(count)'));
                }])->orderBy('consignment_items_count', 'desc');

                // seo
                $this->seo['title'] = __('messages.sentence.most_sale_products');
            }
        })->when($request->input('available'), function ($query, $available) {
            if ($available) {
                $query->whereHas('inventories', function ($query) {
                    $query->where('status', 'publish')->where('count', '>', 0);
                });
            }
        })->when($request->input('discount'), function ($query, $discount) {
            if ($discount) {
                $query->whereHas('inventories', function ($query) {
                    $query->where('status', 'publish')->where('count', '>', 0)->where('discount_price', '!=', null);
                });

                // seo
                $this->seo['title'] = __('messages.sentence.discount_products');
            }
        });

        return $products;
    }

    /** product search */
    public function productSearch(Request $request, $products)
    {
        $search = $request->input('search');
        if (!$search) {
            return $products;
        }

        $search = trim($search);
        $search = preg_replace('/\s+/', ' ', $search);

        $productsCollection = $products->get();

        $filtered = $productsCollection->filter(function ($product) use ($search) {
            $title = trim(preg_replace('/\s+/', ' ', $product->title));

            $cleanContent = strip_tags($product->content);
            $cleanContent = str_replace(["\xC2\xA0", '&nbsp;'], ' ', $cleanContent);
            $cleanContent = preg_replace('/\s+/', ' ', $cleanContent);
            $cleanContent = trim($cleanContent);

            return mb_stripos($title, $search) !== false
                || mb_stripos($cleanContent, $search) !== false;
        });

        $products = Product::query()->whereIn('id', $filtered->pluck('id'));

        // seo
        $this->seo['title'] = $search;

        // keywords
        $this->keywords['search']['query'] = $search;
        $this->keywords['search']['title'] = $search;

        return $products;
    }

    /** product lists */
    public function productLists(Request $request, $products)
    {
        $products->when($request->input('random'), function ($query, $random) {
            if ($random) {
                $query->inRandomOrder();

                // seo
                $this->seo['title'] = __('messages.sentence.random_products');
            }
        });

        return $products;
    }

    /** product categories */
    public function productCategories(Request $request, $products)
    {
        $products->when($request->input('cat'), function ($query, $cat) use ($request) {
            $category = ProductCategory::where('status', 'publish')->where('slug', $cat)->first();
            if ($category != null) {
                $query->where(function ($query) use ($category) {
                    $query->where('category_id', $category->id);

                    if ($category->childs != null) {
                        foreach ($category->childs as $child) {
                            $query->orWhere('category_id', $child['id']);

                            if ($child['childs'] != null) {
                                foreach ($child['childs'] as $ch) {
                                    $query->orWhere('category_id', $ch['id']);
                                }
                            }
                        }
                    }
                });

                $category->update([
                    'view_count' => $category->view_count + 1
                ]);

                // seo
                $this->seo['title'] = $category->seo_title != null ? $category->seo_title : $category->title;
                $this->seo['description'] = $category->seo_description;
                $this->seo['keywords'] = $category->get_seo_keywords;
                $this->seo['image'] = $category->get_image;
                $this->seo['type'] = 'category';

                // keywords
                $this->keywords['cat']['query'] = $request->cat;
                $this->keywords['cat']['title'] = $category->title;
                $this->keywords['cat']['content'] = $category->content;
            }
        });

        return $products;
    }

    /** product tags */
    public function productTags(Request $request, $products)
    {
        $products->when($request->input('tag'), function ($query, $tag) use ($request) {
            $tag = ProductTag::where('status', 'publish')->where('slug', $tag)->first();
            if ($tag != null) {
                $query->whereHas('productTags', function ($query) use ($tag) {
                    $query->where('slug', $tag->slug);
                });

                $tag->update([
                    'view_count' => $tag->view_count + 1
                ]);

                // seo
                $this->seo['title'] = $tag->title;
                $this->seo['type'] = 'tag';

                // keywords
                $this->keywords['tag']['query'] = $request->tag;
                $this->keywords['tag']['title'] = $tag->title;
            }
        });

        return $products;
    }

    /** product colors */
    public function productColors(Request $request, $products)
    {
        $products->when($request->input('color'), function ($query, $color) use ($request) {
            $color = Cache::remember('color_' . $color, now()->addMonth(), function () use ($color) {
                return Color::where('color', $color)->first();
            });
            if ($color != null) {
                $query->whereHas('inventories', function ($query) use ($color) {
                    $query->whereHas('inventoryProps', function ($query) use ($color) {
                        $query->where('type', 'color')->where('value', 'like',  "%{$color->color}%");
                    });
                });

                // keywords
                $this->keywords['color']['query'] = $request->color;
                $this->keywords['color']['title'] = $color->label;
            }
        });

        return $products;
    }

    /** product brands */
    public function productBrands(Request $request, $products)
    {
        $products->when($request->input('brand'), function ($query, $brand) use ($request) {
            $brand = Brand::where('slug', $brand)->first();
            if ($brand != null) {
                $query->where('brand_id', $brand->id);

                // seo
                $this->seo['title'] = $brand->seo_title != null ? __('messages.word.brand') . ' ' . $brand->seo_title : __('messages.word.brand') . ' ' . $brand->title;
                $this->seo['description'] = $brand->seo_description;
                $this->seo['keywords'] = $brand->get_seo_keywords;
                $this->seo['image'] = $brand->get_logo;
                $this->seo['type'] = 'brand';

                // keywords
                $this->keywords['brand']['query'] = $request->brand;
                $this->keywords['brand']['title'] = $brand->title;
            }
        });

        return $products;
    }

    /** get best price */
    public function getBestPrice($products)
    {
        $bestPrice = array_keys($products->getCollection()->groupBy(function ($data) {
            return $data['best_price'];
        })->toArray());
        $bestPrice = array_values(array_filter($bestPrice, fn($value) => !is_null($value) && $value !== ''));

        return [
            'min' => count($bestPrice) > 0 ? min($bestPrice) : 0,
            'max' => count($bestPrice) > 0 ? max($bestPrice) : 0,
        ];
    }

    /** get all tags */
    public function getAllTags($products)
    {
        $tags = [];
        foreach ($products as $product) {
            $tags = array_merge($tags, $product['tags']->toArray());
        }

        return collect($tags)->unique('title')->values()->all();
    }

    /** get best price */
    public function productSorts(Request $request, $products)
    {
        // sort by expensive
        if ($request->exists('expensive') && $request->expensive) {
            $sortedProducts = $products->getCollection()->sortByDesc(function ($product) {
                return $product['best_price'];
            })->values();
            $products->setCollection($sortedProducts);
        }

        // sort by cheapest
        if ($request->exists('cheapest') && $request->cheapest) {
            $sortedProducts = $products->getCollection()->sortBy(function ($product) {
                return $product['best_price'];
            })->values();
            $products->setCollection($sortedProducts);
        }

        // sort by price
        if ($request->exists('price') && $request->price) {
            $prices = explode('-', $request->price);

            $filteredProduct = $products->getCollection()->filter(function ($product) use ($prices) {
                return $product['best_price'] >= $prices[0] && $product['best_price'] <= $prices[1];
            })->values();
            $products->setCollection($filteredProduct);
        }

        return $products;
    }

    /** get product brands */
    public function getProductBrands($products)
    {
        $brands = [];
        foreach ($products as $product) {
            if ($product['brand'] != null) {
                $brands[$product['brand']['uuid']] = $product['brand'];
            }
        }

        return $brands;
    }

    /** Product List Schema */
    public function productListSchema($products)
    {
        if (count($products) == 0)
            return null;

        $schema = [
            "@context" => "https://schema.org",
            "@type" => "ItemList",
            "itemListElement" => [],
        ];

        $finalProduct = [];
        for ($i = 0; $i < 24; $i++) {
            if (isset($products[$i]))
                $finalProduct[] = $products[$i];
        }

        foreach ($finalProduct as $key => $product) {
            $schema["itemListElement"][] = [
                "@type" => "ListItem",
                "position" => $key + 1,
                "item" => [
                    "@type" => "Product",
                    "name" => $product['title'],
                    "url" => urldecode(route("main.products.show", $product['slug'])),
                    "image" => urldecode($product['get_images'][0]['link']),
                    "aggregateRating" => [
                        "@type" => "AggregateRating",
                        "ratingValue" => (float)$product['comments_info']['rating'],
                        "reviewCount" => !empty($product['comments_info']['count']) ? (int)str_replace(',', '', $product['comments_info']['count']) : 1,
                        "bestRating" => 5,
                        "worstRating" => 1
                    ],
                ]
            ];
        }

        return json_encode($schema, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    /** set common request keywords */
    public function setCommonRequestKeywords(Request $request)
    {
        $this->keywords['page'] = $request->page;
        $this->keywords['price'] = [
            'min' => $request->exists('price') && $request->exists('price') != null ? explode('-', $request->price)[0] : null,
            'max' => $request->exists('price') && $request->exists('price') != null ? explode('-', $request->price)[1] : null,
        ];
        $this->keywords['available'] = $request->available;
        $this->keywords['discount'] = $request->discount;
    }

    /** add search to history */
    public function addSearchToHistory(Request $request)
    {
        if ($request->exists('search') && $request->search != null && $request->search != "") {
            if (str_word_count($request->search) > 4 || mb_strlen($request->search) > 60) {
                return; // ذخیره نکن
            }

            $lastSearchList = $request->session()->get('last_search_list', []);
            if (count($lastSearchList) > 6) {
                $lastSearchList = array_slice($lastSearchList, -6, 6);
            }
            if (isset($lastSearchList[array_search($request->search, $lastSearchList)])) {
                unset($lastSearchList[array_search($request->search, $lastSearchList)]);
            }
            $lastSearchList[] = $request->search;
            session(['last_search_list' => $lastSearchList]);

            /** add search to db */
            $userID = auth()->check() ? auth()->user()->id : null;
            Search::create([
                'user_id' => $userID,
                'keyword' => $request->search
            ]);
        }
    }
}
