<?php

namespace App\Services\Database;

use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class SchemaUpdater
{
    /**
     * Check and update table schema by adding a new column if it does not exist.
     *
     * Supports additional attributes like nullable, default, unique, index, primary,
     * and also allows defining foreign key constraints.
     *
     * @param string $table        The name of the table.
     * @param string $column       The name of the column to add.
     * @param string $type         The type of the column (e.g., string, unsignedBigInteger).
     * @param array $attributes    Column attributes such as 'nullable', 'default', 'unique', 'index', etc.
     * @param string|null $afterColumn Optional column name to position the new column after.
     * @param array|null $foreign  Optional foreign key definition with keys: 'references', 'on', 'onDelete', 'onUpdate'.
     *
     * @return bool                True if the column was added or already exists, false on failure.
     */
    public function addColumnToTable(
        string $table,
        string $column,
        string $type = 'string',
        array $attributes = [],
        ?string $afterColumn = null,
        ?array $foreign = null // ← افزودن پارامتر foreign
    ): bool {
        $cacheKey = "schema_add_column_{$table}_{$column}";
        if (Cache::has($cacheKey)) {
            return true;
        }

        if (!Schema::hasColumn($table, $column)) {
            try {
                Schema::table($table, function (Blueprint $tableBlueprint) use ($table, $column, $type, $attributes, $afterColumn, $foreign) {
                    $columnDefinition = $type === 'string' && isset($attributes['length'])
                        ? $tableBlueprint->string($column, $attributes['length'])
                        : $tableBlueprint->$type($column);

                    if (!empty($attributes['nullable'])) {
                        $columnDefinition->nullable();
                    }

                    if (array_key_exists('default', $attributes)) {
                        $columnDefinition->default($attributes['default']);
                    }

                    if (!empty($attributes['unique'])) {
                        $tableBlueprint->unique($column);
                    }

                    if (!empty($attributes['index'])) {
                        $tableBlueprint->index($column);
                    }

                    if (!empty($attributes['primary'])) {
                        $tableBlueprint->primary($column);
                    }

                    if ($afterColumn !== null && Schema::hasColumn($table, $afterColumn)) {
                        $columnDefinition->after($afterColumn);
                    }

                    // foreign key
                    if (is_array($foreign) && isset($foreign['references'], $foreign['on'])) {
                        $foreignColumn = $tableBlueprint->foreign($column)
                            ->references($foreign['references'])
                            ->on($foreign['on']);

                        if (!empty($foreign['onDelete'])) {
                            $foreignColumn->onDelete($foreign['onDelete']);
                        }
                        if (!empty($foreign['onUpdate'])) {
                            $foreignColumn->onUpdate($foreign['onUpdate']);
                        }
                    }
                });

                Cache::forever($cacheKey, true);
                return true;
            } catch (\Exception $e) {
                return false;
            }
        }

        Cache::forever($cacheKey, true);
        return true;
    }

    /**
     * Create a new table if it doesn't exist.
     *
     * @param string $table
     * @param array $columns
     * @return bool
     */
    public function addTable(string $table, array $columns): bool
    {
        $cacheKey = "schema_add_table_{$table}";
        if (Cache::has($cacheKey)) {
            return true;
        }

        if (!Schema::hasTable($table)) {
            try {
                Schema::create($table, function (Blueprint $tableBlueprint) use ($columns) {
                    $tableBlueprint->id();
                    $foreignKeys = [];

                    foreach ($columns as $column) {
                        $name = $column['name'] ?? null;
                        $type = $column['type'] ?? 'string';
                        $attributes = $column['attributes'] ?? [];
                        $foreign = $column['foreign'] ?? null;

                        if (!$name) {
                            continue;
                        }

                        // Create column
                        $columnDefinition = $type === 'string' && isset($attributes['length'])
                            ? $tableBlueprint->string($name, $attributes['length'])
                            : $tableBlueprint->$type($name);

                        if (!empty($attributes['nullable'])) {
                            $columnDefinition->nullable();
                        }

                        if (array_key_exists('default', $attributes)) {
                            $columnDefinition->default($attributes['default']);
                        }

                        // Constraints that should be applied on the table (not column object)
                        if (!empty($attributes['unique'])) {
                            $tableBlueprint->unique($name);
                        }

                        if (!empty($attributes['index'])) {
                            $tableBlueprint->index($name);
                        }

                        if (!empty($attributes['primary'])) {
                            $tableBlueprint->primary($name);
                        }

                        // Save foreign keys for later
                        if (is_array($foreign) && isset($foreign['references'], $foreign['on'])) {
                            $foreignKeys[] = [
                                'column' => $name,
                                'references' => $foreign['references'],
                                'on' => $foreign['on'],
                                'onDelete' => $foreign['onDelete'] ?? null,
                                'onUpdate' => $foreign['onUpdate'] ?? null,
                            ];
                        }
                    }

                    $tableBlueprint->timestamps();

                    // Add foreign keys
                    foreach ($foreignKeys as $fk) {
                        $foreignColumn = $tableBlueprint->foreign($fk['column'])
                            ->references($fk['references'])
                            ->on($fk['on']);

                        if ($fk['onDelete']) {
                            $foreignColumn->onDelete($fk['onDelete']);
                        }
                        if ($fk['onUpdate']) {
                            $foreignColumn->onUpdate($fk['onUpdate']);
                        }
                    }
                });

                Cache::forever($cacheKey, true);
                return true;
            } catch (\Exception $e) {
                return false;
            }
        }

        Cache::forever($cacheKey, true);
        return true;
    }

    /**
     * Edit an existing column in a table.
     *
     * @param string $table
     * @param string $column
     * @param array $updates
     * @return bool
     */
    public function EditColumnInTable(string $table, string $column, array $updates): bool
    {
        // Check if schema update has already been performed
        $cacheKey = "schema_edit_column_{$table}_{$column}_" . md5(json_encode($updates));
        if (Cache::has($cacheKey)) {
            return true;
        }

        // Check if column exists
        if (Schema::hasColumn($table, $column)) {
            try {
                Schema::table($table, function (Blueprint $table) use ($column, $updates) {
                    $newColumnName = $updates['new_name'] ?? $column;
                    $type = $updates['type'] ?? null;
                    $attributes = $updates['attributes'] ?? [];

                    // Create column definition
                    if ($type) {
                        $columnDefinition = $type === 'string' && isset($attributes['length'])
                            ? $table->string($newColumnName, $attributes['length'])
                            : $table->$type($newColumnName);
                    } else {
                        // If type is not specified, use change() to modify existing column
                        $columnDefinition = $table->string($newColumnName)->change();
                    }

                    // Apply nullable attribute
                    if (isset($attributes['nullable'])) {
                        $columnDefinition->nullable($attributes['nullable']);
                    }

                    // Apply default value
                    if (array_key_exists('default', $attributes)) {
                        $columnDefinition->default($attributes['default']);
                    }

                    // Apply other attributes dynamically
                    foreach ($attributes as $method => $value) {
                        if (!in_array($method, ['nullable', 'default', 'length']) && method_exists($columnDefinition, $method)) {
                            $columnDefinition->$method($value);
                        }
                    }

                    // Rename column if new_name is provided and different from current name
                    if (isset($updates['new_name']) && $updates['new_name'] !== $column) {
                        $table->renameColumn($column, $updates['new_name']);
                    }

                    // Use change() to apply modifications
                    $columnDefinition->change();
                });

                // Mark as updated in cache (persists forever)
                Cache::forever($cacheKey, true);
                return true;
            } catch (\Exception $e) {
                // Log error if needed
                //app('App\Exceptions\Handler')->report($e);
                return false;
            }
        }

        // Column does not exist
        return false;
    }

    /**
     * Update a foreign key constraint to set column to NULL on parent deletion.
     *
     * This method will drop the existing foreign key on the specified column and
     * recreate it with ON DELETE SET NULL behavior.
     *
     * @param string $table        The name of the table containing the foreign key column.
     * @param string $column       The name of the foreign key column to update.
     * @param string $foreignTable The referenced parent table name.
     * @return bool                True on success, false on failure or if already applied.
     */
    public function UpdateForeignKeyToNullOnDelete(string $table, string $column, string $foreignTable): bool
    {
        // Generate a unique cache key based on input
        $cacheKey = "schema_fk_setnull_{$table}_{$column}_{$foreignTable}";

        // Check if this change has already been applied
        if (Cache::has($cacheKey)) {
            return true;
        }

        // Check if column exists before proceeding
        if (!Schema::hasColumn($table, $column)) {
            return false;
        }

        try {
            Schema::table($table, function (Blueprint $table) use ($column, $foreignTable) {
                // Drop existing foreign key
                $table->dropForeign([$column]);

                // Recreate it with ON DELETE SET NULL
                $table->foreign($column)
                    ->references('id')
                    ->on($foreignTable)
                    ->nullOnDelete();
            });

            // Mark the update as applied in cache (persists forever)
            Cache::forever($cacheKey, true);

            return true;
        } catch (\Exception $e) {
            // You can log the exception if needed
            // report($e);
            return false;
        }
    }
}
