<?php

namespace App\Services\Cart;

use App\Helpers\UUIDHelper;
use App\Models\Consignment;
use App\Models\ConsignmentItem;
use App\Models\Discount;
use App\Models\Inventory;
use App\Models\OrderGroup;
use App\Models\PaymentMeta;
use App\Models\Prefactor;
use App\Models\Setting;
use App\Models\Transaction;
use App\Models\User;
use App\Notifications\OrderRegistration;
use App\Notifications\RequestConsignment;
use App\Notifications\WalletCharge;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CartService
{
    use UUIDHelper;

    /**
     * validate payment gateway limit price
     */
    public function validatePaymentGatewayPrice($order, $count, $price)
    {
        // get order consignments
        $consignments = $order->consignments()->with('consignmentItems')->get();
        // add consignment items
        $consignmentItems = new Collection();
        foreach ($consignments as $consignment) {
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get()->map(fn($item) => [
                'safe' => $item->safe
            ]));
        }
        // get total final price
        $totalFinalPrice = $consignmentItems->sum('safe.get_full_count_price');
        // add current item prices
        $totalFinalPrice = $totalFinalPrice + ($count * $price);

        // get pg price limit
        $pgPriceLimit = Setting::where('key', 'payment_gateway_price_limit')->first() != null ? (int)str_replace(',', '', Setting::where('key', 'payment_gateway_price_limit')->first()->value) : 50000000;

        // handle payment gateway price limit
        if ($totalFinalPrice > $pgPriceLimit) {
            return $pgPriceLimit;
        }

        return false;
    }

    /**
     * activated user address
     */
    public function activateAddress($user, $address)
    {
        /** check user validation */
        if ($address->user_id != $user->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        foreach ($user->addresses()->where('id', '!=', $address->id)->get() as $adrs) {
            $adrs->update([
                'active' => false
            ]);
        }

        $address->update([
            'active' => true
        ]);
    }

    /**
     * Separation consignment by send method
     */
    public function sepratedConsignmentInformation($order, $selectedAddress, $withoutCancelled = false)
    {
        $consignments = $order->consignments()->with(['store.paymentMeta', 'consignmentItems.product', 'consignmentItems.store', 'consignmentItems.inventory'])->get();

        // main payment meta
        $paymentMeta = PaymentMeta::whereNull('store_id')->first();

        // default post ways
        $defaultPostWays = $paymentMeta?->get_post_ways ?? [];
        $defaultPostWays = array_values(array_filter($defaultPostWays, fn($way) => $way['status'] ?? false));

        $consignmentsInformation = [
            'main' => [
                'uuid' => $order->uuid,
                'store' => [
                    'id' => null,
                    'uuid' => null,
                    'name' => null,
                    'get_logo' => null,
                ],
                'items' => [],
                'totalFinalPrice' => 0,
                'totalDiscount' => 0,
                'totalPrice' => 0,
                'totalWeight' => 0,
                'get_post_ways' => $defaultPostWays,
                'send_method' => $order->get_send_method,
            ]
        ];

        foreach ($consignments as $consignment) {
            $store = $consignment->store;

            if ($store && $store->direct_post) {
                $rawItems = $consignment->consignmentItems->all();

                $items = array_filter(array_map(function ($item) use ($withoutCancelled) {
                    if ($withoutCancelled && $item->status == "cancelled") return null;

                    return [
                        'id' => $item->id,
                        'get_props' => $item->get_props,
                        'inventory_id' => $item->inventory_id,
                        'count' => $item->count,
                        'discount' => $item->discount,
                        'price' => $item->price,
                        'get_full_count_price' => $item->get_full_count_price,
                        'get_full_count_discount' => $item->get_full_count_discount,
                        'get_full_count_weight_kg' => $item->get_full_count_weight_kg,
                        'inventory' => [
                            'weight' => $item->inventory?->weight ?? 0
                        ],
                        'product' => [
                            'id' => $item->product->id,
                            'title' => $item->product->title,
                            'slug' => $item->product->slug,
                            'category' => $item->product->productCategory->title,
                            'get_images' => $item->product->get_images,
                        ],
                        'store' => $item->store ? [
                            'id' => $item->store->id,
                            'uuid' => $item->store->uuid,
                            'name' => $item->store->name,
                        ] : null,
                    ];
                }, $rawItems));

                $totalFinalPrice = array_sum(array_column($items, 'get_full_count_price'));
                $totalDiscount = array_sum(array_column($items, 'get_full_count_discount'));
                $totalWeight = array_sum(array_column($items, 'get_full_count_weight_kg'));

                $storePaymentMeta = $store->paymentMeta;
                $storePostWays = $storePaymentMeta?->get_post_ways ?? [];
                $storePostWays = array_values(array_filter($storePostWays, fn($way) => $way['status'] ?? false));

                // free send
                if ($storePaymentMeta->free_post_min_price != null && $storePaymentMeta->free_post_min_price > 0) {
                    $freeSend = $totalFinalPrice >= $storePaymentMeta->free_post_min_price;
                } else {
                    $freeSend = false;
                }

                $consignmentsInformation[] = [
                    'uuid' => $consignment->uuid,
                    'store' => [
                        'id' => $store->id,
                        'uuid' => $store->uuid,
                        'name' => $store->name,
                        'get_logo' => $store->get_logo,
                    ],
                    'items' => $items,
                    'totalFinalPrice' => $totalFinalPrice,
                    'totalDiscount' => $totalDiscount,
                    'totalPrice' => $totalFinalPrice + $totalDiscount,
                    'totalWeight' => ceil($totalWeight),
                    'get_post_ways' => $storePostWays,
                    'send_method' => $consignment->get_send_method,
                    'freeSend' => $freeSend,
                ];
            } else {
                foreach ($consignment->consignmentItems as $item) {
                    if ($withoutCancelled && $item->status === 'cancelled') continue;

                    $itemData = [
                        'id' => $item->id,
                        'get_props' => $item->get_props,
                        'inventory_id' => $item->inventory_id,
                        'count' => $item->count,
                        'discount' => $item->discount,
                        'price' => $item->price,
                        'get_full_count_price' => $item->get_full_count_price,
                        'get_full_count_discount' => $item->get_full_count_discount,
                        'get_full_count_weight_kg' => $item->get_full_count_weight_kg,
                        'inventory' => [
                            'weight' => $item->inventory?->weight ?? 0
                        ],
                        'product' => [
                            'id' => $item->product->id,
                            'title' => $item->product->title,
                            'slug' => $item->product->slug,
                            'category' => $item->product->productCategory->title,
                            'get_images' => $item->product->get_images,
                        ],
                        'store' => $item->store ? [
                            'id' => $item->store->id,
                            'uuid' => $item->store->uuid,
                            'name' => $item->store->name,
                        ] : null,
                    ];

                    $consignmentsInformation['main']['items'][] = $itemData;
                    $consignmentsInformation['main']['totalFinalPrice'] += $itemData['get_full_count_price'];
                    $consignmentsInformation['main']['totalDiscount'] += $itemData['get_full_count_discount'];
                    $consignmentsInformation['main']['totalWeight'] += $itemData['get_full_count_weight_kg'] ?? 0;
                }

                $consignmentsInformation['main']['totalPrice'] =
                    $consignmentsInformation['main']['totalFinalPrice'] +
                    $consignmentsInformation['main']['totalDiscount'];

                $consignmentsInformation['main']['totalWeight'] = ceil($consignmentsInformation['main']['totalWeight']);

                // free send
                if ($paymentMeta->free_post_min_price != null && $paymentMeta->free_post_min_price > 0) {
                    $freeSend = $consignmentsInformation['main']['totalFinalPrice'] >= $paymentMeta->free_post_min_price;
                } else {
                    $freeSend = false;
                }
                $consignmentsInformation['main']['freeSend'] = $freeSend;
            }
        }

        // remove empty consignment
        foreach ($consignmentsInformation as $key => $consignment) {
            if (count($consignment['items']) == 0) {
                unset($consignmentsInformation[$key]);
            }
        }

        // remove post ways where totalWeight > max_weight
        foreach ($consignmentsInformation as &$consignment) {
            $totalWeight = $consignment['totalWeight'] ?? 0;

            if (!empty($consignment['get_post_ways'])) {
                $consignment['get_post_ways'] = array_values(array_filter(
                    $consignment['get_post_ways'],
                    function ($way) use ($totalWeight) {
                        $maxWeight = $way['max_weight'] ?? null;

                        // remove method if max_weight != null
                        if ($maxWeight !== null) {
                            return $totalWeight <= (float)$maxWeight;
                        }

                        // skip if max_weight is null
                        return true;
                    }
                ));
            }
        }
        unset($consignment);

        // validate peyk: disable all first
        foreach ($consignmentsInformation as &$consignment) {
            if (!empty($consignment['get_post_ways'])) {
                foreach ($consignment['get_post_ways'] as &$way) {
                    if (($way['uuid'] ?? null) === 'peyk') {
                        $way['status'] = false;
                    }
                }
                unset($way);
            }
        }
        unset($consignment);

        // active peyk for true address
        $city = null;
        if ($selectedAddress != null && $selectedAddress['type'] === 'user') {
            $city = $selectedAddress['address']['city'] ?? null;
        }
        if ($city) {
            foreach ($consignmentsInformation as &$consignment) {
                if (!empty($consignment['get_post_ways'])) {
                    foreach ($consignment['get_post_ways'] as &$way) {
                        if (($way['uuid'] ?? null) === 'peyk') {
                            $places = $way['places'] ?? [];
                            foreach ($places as $place) {
                                if (Str::contains($place, $city)) {
                                    $way['status'] = true;
                                    break;
                                }
                            }
                        }
                    }
                    unset($way);
                }
            }
            unset($consignment);
        }

        // delete all status=false send methods
        foreach ($consignmentsInformation as &$consignment) {
            if (!empty($consignment['get_post_ways'])) {
                $consignment['get_post_ways'] = array_values(array_filter(
                    $consignment['get_post_ways'],
                    fn($way) => $way['status'] ?? false
                ));
            }
        }
        unset($consignment);

        // calculate sendPrice based on send_method and totalWeight
        foreach ($consignmentsInformation as &$consignment) {
            $sendMethod = $consignment['send_method'] ?? null;
            $totalWeight = $consignment['totalWeight'] ?? 0;

            if ($consignment['freeSend']) {
                $consignment['sendPrice'] = 0;
            } else {
                if (!empty($sendMethod) && ($sendMethod['type'] ?? null) === 'post') {
                    $excessWeightStr = $sendMethod['item']['price']['excessWeight'] ?? '0';
                    $basePriceStr = $sendMethod['item']['price']['base'] ?? '0';

                    $excessWeight = (int) str_replace(',', '', $excessWeightStr);
                    $basePrice = (int) str_replace(',', '', $basePriceStr);

                    $additionalPrice = 0;
                    if ($totalWeight > 0) {
                        $finalTotalWeight = max($totalWeight - 1, 0);
                        $additionalPrice = $finalTotalWeight * $excessWeight;
                    }

                    $consignment['sendPrice'] = $basePrice + $additionalPrice;
                } else {
                    $consignment['sendPrice'] = 0;
                }
            }
        }
        unset($consignment);

        // افزودن sendPrice به totalFinalPrice هر مرسوله
        foreach ($consignmentsInformation as &$consignment) {
            if (isset($consignment['sendPrice']) && isset($consignment['totalFinalPrice'])) {
                $consignment['totalFinalPrice'] += $consignment['sendPrice'];
            }
        }
        unset($consignment);

        // تقسیم تخفیف بین مرسوله‌ها بر اساس totalFinalPrice شامل sendPrice
        $discountData = $order->discount != null ? unserialize($order->discount) : null;
        if ($discountData != null && $discountData['amount'] != null) {
            $discount = (int) str_replace(',', '', $discountData['amount']);
            if ($discount > 0) {
                // فقط مرسوله‌هایی که مبلغ دارند
                $discountableConsignments = array_filter(
                    $consignmentsInformation,
                    fn($c) => ($c['totalFinalPrice'] ?? 0) > 0
                );

                // پایه تقسیم: جمع totalFinalPrice هر مرسوله (شامل sendPrice)
                $totalPriceForDiscount = array_sum(array_map(fn($c) => $c['totalFinalPrice'], $discountableConsignments));

                if ($totalPriceForDiscount > 0) {
                    $sharedDiscountSum = 0;
                    $lastKey = array_key_last($discountableConsignments);

                    foreach ($consignmentsInformation as $key => &$consignment) {
                        $total = $consignment['totalFinalPrice'] ?? 0;

                        if ($total > 0) {
                            // سهم نسبی مرسوله از کل مبلغ
                            $shareRatio = $total / $totalPriceForDiscount;
                            $sharedDiscount = (int) round($shareRatio * $discount);

                            // آخرین مرسوله باقیمانده‌ی تخفیف را می‌گیرد
                            if ($key === $lastKey) {
                                $sharedDiscount = $discount - $sharedDiscountSum;
                            }

                            $sharedDiscountSum += $sharedDiscount;

                            // کسر تخفیف از totalFinalPrice و ذخیره couponDiscountPrice
                            $consignment['couponDiscountPrice'] = $sharedDiscount;
                            $consignment['totalFinalPrice'] = max(0, $total - $sharedDiscount);

                            // ذخیره اطلاعات تخفیف
                            $consignment['couponDiscount'] = [
                                'information' => $discountData['information'],
                                'amount' => $sharedDiscount,
                            ];
                        } else {
                            $consignment['couponDiscountPrice'] = 0;
                            $consignment['couponDiscount'] = null;
                        }
                    }
                    unset($consignment);
                }
            }
        }

        // financial values
        $financial = [
            'totalPrice' => 0,
            'totalDiscount' => 0,
            'totalFinalPrice' => 0,
            'totalCouponDiscountPrice' => 0,
            'sendPrice' => 0,
        ];
        foreach ($consignmentsInformation as $consignment) {
            $financial['totalPrice'] += $consignment['totalPrice'] ?? 0;
            $financial['totalDiscount'] += $consignment['totalDiscount'] ?? 0;
            $financial['totalFinalPrice'] += $consignment['totalFinalPrice'] ?? 0;
            $financial['totalCouponDiscountPrice'] += $consignment['couponDiscountPrice'] ?? 0;
            $financial['sendPrice'] += $consignment['sendPrice'] ?? 0;
        }

        return [
            'consignments' => $consignmentsInformation,
            'financial' => $financial,
        ];
    }

    /**
     * single consignment by send method
     */
    public function singleConsignmentInformation($order)
    {
        $consignments = $order->consignments()->with('store')->with('consignmentItems')->get();

        $consignmentsInformation = [
            'main' => [
                'uuid' => $order->uuid,
                'items' => [],
                'totalFinalPrice' => 0,
                'totalDiscount' => 0,
                'totalPrice' => 0,
                'totalWeight' => 0,
                'send_method' => $order->get_send_method,
            ]
        ];

        foreach ($consignments as $consignment) {
            foreach ($consignment->consignmentItems as $item) {
                $itemData = [
                    'id' => $item->id,
                    'get_props' => $item->get_props,
                    'inventory_id' => $item->inventory_id,
                    'count' => $item->count,
                    'discount' => $item->discount,
                    'price' => $item->price,
                    'get_full_count_price' => $item->get_full_count_price,
                    'get_full_count_discount' => $item->get_full_count_discount,
                    'get_full_count_weight_kg' => $item->get_full_count_weight_kg,
                    'inventory' => [
                        'weight' => $item->inventory?->weight ?? 0
                    ],
                    'product' => [
                        'id' => $item->product->id,
                        'title' => $item->product->title,
                        'slug' => $item->product->slug,
                        'category' => $item->product->productCategory->title,
                        'get_images' => $item->product->get_images,
                    ],
                    'store' => $item->store ? [
                        'id' => $item->store->id,
                        'uuid' => $item->store->uuid,
                        'name' => $item->store->name,
                    ] : null,
                ];

                $consignmentsInformation['main']['items'][] = $itemData;
                $consignmentsInformation['main']['totalFinalPrice'] += $itemData['get_full_count_price'];
                $consignmentsInformation['main']['totalDiscount'] += $itemData['get_full_count_discount'];
                $consignmentsInformation['main']['totalWeight'] += $itemData['get_full_count_weight_kg'] ?? 0;
            }

            $consignmentsInformation['main']['totalPrice'] =
                $consignmentsInformation['main']['totalFinalPrice'] +
                $consignmentsInformation['main']['totalDiscount'];
        }

        // تقسیم تخفیف بین مرسوله‌ها بر اساس totalFinalPrice هر مرسوله
        $discount = $order->discount != null ? unserialize($order->discount) : null;
        if ($discount != null && $discount['amount'] != null) {
            $discount = (int)str_replace(',', '', $discount['amount']);
            if ($discount > 0) {
                // فقط مرسوله‌هایی که تخفیف می‌گیرند
                $discountableConsignments = array_filter($consignmentsInformation, fn($c) => isset($c['totalFinalPrice']) && $c['totalFinalPrice'] > 0);
                $totalDiscountBase = array_sum(array_map(fn($c) => $c['totalFinalPrice'], $discountableConsignments));

                $sharedDiscountSum = 0;
                $lastKey = array_key_last($discountableConsignments);

                foreach ($consignmentsInformation as $key => &$consignment) {
                    $totalFinalPrice = $consignment['totalFinalPrice'] ?? 0;

                    if ($totalFinalPrice > 0 && $totalDiscountBase > 0) {
                        // محاسبه سهم نسبی این مرسوله
                        $shareRatio = $totalFinalPrice / $totalDiscountBase;
                        $sharedDiscount = (int) round($shareRatio * $discount);

                        // اطمینان از اینکه مجموع تخفیف دقیقاً برابر باشد
                        if ($key === $lastKey) {
                            $sharedDiscount = $discount - $sharedDiscountSum;
                        }

                        $sharedDiscountSum += $sharedDiscount;

                        $consignment['couponDiscountPrice'] = $sharedDiscount;
                        $consignment['totalFinalPrice'] = max(0, $consignment['totalFinalPrice'] - $sharedDiscount);
                    } else {
                        $consignment['shared_discount'] = 0;
                    }
                }
                unset($consignment);
            }
        }

        // financial values
        $financial = [
            'totalPrice' => 0,
            'totalDiscount' => 0,
            'totalFinalPrice' => 0,
            'totalCouponDiscountPrice' => 0,
            'sendPrice' => 0,
        ];
        foreach ($consignmentsInformation as $consignment) {
            $financial['totalPrice'] += $consignment['totalPrice'] ?? 0;
            $financial['totalDiscount'] += $consignment['totalDiscount'] ?? 0;
            $financial['totalFinalPrice'] += $consignment['totalFinalPrice'] ?? 0;
            $financial['totalCouponDiscountPrice'] += $consignment['couponDiscountPrice'] ?? 0;
        }

        return [
            'consignments' => $consignmentsInformation,
            'financial' => $financial,
        ];
    }

    /**
     * register order
     */
    public function registerOrder($paymentTransactionUuid = null, $snapppayTransactionID = null)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        try {
            $order = null;

            $orderGroupUUID = DB::transaction(function () use ($user, &$order, $paymentTransactionUuid, $snapppayTransactionID) {
                /** @var \App\Models\Order $order */
                $order = $user->orders()->where('status', 'awaiting')->lockForUpdate()->first();

                if (!$order || !$order->address || $order->is_locked) {
                    throw new \Exception('Order invalid or locked');
                }

                // قفل کردن سفارش تا پایان عملیات
                $order->update(['is_locked' => true]);

                // گرفتن مرسوله‌ها و آیتم‌ها
                $consignments = $order->consignments()->with('consignmentItems')->get();
                $consignmentItems = $consignments->flatMap(fn($c) => $c->consignmentItems);

                if ($consignmentItems->isEmpty()) {
                    throw new \Exception('consignentItems is empty');
                }

                // چک کردن ادرس فعال
                $selectedAddress = $order->address ? @unserialize($order->address) : null;
                if (!$selectedAddress) {
                    throw new \Exception('address is empty or invalid');
                }

                $consignmentsInformation = ($selectedAddress['type'] ?? null) === 'user'
                    ? $this->sepratedConsignmentInformation($order, $selectedAddress)
                    : $this->singleConsignmentInformation($order);

                $totalFinalPrice = $consignmentsInformation['financial']['totalFinalPrice'];

                // قفل کردن کیف پول کاربر
                $wallet = $user->wallet()->lockForUpdate()->first();
                if (!$wallet || $wallet->amount < $totalFinalPrice) {
                    throw new \Exception('Insufficient wallet amount');
                }

                $walletNewAmount = $wallet->amount - $totalFinalPrice;
                $transactionId = $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id');
                $orderGroupUUID = $snapppayTransactionID != null ? $snapppayTransactionID : $this->generateUniqueRandomNumber(9, \App\Models\OrderGroup::class, 'uuid');

                // آپدیت وضعیت آیتم‌های مرسوله و موجودی انبار
                foreach ($consignmentItems as $consignmentItem) {
                    $status = match (true) {
                        $consignmentItem->store && $consignmentItem->consignment->send_type === 'direct' => 'awaiting_seller_send',
                        $consignmentItem->store && $consignmentItem->consignment->send_type === null => 'awaiting_request_from_seller',
                        default => 'ready_to_send',
                    };
                    $consignmentItem->update(['status' => $status]);

                    $inventory = Inventory::where('id', $consignmentItem->inventory_id)->lockForUpdate()->first();
                    if ($inventory) {
                        $finalCount = max(0, $inventory->count - $consignmentItem->count);
                        $inventory->update(['count' => $finalCount]);
                    }
                }

                // ایجاد تراکنش کسر مبلغ
                $transaction = $user->transactions()->create([
                    'amount' => -1 * $totalFinalPrice,
                    'transaction_id' => $transactionId,
                    'description' => __('messages.transction_decrease_for_register_order', ['order' => $orderGroupUUID]),
                    'type' => 'order_decrease',
                    'status' => 'accepted',
                ]);

                // بروزرسانی کیف پول
                $wallet->update(['amount' => $walletNewAmount]);

                // ایجاد گروه سفارش والد
                $orderGroup = $user->orderGroups()->create([
                    'transaction_id' => $transaction->id,
                    'uuid' => $orderGroupUUID,
                    'payment_transaction_uuid' => $paymentTransactionUuid,
                ]);

                // ایجاد سفارش جدید برای هر مرسوله
                foreach ($consignmentsInformation['consignments'] as $consignment) {
                    $newOrder = $user->orders()->create([
                        'order_group_id' => $orderGroup->id,
                        'uuid' => $this->generateUniqueRandomNumber(9, \App\Models\Order::class, 'uuid'),
                        'send_by' => $consignment['store']['id'] ?? null,
                        'discount' => isset($consignment['couponDiscount']) ? serialize($consignment['couponDiscount']) : null,
                        'address' => $order->address,
                        'send_method' => serialize(array_merge(
                            $consignment['send_method'] ?? [],
                            [
                                'price' => $consignment['sendPrice'] ?? 0,
                                'free' => $consignment['freeSend'] ?? false,
                                'weight' => $consignment['totalWeight'] ?? 0,
                            ]
                        )),
                        'description' => $order->description,
                        'status' => 'registered',
                    ]);

                    // انتقال مالکیت آیتم‌های مرسوله به سفارش جدید
                    foreach ($consignment['items'] as $item) {
                        $cnsItem = ConsignmentItem::where('id', $item['id'])
                            ->whereHas('consignment', fn($q) => $q->where('user_id', $user->id))
                            ->first();

                        if ($cnsItem) {
                            $cnsItem->consignment->update(['order_id' => $newOrder->id]);
                        }
                    }
                }

                // امتیاز باشگاه مشتریان
                $clubSetting = Setting::where('key', 'customer_club')->first();
                $club = $clubSetting ? @unserialize($clubSetting->value) : ['price' => 0, 'score' => 0];
                $score = ($club['price'] > 0 && $club['score'] > 0)
                    ? round(($totalFinalPrice / str_replace(',', '', $club['price'])) * str_replace(',', '', $club['score']))
                    : 0;

                $user->scores()->create([
                    'score' => $score,
                    'description' => __('messages.sentence.increase_points_for_orders') . ' ' . $orderGroup->uuid,
                    'type' => 'increase',
                ]);

                // افزایش کش بک
                $userPaidAmount = $totalFinalPrice;
                $cashBackPercent = Cache::remember('setting_cashback_percent', now()->addDay(), function () {
                    $cashBackPercent = Setting::where('key', 'cashback_percent')->first();
                    return $cashBackPercent != null ? $cashBackPercent->value : null;
                });
                if (!is_null($cashBackPercent) && is_numeric($cashBackPercent) && $cashBackPercent > 0 && $cashBackPercent <= 100) {
                    // محاسبه مبلغ کش بک
                    $cashBackAmount = $userPaidAmount * ($cashBackPercent / 100);

                    // شارژ کیف پول
                    $wallet->update([
                        'amount' => $wallet->amount + $cashBackAmount,
                    ]);

                    // افزودن به جدول تراکنش ها
                    $cashbackTransction = $user->transactions()->create([
                        'amount' => $cashBackAmount,
                        'transaction_id' => $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id'),
                        'description' => __('messages.transction_increase_for_cashback', ['order_group_uuid' => $orderGroup->uuid]),
                        'type' => 'coupon_charge',
                        'status' => 'accepted',
                    ]);

                    // ارسال اعلان به مشتری
                    $user->notify(new WalletCharge($cashbackTransction, 'cashback'));
                }

                // علامت زدن کد تخفیف به عنوان استفاده شده
                $discount = $order->discount ? @unserialize($order->discount) : null;
                if ($discount && isset($discount['amount'])) {
                    Discount::where('code', $discount['information']['code'])->first()?->users()->attach($user->id);
                }

                // بررسی و حذف مرسوله های خالی
                foreach ($order->consignments()->get() as $tempCnsItem) {
                    if ($tempCnsItem->consignmentItems()->count() == 0) {
                        $tempCnsItem->delete();
                    }
                }

                // بررسی باقی نماندن مرسوله در سفارش قبلی
                if ($order->consignments()->exists()) {
                    throw new \Exception('Cannot delete order with consignments still attached');
                }

                return $orderGroup->uuid;
            });

            DB::afterCommit(function () use ($user, $order, $orderGroupUUID) {
                // یافتن مجدد سفارش مادر
                $orderGroup = OrderGroup::where('uuid', $orderGroupUUID)->first();

                // ارسال نوتیفیکیشن به کاربر
                $user->notify(new OrderRegistration($orderGroup->uuid));

                // نوتیفیکیشن های اطلاعرسانی سفارش جدید برای مدیران و فروشندگان
                foreach ($orderGroup->orders as $childOrder) {
                    if ($childOrder->consignments()->where('store_id', null)->first() != null || $childOrder->consignments()->where('send_type', 'direct')->first() == null) {
                        // ارسال نوتیفیکیشن به ادمین‌ها
                        User::whereIn('type', ['owner', 'admin'])->cursor()->each(function ($admin) use ($orderGroup) {
                            if ($admin->type === 'owner' || $admin->can('manage-orders')) {
                                $admin->notify(new RequestConsignment($orderGroup->uuid));
                            }
                        });
                    } else {
                        // ارسال نوتیفیکیشن به فروشنده
                        $firstConsignment = $childOrder->consignments()->first();
                        $firstConsignment->store->user->notify(new RequestConsignment($childOrder->uuid));
                    }
                }

                // حذف سفارش خالی
                if ($order && !$order->consignments()->exists()) {
                    $order->delete();
                }
            });

            return $orderGroupUUID;
        } catch (\Exception $e) {
            // اگر عملیات ناقص بود، is_locked را ریست کن
            DB::transaction(function () use ($user) {
                $order = $user->orders()->where('status', 'awaiting')->where('is_locked', true)->lockForUpdate()->first();
                if ($order) {
                    $order->update(['is_locked' => false]);
                }
            });

            report($e);
            return false;
        }
    }

    /**
     * register order
     */
    public function registerPrefactorOrder($uuid, $paymentTransactionUuid = null, $snapppayTransactionID = null)
    {
        /** user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        /** @var \App\Models\User $user */
        $prefactor = Prefactor::where('uuid', $uuid)->where('user_id', $user->id)->first();
        if ($prefactor == null || $prefactor->status != 'waiting_payment') {
            return false;
        }

        try {
            $order = null;

            $orderGroupUUID = DB::transaction(function () use ($user, &$order, $prefactor, $paymentTransactionUuid, $snapppayTransactionID) {
                /** @var \App\Models\Order $order */
                $order = $user->orders()->where('id', $prefactor->order->id)->where('status', 'prefactor')->lockForUpdate()->first();

                if (!$order || !$order->address || $order->is_locked) {
                    throw new \Exception('Order invalid or locked');
                }

                // قفل کردن سفارش تا پایان عملیات
                $order->update(['is_locked' => true]);

                // گرفتن مرسوله‌ها و آیتم‌ها
                $consignments = $order->consignments()->with('consignmentItems')->get();
                $consignmentItems = $consignments->flatMap(fn($c) => $c->consignmentItems);

                if ($consignmentItems->isEmpty()) {
                    throw new \Exception('consignentItems is empty');
                }

                // چک کردن ادرس فعال
                $selectedAddress = $order->address ? @unserialize($order->address) : null;
                if (!$selectedAddress) {
                    throw new \Exception('address is empty or invalid');
                }

                $consignmentsInformation = ($selectedAddress['type'] ?? null) === 'user'
                    ? $this->sepratedConsignmentInformation($order, $selectedAddress)
                    : $this->singleConsignmentInformation($order);

                $totalFinalPrice = $consignmentsInformation['financial']['totalFinalPrice'];

                // قفل کردن کیف پول کاربر
                $wallet = $user->wallet()->lockForUpdate()->first();
                if (!$wallet || $wallet->amount < $totalFinalPrice) {
                    throw new \Exception('Insufficient wallet amount');
                }

                $walletNewAmount = $wallet->amount - $totalFinalPrice;
                $transactionId = $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id');
                $orderGroupUUID = $snapppayTransactionID != null ? $snapppayTransactionID : $this->generateUniqueRandomNumber(9, \App\Models\OrderGroup::class, 'uuid');

                // آپدیت وضعیت آیتم‌های مرسوله و موجودی انبار
                foreach ($consignmentItems as $consignmentItem) {
                    $status = match (true) {
                        $consignmentItem->store && $consignmentItem->consignment->send_type === 'direct' => 'awaiting_seller_send',
                        $consignmentItem->store && $consignmentItem->consignment->send_type === null => 'awaiting_request_from_seller',
                        default => 'ready_to_send',
                    };
                    $consignmentItem->update(['status' => $status]);

                    $inventory = Inventory::where('id', $consignmentItem->inventory_id)->lockForUpdate()->first();
                    if ($inventory) {
                        $finalCount = max(0, $inventory->count - $consignmentItem->count);
                        $inventory->update(['count' => $finalCount]);
                    }
                }

                // ایجاد تراکنش کسر مبلغ
                $transaction = $user->transactions()->create([
                    'amount' => -1 * $totalFinalPrice,
                    'transaction_id' => $transactionId,
                    'description' => __('messages.transction_decrease_for_register_order', ['order' => $orderGroupUUID]),
                    'type' => 'order_decrease',
                    'status' => 'accepted',
                ]);

                // بروزرسانی کیف پول
                $wallet->update(['amount' => $walletNewAmount]);

                // ایجاد گروه سفارش والد
                $orderGroup = $user->orderGroups()->create([
                    'transaction_id' => $transaction->id,
                    'uuid' => $orderGroupUUID,
                    'payment_transaction_uuid' => $paymentTransactionUuid,
                ]);

                // ایجاد سفارش جدید برای هر مرسوله
                foreach ($consignmentsInformation['consignments'] as $consignment) {
                    $newOrder = $user->orders()->create([
                        'order_group_id' => $orderGroup->id,
                        'uuid' => $this->generateUniqueRandomNumber(9, \App\Models\Order::class, 'uuid'),
                        'send_by' => $consignment['store']['id'] ?? null,
                        'discount' => isset($consignment['couponDiscount']) ? serialize($consignment['couponDiscount']) : null,
                        'address' => $order->address,
                        'send_method' => serialize(array_merge(
                            $consignment['send_method'] ?? [],
                            [
                                'price' => $consignment['sendPrice'] ?? 0,
                                'free' => $consignment['freeSend'] ?? false,
                                'weight' => $consignment['totalWeight'] ?? 0,
                            ]
                        )),
                        'description' => $order->description,
                        'status' => 'registered',
                    ]);

                    // انتقال مالکیت آیتم‌های مرسوله به سفارش جدید
                    foreach ($consignment['items'] as $item) {
                        $cnsItem = ConsignmentItem::where('id', $item['id'])
                            ->whereHas('consignment', fn($q) => $q->where('user_id', $user->id))
                            ->first();

                        if ($cnsItem) {
                            $cnsItem->consignment->update(['order_id' => $newOrder->id]);
                        }
                    }
                }

                // امتیاز باشگاه مشتریان
                $clubSetting = Setting::where('key', 'customer_club')->first();
                $club = $clubSetting ? @unserialize($clubSetting->value) : ['price' => 0, 'score' => 0];
                $score = ($club['price'] > 0 && $club['score'] > 0)
                    ? round(($totalFinalPrice / str_replace(',', '', $club['price'])) * str_replace(',', '', $club['score']))
                    : 0;

                $user->scores()->create([
                    'score' => $score,
                    'description' => __('messages.sentence.increase_points_for_orders') . ' ' . $orderGroup->uuid,
                    'type' => 'increase',
                ]);

                // افزایش کش بک
                $userPaidAmount = $totalFinalPrice;
                $cashBackPercent = Cache::remember('setting_cashback_percent', now()->addDay(), function () {
                    $cashBackPercent = Setting::where('key', 'cashback_percent')->first();
                    return $cashBackPercent != null ? $cashBackPercent->value : null;
                });
                if (!is_null($cashBackPercent) && is_numeric($cashBackPercent) && $cashBackPercent > 0 && $cashBackPercent <= 100) {
                    // محاسبه مبلغ کش بک
                    $cashBackAmount = $userPaidAmount * ($cashBackPercent / 100);

                    // شارژ کیف پول
                    $wallet->update([
                        'amount' => $wallet->amount + $cashBackAmount,
                    ]);

                    // افزودن به جدول تراکنش ها
                    $cashbackTransction = $user->transactions()->create([
                        'amount' => $cashBackAmount,
                        'transaction_id' => $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id'),
                        'description' => __('messages.transction_increase_for_cashback', ['order_group_uuid' => $orderGroup->uuid]),
                        'type' => 'coupon_charge',
                        'status' => 'accepted',
                    ]);

                    // ارسال اعلان به مشتری
                    $user->notify(new WalletCharge($cashbackTransction, 'cashback'));
                }

                // علامت زدن کد تخفیف به عنوان استفاده شده
                $discount = $order->discount ? @unserialize($order->discount) : null;
                if ($discount && isset($discount['amount'])) {
                    Discount::where('code', $discount['information']['code'])->first()?->users()->attach($user->id);
                }

                // تبدیل وضعیت پیش فاکتور به پرداخت شده
                $prefactor->update([
                    'order_id' => null,
                    'order_group_id' => $orderGroup->id,
                    'status' => 'paid',
                ]);

                // بررسی و حذف مرسوله های خالی
                foreach ($order->consignments()->get() as $tempCnsItem) {
                    if ($tempCnsItem->consignmentItems()->count() == 0) {
                        $tempCnsItem->delete();
                    }
                }

                // بررسی باقی نماندن مرسوله در سفارش قبلی
                if ($order->consignments()->exists()) {
                    throw new \Exception('Cannot delete order with consignments still attached');
                }

                return $orderGroup->uuid;
            });

            DB::afterCommit(function () use ($user, $order, $orderGroupUUID) {
                // یافتن مجدد سفارش مادر
                $orderGroup = OrderGroup::where('uuid', $orderGroupUUID)->first();

                // ارسال نوتیفیکیشن به کاربر
                $user->notify(new OrderRegistration($orderGroup->uuid));

                // نوتیفیکیشن های اطلاعرسانی سفارش جدید برای مدیران و فروشندگان
                foreach ($orderGroup->orders as $childOrder) {
                    if ($childOrder->consignments()->where('store_id', null)->first() != null || $childOrder->consignments()->where('send_type', 'direct')->first() == null) {
                        // ارسال نوتیفیکیشن به ادمین‌ها
                        User::whereIn('type', ['owner', 'admin'])->cursor()->each(function ($admin) use ($orderGroup) {
                            if ($admin->type === 'owner' || $admin->can('manage-orders')) {
                                $admin->notify(new RequestConsignment($orderGroup->uuid));
                            }
                        });
                    } else {
                        // ارسال نوتیفیکیشن به فروشنده
                        $firstConsignment = $childOrder->consignments()->first();
                        $firstConsignment->store->user->notify(new RequestConsignment($childOrder->uuid));
                    }
                }

                // حذف سفارش خالی
                if ($order && !$order->consignments()->exists()) {
                    $order->delete();
                }
            });

            return $orderGroupUUID;
        } catch (\Exception $e) {
            // اگر عملیات ناقص بود، is_locked را ریست کن
            $prefactor->refresh();
            DB::transaction(function () use ($user, $prefactor) {
                $order = $user->orders()->where('id', $prefactor->order->id)->where('is_locked', true)->lockForUpdate()->first();
                if ($order) {
                    $order->update(['is_locked' => false]);
                }
            });

            report($e);
            return false;
        }
    }
}
