<?php

namespace App\Services\Admin\Stores;

use App\Models\Store;
use App\Models\User;
use Illuminate\Http\Request;

class StoresService
{
    /** get stores list */
    public function getStores(Request $request)
    {
        /** @var LengthAwarePaginator */
        $stores = Store::query()->where('status', '!=', 'proccess')->when($request->input('search'), function ($query, $search) {
            $query->where('name', 'like', "%{$search}%");
        })->when($request->input('active'), function ($query, $active) {
            if ($active) {
                $query->where('status', 'active');
            }
        })->when($request->input('banned'), function ($query, $banned) {
            if ($banned) {
                $query->where('status', 'banned');
            }
        })->when($request->input('awaiting'), function ($query, $awaiting) {
            if ($awaiting) {
                $query->where('status', 'awaiting');
            }
        })->when($request->input('reject'), function ($query, $reject) {
            if ($reject) {
                $query->where('status', 'reject');
            }
        })->orderBy('created_at', 'desc')->with('user')->with('user.wallet')->paginate(20)->through(fn($store) => [
            'id' => $store->id,
            'uuid' => $store->uuid,
            'name' => $store->name,
            'get_logo' => $store->get_logo,
            'get_formatted_view_count' => $store->get_formatted_view_count,
            'jalali_created_at' => $store->jalali_created_at,
            'status' => $store->status,
            'user' => [
                'id' => $store->user->id,
                'get_name' => $store->user->get_name,
                'wallet' => [
                    'id' => $store->user->wallet->id,
                    'get_formatted_amount' => $store->user->wallet->get_formatted_amount,
                ],
            ],
        ]);

        $stores->withQueryString();

        return $stores;
    }

    /** get consignments list */
    public function getConsignments(User $user)
    {
        /** @var LengthAwarePaginator */
        $consignments = $user->store->consignments()->where('status', '!=', 'created')->orderBy('created_at', 'desc')->paginate(20)->through(fn($consignment) => [
            'id' => $consignment->id,
            'uuid' => $consignment->uuid,
            'status' => $consignment->status,
            'jalali_created_at' => $consignment->jalali_created_at,
            'jalali_updated_at' => $consignment->jalali_updated_at,
            'order' => [
                'id' => $consignment->order->id
            ],
        ]);

        $consignments->withQueryString();

        return $consignments;
    }

    /** get transactions list */
    public function getTransactions(User $user)
    {
        /** @var LengthAwarePaginator */
        $transactions = $user->transactions()->orderBy('created_at', 'desc')->paginate(20)->through(fn($transaction) => [
            'id' => $transaction->id,
            'amount' => $transaction->amount,
            'type' => $transaction->type,
            'get_formatted_amount' => $transaction->get_formatted_amount,
            'description' => $transaction->description,
            'transaction_id' => $transaction->transaction_id,
            'jalali_created_at' => $transaction->jalali_created_at,
            'status' => $transaction->status,
        ]);
        $transactions->withQueryString();

        return $transactions;
    }

    /** get tickets list */
    public function getTickets(User $user)
    {
        /** @var LengthAwarePaginator */
        $tickets = $user->tickets()->orderByRaw("FIELD(status , 'waiting_admin', 'waiting_user', 'closed') ASC")
            ->orderByRaw("FIELD(priority , 'high', 'normal', 'low') ASC")
            ->orderBy('updated_at', 'asc')->paginate(20)->through(fn($ticket) => [
                'id' => $ticket->id,
                'title' => $ticket->title,
                'priority' => $ticket->priority,
                'jalali_updated_at' => $ticket->jalali_updated_at,
                'status' => $ticket->status,
            ]);
        $tickets->withQueryString();

        return $tickets;
    }

    /** get awaiting store list */
    public function getAwaitingStores(Request $request)
    {
        /** @var LengthAwarePaginator */
        $stores = Store::query()->where('status', 'awaiting')->when($request->input('search'), function ($query, $search) {
            $query->where('name', 'like', "%{$search}%");
        })->orderBy('created_at', 'asc')->with('user')->with('user.wallet')->paginate(20)->through(fn($store) => [
            'id' => $store->id,
            'uuid' => $store->uuid,
            'name' => $store->name,
            'get_logo' => $store->get_logo,
            'get_formatted_view_count' => $store->get_formatted_view_count,
            'jalali_created_at' => $store->jalali_created_at,
            'status' => $store->status,
            'user' => [
                'id' => $store->user->id,
                'get_name' => $store->user->get_name,
                'wallet' => [
                    'id' => $store->user->wallet->id,
                    'get_formatted_amount' => $store->user->wallet->get_formatted_amount,
                ],
            ],
        ]);

        $stores->withQueryString();

        return $stores;
    }
}
