<?php

namespace App\Services\Admin\Storage;

use App\Models\PaymentMeta;
use App\Models\Product;
use App\Models\Storage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StorageService
{
    /** get storage list */
    public function getStorages(Request $request)
    {
        $storages = Storage::query()->where('store_id', null)->when($request->input('search'), function ($query, $search) {
            $query->where('name', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->get();

        return $storages;
    }

    /** get storage product list */
    public function getStorageProducts(Request $request, Storage $storage)
    {
        /** @var LengthAwarePaginator */
        $products = Product::with('inventories')->where('status', '!=', 'waiting')->where('status', '!=', 'deleted')->where('status', '!=', 'deep_deleted')->where('status', '!=', 'rejected')
            ->whereHas('inventories', function ($query) use ($storage) {
                $query->where('storage_id', $storage->id);
            })->withCount(['consignmentItems' => function ($query) {
                $query->where(function ($query) {
                    $query->where('status', 'sent')->orWhere('status', 'delivered');
                })->select(DB::raw('SUM(count)'));
            }])->when($request->input('search'), function ($query, $search) {
                $query->where(DB::raw('concat(`title`, `uuid`)'), 'like', "%{$search}%");
            })->when($request->input('publish'), function ($query, $publish) {
                if ($publish) {
                    $query->where('status', 'publish');
                }
            })->when($request->input('awaiting'), function ($query, $awaiting) {
                if ($awaiting) {
                    $query->where('status', 'awaiting');
                }
            })->when($request->input('draft'), function ($query, $draft) {
                if ($draft) {
                    $query->where('status', 'draft');
                }
            })->when($request->input('mostViewer'), function ($query, $mostViewer) {
                if ($mostViewer) {
                    $query->orderBy('view_count', 'desc');
                }
            })->when($request->input('mostSale'), function ($query, $mostSale) {
                if ($mostSale) {
                    $query->orderBy('consignment_items_count', 'desc');
                }
            })->when($request->input('hasPrice'), function ($query, $hasPrice) use ($storage) {
                if ($hasPrice) {
                    $query->whereHas('inventories', function ($query) use ($storage) {
                        $query->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0);
                    });
                }
            })->when($request->input('outofstock'), function ($query, $outofstock) use ($storage) {
                if ($outofstock) {
                    $query->whereHas('inventories', function ($query) use ($storage) {
                        $query->where('storage_id', $storage->id)->where('status', 'publish');
                    })->whereDoesntHave('inventories', function ($query) use ($storage) {
                        $query->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0);
                    });
                }
            })->orderBy('created_at', 'desc')->paginate(20)->through(fn($product) => [
                'id' => $product->id,
                'user_id' => $product->user_id,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'is_vip' => $product->is_vip,
                'inventory_in_storage' => $product->inventories()->where('storage_id', $storage->id)->get()->count(),
                'inventory_count_in_storage' => $product->inventories()->where('storage_id', $storage->id)->get()->sum('count'),
                'inventory_status' => $this->inventoryStatus($product, $storage),
                'status' => $product->status,
                'user' => [
                    'id' => $product->user->id,
                    'get_name' => $product->user->get_name,
                    'type' => $product->user->type,
                    'seller' => $product->user->seller,
                    'store' => $product->user->store != null ? [
                        'id' => $product->user->store->id,
                        'name' => $product->user->store->name,
                        'uuid' => $product->user->store->uuid,
                    ] : null
                ]
            ]);
        $products->withQueryString();

        return $products;
    }

    private function inventoryStatus(Product $product, Storage $storage)
    {
        return [
            'inventory' => $product->inventories()->where('storage_id', $storage->id)->where('status', 'publish')->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->exists(),
            'instock' => $product->inventories()->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0)->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->exists(),
            'inquery' => !$product->in_stock_status
        ];
    }

    /** check payment meta */
    public function checkPaymentMeta(): bool
    {
        $paymentMeta = PaymentMeta::where('store_id', null)->first();
        if ($paymentMeta == null)
            return false;

        if (count($paymentMeta->get_post_ways) == 0)
            return false;

        return true;
    }
}
