<?php

namespace App\Services\Admin\Statistics;

use App\Models\Article;
use App\Models\Product;
use App\Models\Statistic;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Morilog\Jalali\Jalalian;

class StatisticsService
{
    /** get page view */
    public function getViewCharts($subDays)
    {
        // past {$subDays} days period
        $period = CarbonPeriod::create(Carbon::now()->subDays($subDays), Carbon::now());

        // chart date labels
        $dates = [];

        // shop charts
        $shopPageView = [];
        $shopPageUniqueView = [];

        // blog charts
        $blogPageView = [];
        $blogPageUniqueView = [];

        // Iterate over the period
        foreach ($period as $date) {
            // date labels
            $dates[] = Jalalian::forge($date)->format('m-d');

            // shop
            $shopPageView[] = Statistic::where('platform', 'shop')->whereDate('created_at', $date)->count();
            $shopPageUniqueView[] = Statistic::where('platform', 'shop')->whereDate('created_at', $date)->distinct()->count('ip');

            // blog
            $blogPageView[] = Statistic::where('platform', 'blog')->whereDate('created_at', $date)->count();
            $blogPageUniqueView[] = Statistic::where('platform', 'blog')->whereDate('created_at', $date)->distinct()->count('ip');
        }

        return [
            'dates' => $dates,

            'totalPageView' => Statistic::whereDate('created_at', '>=', Carbon::now()->subDay($subDays))->count(),
            'shopPageView' => $shopPageView,
            'blogPageView' => $blogPageView,

            'totalPageUniqueView' => Statistic::whereDate('created_at', '>=', Carbon::now()->subDay($subDays))->distinct()->count('ip'),
            'shopPageUniqueView' => $shopPageUniqueView,
            'blogPageUniqueView' => $blogPageUniqueView,
        ];
    }

    /**
     * get products sales
     */
    public function getProductsSales(Request $request)
    {
        $products = Product::where('status', 'publish')
            ->withCount(['consignmentItems' => function ($query) use ($request) {
                $query->select(DB::raw('SUM(count)'))->where(function ($query) {
                    $query->where('status', 'sent')->orWhere('status', 'delivered');
                })->when($request->input('sale_today'), function ($query, $sale_today) {
                    if ($sale_today) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDay());
                    }
                })->when($request->input('sale_week'), function ($query, $sale_week) {
                    if ($sale_week) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDays(7));
                    }
                })->when($request->input('sale_month'), function ($query, $sale_month) {
                    if ($sale_month) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDays(30));
                    }
                });
            }])
            ->orderBy('consignment_items_count', 'desc')->limit(10)->get()->map(fn($product) => [
                'id' => $product->id,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'is_vip' => $product->is_vip,
                'consignment_items_count' => $product->consignment_items_count,
                'jalali_updated_at' => $product->jalali_updated_at,
                'status' => $product->status,
            ]);

        return $products;
    }

    /**
     * get products visit
     */
    public function getProductsVisits(Request $request)
    {
        $products = [];

        if ($request->exists('visit_today') || $request->exists('visit_week') || $request->exists('visit_month')) {
            $productStatistics = Statistic::select('url', DB::raw('count(*) as total'))
                ->where('service', 'product')
                ->when($request->input('visit_today'), function ($query, $visit_today) {
                    if ($visit_today) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDay());
                    }
                })->when($request->input('visit_week'), function ($query, $visit_week) {
                    if ($visit_week) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDays(7));
                    }
                })->when($request->input('visit_month'), function ($query, $visit_month) {
                    if ($visit_month) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDays(30));
                    }
                })->groupBy('url')->orderBy('total', 'desc')->limit(10)->get();

            foreach ($productStatistics as $statistic) {
                $product = Product::where('status', 'publish')->where('slug', urldecode(str_replace('/product/', '', $statistic->url)))->first();
                if ($product != null) {
                    $products[] = [
                        'id' => $product->id,
                        'title' => $product->title,
                        'slug' => $product->slug,
                        'get_images' => $product->get_images,
                        'is_vip' => $product->is_vip,
                        'view_count' => $statistic['total'],
                        'jalali_updated_at' => $product->jalali_updated_at,
                        'status' => $product->status,
                    ];
                }
            }
        } else {
            $products = Product::where('status', 'publish')->orderBy('view_count', 'desc')->limit(10)->get()->map(fn($product) => [
                'id' => $product->id,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'is_vip' => $product->is_vip,
                'view_count' => $product->view_count,
                'jalali_updated_at' => $product->jalali_updated_at,
                'status' => $product->status,
            ]);
        }

        return $products;
    }

    /**
     * get articles visit
     */
    public function getArticlesVisits(Request $request)
    {
        $articles = [];

        if ($request->exists('visit_today') || $request->exists('visit_week') || $request->exists('visit_month')) {
            $articleStatistics = Statistic::select('url', DB::raw('count(*) as total'))
                ->where('service', 'article')
                ->when($request->input('visit_today'), function ($query, $visit_today) {
                    if ($visit_today) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDay());
                    }
                })->when($request->input('visit_week'), function ($query, $visit_week) {
                    if ($visit_week) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDays(7));
                    }
                })->when($request->input('visit_month'), function ($query, $visit_month) {
                    if ($visit_month) {
                        $query->whereDate('created_at', '>=', Carbon::now()->subDays(30));
                    }
                })->groupBy('url')->orderBy('total', 'desc')->limit(20)->get();

            foreach ($articleStatistics as $statistic) {
                $article = Article::where('status', 'publish')->where('slug', urldecode(str_replace('/blog/article/', '', $statistic->url)))->first();
                if ($article != null) {
                    $articles[] = [
                        'id' => $article->id,
                        'title' => $article->title,
                        'slug' => $article->slug,
                        'get_image' => $article->get_image,
                        'editor_choice' => $article->editor_choice,
                        'get_formatted_view_count' => $statistic['total'],
                        'jalali_updated_at' => $article->jalali_updated_at,
                        'status' => $article->status,
                    ];
                }
            }
        } else {
            $articles = Article::where('status', 'publish')->orderBy('view_count', 'desc')->limit(20)->get()->map(fn($article) => [
                'id' => $article->id,
                'title' => $article->title,
                'slug' => $article->slug,
                'get_image' => $article->get_image,
                'editor_choice' => $article->editor_choice,
                'get_formatted_view_count' => $article->get_formatted_view_count,
                'jalali_updated_at' => $article->jalali_updated_at,
                'status' => $article->status,
            ]);
        }

        return $articles;
    }
}
