<?php

namespace App\Services\Admin\Products;

use App\Models\Brand;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\ProductTag;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductService
{
    /** get product list */
    public function getProducts(Request $request, $flag = "all")
    {
        /** @var LengthAwarePaginator */
        $products = Product::query();

        if ($flag == "all") {
            $products = $products->where('status', '!=', 'deleted')
                ->where('status', '!=', 'deep_deleted')
                ->where('status', '!=', 'rejected');
        } else if ($flag == "awaiting") {
            $products = $products->where('status', 'awaiting');
        }

        $products = $products
            ->withCount(['consignmentItems' => function ($query) {
                $query->where(function ($query) {
                    $query->where('status', 'sent')->orWhere('status', 'delivered');
                })->select(DB::raw('SUM(count)'));
            }])
            ->when($request->input('search'), function ($query, $search) {
                $query->where(DB::raw('concat(`title`, `uuid`)'), 'like', "%{$search}%");
            })->when($request->input('publish'), function ($query, $publish) {
                if ($publish) {
                    $query->where('status', 'publish');
                }
            })->when($request->input('awaiting'), function ($query, $awaiting) {
                if ($awaiting) {
                    $query->where('status', 'awaiting');
                }
            })->when($request->input('draft'), function ($query, $draft) {
                if ($draft) {
                    $query->where('status', 'draft');
                }
            })->when($request->input('mostViewer'), function ($query, $mostViewer) {
                if ($mostViewer) {
                    $query->orderBy('view_count', 'desc');
                }
            })->when($request->input('mostSale'), function ($query, $mostSale) {
                if ($mostSale) {
                    $query->orderBy('consignment_items_count', 'desc');
                }
            })->when($request->input('hasPrice'), function ($query, $hasPrice) {
                if ($hasPrice) {
                    $query->where(function ($query) {
                        $query->where('in_stock_status', false)
                            ->orWhere(function ($query) {
                                $query->whereHas('inventories', function ($query) {
                                    $query->where('status', 'publish')->where('count', '>', 0);
                                });
                            });
                    });
                }
            })->when($request->input('noPrice'), function ($query, $noPrice) {
                if ($noPrice) {
                    $query->whereDoesntHave('inventories', function ($query) {
                        $query->where('status', 'publish');
                    });
                }
            })->when($request->input('outofstock'), function ($query, $outofstock) {
                if ($outofstock) {
                    $query->where('in_stock_status', '!=', false)
                        ->whereHas('inventories', function ($query) {
                            $query->where('status', 'publish')->where('count', 0);
                        })
                        ->whereDoesntHave('inventories', function ($query) {
                            $query->where('status', 'publish')->where('count', '>', 0);
                        });
                }
            })->orderByRaw("FIELD(status , 'awaiting') DESC")
            ->orderBy('created_at', 'desc')->with('user')->with('user.store')->paginate(20)->through(fn($product) => [
                'id' => $product->id,
                'user_id' => $product->user_id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'is_vip' => $product->is_vip,
                'get_affiliate' => $product->get_affiliate,
                'get_formatted_view_count' => $product->get_formatted_view_count,
                'consignment_items_count' => $product->consignment_items_count,
                'comments_info' => $product->comments_info,
                'questions_info' => $product->questions_info,
                'answers_info' => $product->answers_info,
                'inventory_status' => $product->inventory_status,
                'jalali_updated_at' => $product->jalali_updated_at,
                'status' => $product->status,
                'user' => [
                    'id' => $product->user->id,
                    'get_name' => $product->user->get_name,
                    'type' => $product->user->type,
                    'seller' => $product->user->seller,
                    'store' => $product->user->store != null ? [
                        'id' => $product->user->store->id,
                        'uuid' => $product->user->store->uuid,
                        'name' => $product->user->store->name,
                    ] : null,
                ],
            ]);

        $products->withQueryString();

        return $products;
    }

    /** get category branch */
    public function getCategoriesBranch()
    {
        $categoriesWithChilds = [];
        $categories = ProductCategory::where('status', 'publish')->where('parent_id', 0)->select('id', 'title', 'slug', 'parent_id')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'specifications' => $category->specifications()->get()->map(fn($specification) => [
                'id' => $specification->id,
                'title' => $specification->title,
                'keys' => $specification->specificationKeys()->get()->map(fn($specificationKey) => [
                    'id' => $specificationKey->id,
                    'key' => $specificationKey->key,
                    'value' => $specificationKey->default_value,
                ])->toArray()
            ])->toArray(),
            'childs' => $category->childs,
            'level' => 'one',
        ])->toArray();

        foreach ($categories as $category) {
            $categoriesWithChilds[] = $category;
            foreach ($category['childs'] as $childLevelTwo) {
                $categoriesWithChilds[] = $childLevelTwo;
                foreach ($childLevelTwo['childs'] as $childLevelThree) {
                    $categoriesWithChilds[] = $childLevelThree;
                }
            }
        }

        return $categoriesWithChilds;
    }

    public function getCategoriesBranchWithSpecification(Product $product)
    {
        $categories = ProductCategory::with([
            'specifications.specificationKeys.specificationValues' => function ($q) use ($product) {
                $q->where('product_id', $product->id);
            },
        ])
            ->where('status', 'publish')
            ->get();

        $map = [];
        foreach ($categories as $cat) {
            $map[$cat->id] = [
                'id' => $cat->id,
                'title' => $cat->title,
                'slug' => $cat->slug,
                'specifications' => $cat->specifications->map(function ($spec) use ($product) {
                    return [
                        'id' => $spec->id,
                        'title' => $spec->title,
                        'keys' => $spec->specificationKeys->map(function ($key) use ($product) {
                            $value = $key->specificationValues->where('product_id', $product->id)->first();
                            return [
                                'id' => $key->id,
                                'key' => $key->key,
                                'value_id' => $value?->id,
                                'value' => $value?->value ?? $key->default_value,
                            ];
                        })->toArray(),
                    ];
                })->toArray(),
                'parent_id' => $cat->parent_id,
                'childs' => [],
            ];
        }

        foreach ($map as $id => &$node) {
            if ($node['parent_id'] != 0 && isset($map[$node['parent_id']])) {
                $map[$node['parent_id']]['childs'][] = &$node;
            }
        }
        unset($node);

        $tree = [];
        foreach ($map as $id => $node) {
            if ($node['parent_id'] == 0 || !isset($map[$node['parent_id']])) {
                $tree[] = $node;
            }
        }

        $walk = function ($nodes, $level = 'one') use (&$walk) {
            $result = [];
            foreach ($nodes as $node) {
                $copy = $node;
                $copy['level'] = $level;
                $result[] = $copy;
                if (!empty($node['childs'])) {
                    $nextLevel = $level === 'one' ? 'two' : ($level === 'two' ? 'three' : 'four');
                    $result = array_merge($result, $walk($node['childs'], $nextLevel));
                }
            }
            return $result;
        };

        $flat = $walk($tree);

        return $flat;
    }

    /** init product brand */
    public function initBrand(User $user, Request $request)
    {
        /** brand */
        $brand = null;
        if ($request->exists('brand') && $request->brand != null && $request->brand != '') {
            $findBrand = Brand::where('title', $request->brand)->get()->first();
            if ($findBrand != null) {
                $brand = $findBrand->id;
            } else {
                $brand = $user->brands()->create([
                    'uuid' => 'brand-' . rand(1000000, 9999999),
                    'title' => $request->brand,
                    'status' => 'publish',
                ]);
                $brand = $brand->id;
            }
        }

        return $brand;
    }

    /** add product price model */
    public function addPriceModel(Product $product, Request $request)
    {
        if ($request->exists('price_model') && is_array($request->price_model)) {
            $pmID = 0;
            $product->productProps()->delete();
            foreach ($request->price_model as $pm) {
                if ($pm['active'] && $pm['name'] != null) {
                    $pmModel = $product->productProps()->create([
                        'child' => $pmID,
                        'name' => $pm['name'],
                        'type' => $pm['type'],
                    ]);

                    $pmID = $pmModel->id;
                }
            }
        }
    }

    /** add product specifications */
    public function addSpecifications(Product $product, Request $request)
    {
        if ($request->exists('specifications') && is_array($request->specifications)) {
            foreach ($request->specifications as $specification) {
                foreach ($specification['keys'] as $key) {
                    if ($key['value'] != null) {
                        $product->specificationValues()->create([
                            'specification_key_id' => $key['id'],
                            'value' => $key['value'],
                        ]);
                    }
                }
            }
        }
    }

    /** update product specifications */
    public function updateSpecifications(Product $product, Request $request)
    {
        if ($request->exists('specifications') && is_array($request->specifications)) {
            foreach ($request->specifications as $specification) {
                foreach ($specification['keys'] as $key) {
                    $spValueExists = $product->specificationValues()->where('id', $key['value_id'])->first();
                    if ($spValueExists != null) {
                        if ($key['value'] != null) {
                            $spValueExists->update([
                                'value' => $key['value'],
                            ]);
                        } else {
                            $spValueExists->delete();
                        }
                    } else {
                        if ($key['value'] != null) {
                            $product->specificationValues()->create([
                                'specification_key_id' => $key['id'],
                                'value' => $key['value'],
                            ]);
                        }
                    }
                }
            }
        }
    }

    /** sync product tags */
    public function syncTags(Product $product, Request $request)
    {
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            foreach (json_decode($request->tags) as $tag) {
                $findTag = ProductTag::where('title', $tag->value)->get()->first();
                if ($findTag != null) {
                    $tags[] = $findTag->id;
                } else {
                    $createdTag = ProductTag::create([
                        'title' => $tag->value,
                        'status' => 'publish',
                    ]);
                    $tags[] = $createdTag->id;
                }
            }
            $product->productTags()->sync($tags);
        }
    }

    /** update and sync product tags */
    public function updateAndSyncTags(Product $product, Request $request)
    {
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            if (is_array($request->tags)) {
                foreach ($request->tags as $tag) {
                    $findTag = ProductTag::where('title', $tag)->get()->first();
                    if ($findTag != null) {
                        $tags[] = $findTag->id;
                    } else {
                        $createdTag = ProductTag::create([
                            'title' => $tag,
                            'status' => 'publish',
                        ]);
                        $tags[] = $createdTag->id;
                    }
                }
            } else {
                foreach (json_decode($request->tags) as $tag) {
                    $findTag = ProductTag::where('title', $tag->value)->get()->first();
                    if ($findTag != null) {
                        $tags[] = $findTag->id;
                    } else {
                        $createdTag = ProductTag::create([
                            'title' => $tag->value,
                            'status' => 'publish',
                        ]);
                        $tags[] = $createdTag->id;
                    }
                }
            }


            $product->productTags()->sync($tags);
        }
    }

    /** get product trash list */
    public function getTrashList(Request $request)
    {
        /** @var LengthAwarePaginator */
        $products = Product::query()->where('status', 'deleted')->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->with('user')->paginate(20)->through(fn($product) => [
            'id' => $product->id,
            'user_id' => $product->user_id,
            'uuid' => $product->uuid,
            'title' => $product->title,
            'slug' => $product->slug,
            'get_images' => $product->get_images,
            'is_vip' => $product->is_vip,
            'get_affiliate' => $product->get_affiliate,
            'get_formatted_view_count' => $product->get_formatted_view_count,
            'consignment_items_count' => $product->consignment_items_count,
            'comments_info' => $product->comments_info,
            'questions_info' => $product->questions_info,
            'answers_info' => $product->answers_info,
            'jalali_updated_at' => $product->jalali_updated_at,
            'status' => $product->status,
            'user' => [
                'id' => $product->user->id,
                'get_name' => $product->user->get_name,
                'type' => $product->user->type,
                'seller' => $product->user->seller,
                'store' => $product->user->store != null ? [
                    'id' => $product->user->store->id,
                    'uuid' => $product->user->store->uuid,
                    'name' => $product->user->store->name,
                ] : null,
            ],
        ]);
        $products->withQueryString();

        return $products;
    }

    /********************** clone ********************/
    /** clone product price model */
    public function clonePriceModel(Product $original, Product $cloned)
    {
        $pmID = 0;

        // پاک کردن هر چی هست (احتیاطی)
        $cloned->productProps()->delete();

        foreach ($original->productProps as $pm) {
            $pmModel = $cloned->productProps()->create([
                'child' => $pmID,
                'name'  => $pm->name,
                'type'  => $pm->type,
            ]);

            $pmID = $pmModel->id;
        }
    }

    /** clone product specifications */
    public function cloneSpecifications(Product $original, Product $cloned)
    {
        foreach ($original->specificationValues as $spec) {
            $cloned->specificationValues()->create([
                'specification_key_id' => $spec->specification_key_id,
                'value'                => $spec->value,
            ]);
        }
    }

    /** clone product tags */
    public function cloneTags(Product $original, Product $cloned)
    {
        $tags = $original->productTags->pluck('id')->toArray();
        $cloned->productTags()->sync($tags);
    }
}
