<?php

namespace App\Services\Admin\Financial;

use App\Models\Transaction;
use Illuminate\Http\Request;

class TransactionsService
{
    /** get all transaction list */
    public function getTransactions(Request $request)
    {
        /** @var LengthAwarePaginator */
        $transactions = Transaction::query()->with('user')->when($request->input('search'), function ($query, $search) {
            $query->where('transaction_id', 'like', "%{$search}%");
        })->when($request->input('accepted'), function ($query, $accepted) {
            if ($accepted) {
                $query->where('status', 'accepted');
            }
        })->when($request->input('rejected'), function ($query, $rejected) {
            if ($rejected) {
                $query->where('status', 'rejected');
            }
        })->when($request->input('deposit'), function ($query, $deposit) {
            if ($deposit) {
                $query->where('amount', '>', 0);
            }
        })->when($request->input('withdrawal'), function ($query, $withdrawal) {
            if ($withdrawal) {
                $query->where('amount', '<', 0);
            }
        })->when($request->input('user'), function ($query, $user) {
            if ($user) {
                $query->whereIn('type', ['user_charge', 'coupon_charge', 'order_decrease']);
            }
        })->when($request->input('seller'), function ($query, $seller) {
            if ($seller) {
                $query->whereIn('type', ['seller_revenue', 'seller_returned_decrease']);
            }
        })->when($request->input('affiliate'), function ($query, $affiliate) {
            if ($affiliate) {
                $query->whereIn('type', ['affiliate_revenue', 'affiliate_returned_decrease']);
            }
        })->when($request->input('with_admin'), function ($query, $with_admin) {
            if ($with_admin) {
                $query->whereIn('type', ['admin_increase', 'admin_decrease']);
            }
        })->when($request->input('bank_withdrawal'), function ($query, $bank_withdrawal) {
            if ($bank_withdrawal) {
                $query->whereIn('type', ['bank_withdrawal']);
            }
        })->orderBy('created_at', 'desc')->paginate(20)->through(fn($transaction) => [
            'id' => $transaction->id,
            'transaction_id' => $transaction->transaction_id,
            'amount' => $transaction->amount,
            'type' => $transaction->type,
            'description' => $transaction->description,
            'get_formatted_amount' => $transaction->get_formatted_amount,
            'payment_gateway' => $transaction->payment_gateway,
            'gateway_reference_id' => $transaction->gateway_reference_id,
            'jalali_created_at' => $transaction->jalali_created_at,
            'status' => $transaction->status,
            'user' => [
                'id' => $transaction->user->id,
                'get_name' => $transaction->user->get_name,
                'type' => $transaction->user->type,
            ],
        ]);
        $transactions->withQueryString();

        return $transactions;
    }

    /** get seller transaction list */
    public function getSellerTransactions(Request $request)
    {
        /** @var LengthAwarePaginator */
        $transactions = Transaction::query()->with('user')->whereIn('type', ['seller_revenue', 'seller_returned_decrease'])->when($request->input('search'), function ($query, $search) {
            $query->where('transaction_id', 'like', "%{$search}%");
        })->when($request->input('accepted'), function ($query, $accepted) {
            if ($accepted) {
                $query->where('status', 'accepted');
            }
        })->when($request->input('rejected'), function ($query, $rejected) {
            if ($rejected) {
                $query->where('status', 'rejected');
            }
        })->when($request->input('income'), function ($query, $income) {
            if ($income) {
                $query->where('type', 'seller_revenue');
            }
        })->when($request->input('returned'), function ($query, $returned) {
            if ($returned) {
                $query->where('type', 'seller_returned_decrease');
            }
        })->when($request->input('withdrawal'), function ($query, $withdrawal) {
            if ($withdrawal) {
                $query->where('type', 'bank_withdrawal');
            }
        })->orderBy('created_at', 'desc')->paginate(20)->through(fn($transaction) => [
            'id' => $transaction->id,
            'transaction_id' => $transaction->transaction_id,
            'amount' => $transaction->amount,
            'type' => $transaction->type,
            'description' => $transaction->description,
            'get_formatted_amount' => $transaction->get_formatted_amount,
            'payment_gateway' => $transaction->payment_gateway,
            'gateway_reference_id' => $transaction->gateway_reference_id,
            'jalali_created_at' => $transaction->jalali_created_at,
            'status' => $transaction->status,
            'user' => [
                'id' => $transaction->user->id,
                'get_name' => $transaction->user->get_name,
                'type' => $transaction->user->type,
            ],
        ]);
        $transactions->withQueryString();

        return $transactions;
    }

    /** get affiliate transaction list */
    public function getAffiliateTransactions(Request $request)
    {
        /** @var LengthAwarePaginator */
        $transactions = Transaction::query()->with('user')->whereIn('type', ['affiliate_revenue', 'affiliate_returned_decrease'])->when($request->input('search'), function ($query, $search) {
            $query->where('transaction_id', 'like', "%{$search}%");
        })->when($request->input('accepted'), function ($query, $accepted) {
            if ($accepted) {
                $query->where('status', 'accepted');
            }
        })->when($request->input('rejected'), function ($query, $rejected) {
            if ($rejected) {
                $query->where('status', 'rejected');
            }
        })->when($request->input('income'), function ($query, $income) {
            if ($income) {
                $query->where('type', 'affiliate_revenue');
            }
        })->when($request->input('returned'), function ($query, $returned) {
            if ($returned) {
                $query->where('type', 'affiliate_returned_decrease');
            }
        })->when($request->input('withdrawal'), function ($query, $withdrawal) {
            if ($withdrawal) {
                $query->where('type', 'bank_withdrawal');
            }
        })->orderBy('created_at', 'desc')->paginate(20)->through(fn($transaction) => [
            'id' => $transaction->id,
            'transaction_id' => $transaction->transaction_id,
            'amount' => $transaction->amount,
            'type' => $transaction->type,
            'description' => $transaction->description,
            'get_formatted_amount' => $transaction->get_formatted_amount,
            'payment_gateway' => $transaction->payment_gateway,
            'gateway_reference_id' => $transaction->gateway_reference_id,
            'jalali_created_at' => $transaction->jalali_created_at,
            'status' => $transaction->status,
            'user' => [
                'id' => $transaction->user->id,
                'get_name' => $transaction->user->get_name,
                'type' => $transaction->user->type,
            ],
        ]);
        $transactions->withQueryString();

        return $transactions;
    }
}
