<?php

namespace App\Services\Admin\Blog;

use App\Http\Requests\Admin\ArticlesRequest;
use App\Models\Article;
use App\Models\ArticleTag;
use App\Models\Setting;
use Illuminate\Http\Request;

class ArticlesService
{
    /** get article articles list */
    public function getArticles(Request $request, $flag = "all")
    {
        /** @var LengthAwarePaginator */
        $articles = Article::query();

        if ($flag == "all") {
            $articles = $articles->where('status', '!=', 'deleted')
                ->where('status', '!=', 'deep_deleted')
                ->where('status', '!=', 'rejected');
        } else if ($flag == "awaiting") {
            $articles = $articles->where('status', 'awaiting');
        }

        $articles = $articles->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->when($request->input('publish'), function ($query, $publish) {
            if ($publish) {
                $query->where('status', 'publish');
            }
        })->when($request->input('awaiting'), function ($query, $awaiting) {
            if ($awaiting) {
                $query->where('status', 'awaiting');
            }
        })->when($request->input('draft'), function ($query, $draft) {
            if ($draft) {
                $query->where('status', 'draft');
            }
        })->when($request->input('mostViewer'), function ($query, $mostViewer) {
            if ($mostViewer) {
                $query->orderBy('view_count', 'desc');
            }
        })->orderByRaw("FIELD(status , 'awaiting') DESC")
            ->orderBy('created_at', 'desc')->with('user')->with('user.store')->paginate(20)->through(fn($article) => [
                'id' => $article->id,
                'user_id' => $article->user_id,
                'uuid' => $article->uuid,
                'title' => $article->title,
                'slug' => $article->slug,
                'get_image' => $article->get_image,
                'editor_choice' => $article->editor_choice,
                'get_formatted_view_count' => $article->get_formatted_view_count,
                'comments_info' => $article->comments_info,
                'vote' => $article->vote,
                'jalali_updated_at' => $article->jalali_updated_at,
                'status' => $article->status,
                'user' => [
                    'id' => $article->user->id,
                    'get_name' => $article->user->get_name,
                    'type' => $article->user->type,
                    'seller' => $article->user->seller,
                    'store' => $article->user->store != null ? [
                        'id' => $article->user->store->id,
                        'uuid' => $article->user->store->uuid,
                        'name' => $article->user->store->name,
                    ] : null,
                ],
            ]);

        $articles->withQueryString();

        return $articles;
    }

    /** validate slug */
    public function validateSlug(ArticlesRequest $request, Article $article = null)
    {
        $slug = null;

        if ($article != null)
            $slug = $article->slug;

        if ($request->exists('slug') && $request->slug != null && $request->slug != $slug) {
            $articleWithSlug = Article::where('slug', $request->slug)->first();
            if ($articleWithSlug != null)
                return false;
        }

        return true;
    }

    /** validate source */
    public function validateSource(ArticlesRequest $request, Article $article = null)
    {
        $source = Setting::where('key', 'shop_title')->first() != null ? Setting::where('key', 'shop_title')->first()->value : null;

        if ($article != null)
            $source = $article->source;

        if ($request->exists('source') && $request->source != null)
            $source = $request->source;

        return $source;
    }

    /** validate article type */
    public function validateType(ArticlesRequest $request, Article $article = null)
    {
        $type = $request->type;
        if ($request->type == "video") {
            $type = "video";
            $request->validate([
                'video' => 'required',
            ]);
        } else if ($request->type == "podcast") {
            $type = "podcast";
            $request->validate([
                'podcast' => 'required',
            ]);
        }

        return $type;
    }

    /** sync tags */
    public function syncTags(ArticlesRequest $request, Article $article)
    {
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            foreach (json_decode($request->tags) as $tag) {
                $findTag = ArticleTag::where('title', $tag->value)->get()->first();
                if ($findTag != null) {
                    $tags[] = $findTag->id;
                } else {
                    $createdTag = ArticleTag::create([
                        'title' => $tag->value,
                        'status' => 'publish',
                    ]);
                    $tags[] = $createdTag->id;
                }
            }
            $article->articleTags()->sync($tags);
        }
    }
}
