<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Cache;
use Laravel\Sanctum\HasApiTokens;
use Morilog\Jalali\Jalalian;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'username',
        'phone',
        'phone_verified_at',
        'email',
        'email_verified_at',
        'password',
        'avatar',
        'national_number',
        'birthday',
        'gender',
        'bio',
        'social',
        'seller',
        'type',
        'status',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'phone_verified_at' => 'datetime',
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'seller' => 'boolean',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'role',
        'get_name',
        'get_avatar',
        'get_social',
        'jalali_birthday',
        'articles_count',
        'products_count',
        'get_score',
        'get_level',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getRoleAttribute()
    {
        if ($this->type == 'owner') {
            return ['label' => __('messages.word.owner')];
        }
        return $this->roles()->first();
    }

    public function getGetNameAttribute()
    {
        return $this->name != null ? $this->name : $this->username;
    }

    public function getGetAvatarAttribute()
    {
        $avatar = asset('img/users/default-avatar.png');
        if ($this->avatar != null) {
            $file = Cache::remember('files_' . $this->avatar, now()->addMonth(), function () {
                return File::find($this->avatar);
            });
            if ($file != null && $file->type == "image") {
                $avatar = asset($file->url);
            }
        }
        return $avatar;
    }

    public function getGetSocialAttribute()
    {
        if ($this->social != null) {
            return unserialize($this->social);
        } else {
            return [
                'instagram' => null,
                'telegram' => null,
                'twitter' => null,
                'facebook' => null,
            ];
        }
    }

    public function getArticlesCountAttribute()
    {
        return Cache::remember('articles_count_user_' . $this->id, now()->addDay(), function () {
            return $this->articles()->count();
        });
    }

    public function getProductsCountAttribute()
    {
        return Cache::remember('products_count_user_' . $this->id, now()->addDay(), function () {
            return $this->products()->count();
        });
    }

    public function getGetScoreAttribute()
    {
        $scoreIncrease = Cache::remember('score_increase_' . $this->id, now()->addDay(), function () {
            return $this->scores()->where('type', 'increase')->get();
        });
        $scoreDecrease = Cache::remember('score_decrease_' . $this->id, now()->addDay(), function () {
            return $this->scores()->where('type', 'decrease')->get();
        });

        $score = $scoreIncrease->sum('score') - $scoreDecrease->sum('score');

        return [
            'score' => $score < 0 ? 0 : $score,
            'formatted_score' => number_format($score < 0 ? 0 : $score),
        ];
    }

    public function getGetLevelAttribute()
    {
        // get customer club levels
        $silverLevel = Cache::remember('setting_silver_level', now()->addMonth(), function () {
            return Setting::where('key', 'silver_level')->first();
        });
        $silverLevel = $silverLevel != null ? str_replace(',', '', $silverLevel->value) : null;
        $goldLevel = Cache::remember('setting_gold_level', now()->addMonth(), function () {
            return Setting::where('key', 'gold_level')->first();
        });
        $goldLevel = $goldLevel != null ? str_replace(',', '', $goldLevel->value) : null;

        $level = 'bronze';
        if ($goldLevel != null && $this->get_score['score'] >= $goldLevel) {
            $level = 'gold';
        } else if ($silverLevel != null && $this->get_score['score'] >= $silverLevel) {
            $level = 'silver';
        }

        return $level;
    }

    public function getJalaliBirthdayAttribute()
    {
        $date = [
            Jalalian::forge($this->birthday)->format('%d %B %Y'),
            Jalalian::forge($this->birthday)->format('Y/m/d'),
            Jalalian::forge($this->birthday)->format('H:i - Y/m/d'),
            Jalalian::forge($this->birthday)->ago(),
            Jalalian::forge($this->birthday)->getTimestamp(),
        ];
        return $this->birthday != null ? $date : null;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'name' => $this->name,
            'get_name' => $this->get_name,
            'username' => $this->username,
            'avatar' => $this->avatar,
            'get_avatar' => $this->get_avatar,
            'jalali_birthday' => $this->jalali_birthday,
            'gender' => $this->gender,
            'bio' => $this->bio,
            'social' => $this->social,
            'get_social' => $this->get_social,
            'get_score' => $this->get_score,
            'get_level' => $this->get_level,
            'seller' => $this->seller,
            'type' => $this->type,
            'role' => $this->role != null ? $this->role['label'] : null,
            'articles_count' => $this->articles_count,
            'products_count' => $this->products_count,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
        ];
    }
    /** end append */

    /** ACL / Roles - Permission */
    public function hasRole($role)
    {
        if ($this->type == "owner") {
            return true;
        }

        if (is_string($role)) {
            return $this->roles->contains('name', $role);
        }

        return !!$role->intersect($this->roles)->count();
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($user) {
            Cache::forget('article_user_' . $user->id);
            Cache::forget('comment_user_' . $user->id);
        });

        static::deleted(function ($user) {
            Cache::forget('article_user_' . $user->id);
            Cache::forget('comment_user_' . $user->id);
        });
    }

    /* relationships **************/
    public function tokens(): HasMany
    {
        return $this->hasMany(Token::class);
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class);
    }

    public function reports(): HasMany
    {
        return $this->hasMany(Report::class);
    }

    public function files(): HasMany
    {
        return $this->hasMany(File::class);
    }

    public function products(): HasMany
    {
        return $this->hasMany(Product::class);
    }

    public function brands(): HasMany
    {
        return $this->hasMany(Brand::class);
    }

    public function store(): HasOne
    {
        return $this->hasOne(Store::class);
    }

    public function productFavorites(): HasMany
    {
        return $this->hasMany(ProductFavorite::class);
    }

    public function tickets(): HasMany
    {
        return $this->hasMany(Ticket::class);
    }

    public function ticketAnswers(): HasMany
    {
        return $this->hasMany(TicketAnswer::class);
    }

    public function addresses(): HasMany
    {
        return $this->hasMany(Address::class);
    }

    public function wallet(): HasOne
    {
        return $this->hasOne(Wallet::class);
    }

    public function transactions(): HasMany
    {
        return $this->hasMany(Transaction::class);
    }

    public function searches(): HasMany
    {
        return $this->hasMany(Search::class);
    }

    public function discounts(): BelongsToMany
    {
        return $this->belongsToMany(Discount::class);
    }

    public function productNotifications(): HasMany
    {
        return $this->hasMany(ProductNotification::class);
    }

    public function productComments(): HasMany
    {
        return $this->hasMany(ProductComment::class);
    }

    public function questions(): HasMany
    {
        return $this->hasMany(Question::class);
    }

    public function questionAnswers(): HasMany
    {
        return $this->hasMany(QuestionAnswer::class);
    }

    public function articles(): HasMany
    {
        return $this->hasMany(Article::class);
    }

    public function articleComments(): HasMany
    {
        return $this->hasMany(ArticleComment::class);
    }

    public function scores(): HasMany
    {
        return $this->hasMany(Score::class);
    }

    public function compares(): HasMany
    {
        return $this->hasMany(Compare::class);
    }

    public function statistics(): HasMany
    {
        return $this->hasMany(Statistic::class);
    }

    public function orderGroups(): HasMany
    {
        return $this->hasMany(OrderGroup::class);
    }

    public function orders(): HasMany
    {
        return $this->hasMany(Order::class);
    }

    public function consignments(): HasMany
    {
        return $this->hasMany(Consignment::class);
    }

    public function withdrawals(): HasMany
    {
        return $this->hasMany(Withdrawal::class);
    }

    public function bankCards(): HasMany
    {
        return $this->hasMany(BankCard::class);
    }

    public function stories(): HasMany
    {
        return $this->hasMany(Story::class);
    }

    public function prefactors(): HasMany
    {
        return $this->hasMany(Prefactor::class);
    }

    public function pendingPayments(): HasMany
    {
        return $this->hasMany(PendingPayment::class);
    }
}
