<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class ProductFavorite extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'product_id',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'product_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_product',
        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getGetProductAttribute()
    {
        return [
            'title' => $this->product->title,
            'slug' => $this->product->slug,
            'status' => $this->product->status,
            'get_images' => $this->product->get_images,
            'in_stock_status' => $this->product->in_stock_status,
            'inventory' => $this->product->inventory,
            'best_price' => $this->product->best_price,
            'unique_color' => $this->product->unique_color,
            'comments_info' => $this->product->comments_info,
        ];
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }
    /** end append */

    protected $primaryKey = ['product_id', 'user_id'];
    public $incrementing = false;
    protected $keyType = 'string';

    protected function setKeysForSaveQuery($query)
    {
        $keys = $this->getKeyName();
        if (!is_array($keys)) {
            $keys = [$keys];
        }
        foreach ($keys as $key) {
            $query->where($key, '=', $this->getAttribute($key));
        }
        return $query;
    }

    public function getKeyName()
    {
        return $this->primaryKey;
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($favorite) {
            Cache::forget('product_is_favorite_' . $favorite->product_id . '_' . $favorite->user_id);
        });

        static::deleted(function ($favorite) {
            Cache::forget('product_is_favorite_' . $favorite->product_id . '_' . $favorite->user_id);
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }
}
