<?php

namespace App\Models;

use Cviebrock\EloquentSluggable\Sluggable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class ProductCategory extends Model
{
    use HasFactory, Sluggable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'parent_id',
        'sort_id',

        'title',
        'slug',
        'content',
        'image',

        'seo_title',
        'seo_description',
        'seo_keywords',
        'seo_canonical',

        'view_count',
        'status',
    ];

    protected $casts = [
        'parent_id' => 'integer',
        'sort_id' => 'integer',
        'view_count' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_specifications',
        'childs',
        'get_image',
        'get_formatted_view_count',
        'get_seo_keywords',
        'jalali_created_at',
        'jalali_updated_at',

        'simple_safe',
        'safe',
    ];

    public function getGetSpecificationsAttribute()
    {
        return Cache::remember('get_category_specifications_' . $this->id, now()->addMonth(), function () {
            return $this->specifications()->get()->map(fn($specification) => [
                'id' => $specification->id,
                'title' => $specification->title,
                'keys' => $specification->keys,
            ])->toArray();
        });
    }

    public function getChildsAttribute()
    {
        return $this->where('status', 'publish')->where('parent_id', $this->id)->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'get_image' => $category->get_image,
            'specifications' => $category->get_specifications,
            'get_formatted_view_count' => $category->get_formatted_view_count,
            'products_count' => $category->products()->count(),
            'level' => 'two',
            'childs' => ProductCategory::where('status', 'publish')->where('parent_id', $category->id)->get()->map(fn($childCategory) => [
                'id' => $childCategory->id,
                'title' => $childCategory->title,
                'slug' => $childCategory->slug,
                'get_image' => $childCategory->get_image,
                'specifications' => $childCategory->get_specifications,
                'get_formatted_view_count' => $childCategory->get_formatted_view_count,
                'products_count' => $childCategory->products()->count(),
                'level' => 'three',
                'childs' => null,
            ]),
        ]);
    }

    public function getGetImageAttribute()
    {
        $image = asset('img/temp/pre-load-category.png');
        if ($this->image != null) {
            $file = Cache::remember('files_' . $this->image, now()->addMonth(), function () {
                return File::find($this->image);
            });
            if ($file != null && $file->type == "image") {
                $image = asset($file->url);
            }
        }
        return $image;
    }

    public function getGetFormattedViewCountAttribute()
    {
        return number_format($this->view_count);
    }

    public function getGetSeoKeywordsAttribute()
    {
        $seo_keywords = [];
        foreach (($this->seo_keywords != null ? json_decode($this->seo_keywords) : []) as $keywords) {
            $seo_keywords[] = $keywords->value;
        }

        return implode(',', $seo_keywords);
    }

    public function getAllChildrenIds()
    {
        $ids = [];

        $children = ProductCategory::where('parent_id', $this->id)->get();

        foreach ($children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $child->getAllChildrenIds());
        }

        return $ids;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSimpleSafeAttribute()
    {
        return [
            'id' => $this->id,
            'parent_id' => $this->parent_id,
            'sort_id' => $this->sort_id,
            'title' => $this->title,
            'slug' => $this->slug,
            'get_image' => $this->get_image,
            'get_formatted_view_count' => $this->get_formatted_view_count,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'level' => $this->parent_id == 0 ? 'one' : (ProductCategory::where('id', $this->parent_id)->first() != null && ProductCategory::where('id', $this->parent_id)->first()->parent_id == 0 ? 'two' : 'three'),
            'specifications' => $this->get_specifications,
        ];
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'parent_id' => $this->parent_id,
            'sort_id' => $this->sort_id,
            'title' => $this->title,
            'slug' => $this->slug,
            'content' => $this->content,
            'get_image' => $this->get_image,
            'seo_title' => $this->seo_title,
            'seo_description' => $this->seo_description,
            'seo_keywords' => $this->seo_keywords,
            'get_seo_keywords' => $this->get_seo_keywords,
            'seo_canonical' => $this->seo_canonical,
            'get_formatted_view_count' => $this->get_formatted_view_count,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'childs' => $this->childs,
            'level' => $this->parent_id == 0 ? 'one' : (ProductCategory::where('id', $this->parent_id)->first() != null && ProductCategory::where('id', $this->parent_id)->first()->parent_id == 0 ? 'two' : 'three'),
            'specifications' => $this->get_specifications,
        ];
    }
    /** end append */

    /**
     * Return the sluggable configuration array for this model.
     *
     * @return array
     */
    public function sluggable(): array
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($category) {
            if (array_key_exists('view_count', $category->getDirty())) return;

            // all header categories
            Cache::forget('main_categories');

            // main schema categories
            Cache::forget('main_schema_product_categories');

            // main categories list
            Cache::forget('main_categories_simple_list');
            Cache::forget('main_categories_with_product');
            Cache::forget('main_categories_with_product_' . $category->id);
            Cache::forget('category_for_products_list_' . $category->id);
            Cache::forget('category_products_list_' . $category->id);
        });

        static::deleted(function ($category) {
            // all header categories
            Cache::forget('main_categories');

            // main schema categories
            Cache::forget('main_schema_product_categories');

            // main categories list
            Cache::forget('main_categories_simple_list');
            Cache::forget('main_categories_with_product');
            Cache::forget('main_categories_with_product_' . $category->id);
            Cache::forget('category_for_products_list_' . $category->id);
            Cache::forget('category_products_list_' . $category->id);
        });
    }

    /* relationships **************/
    public function products(): HasMany
    {
        return $this->hasMany(Product::class, 'category_id');
    }

    public function specifications(): HasMany
    {
        return $this->hasMany(Specification::class, 'category_id');
    }
}
