<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class PendingPayment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'order_id',
        'amount',
        'data',
        'type',
        'status',
        'paid_at',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'order_id' => 'integer',
        'amount' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_data',
        'jalali_paid_at',
        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getGetDataAttribute()
    {
        return $this->data != null ? unserialize($this->data) : [];
    }

    public function getJalaliPaidAtAttribute()
    {
        // get days from settings
        $releaseOfOrderAmounts = Cache::remember('setting_release_of_order_amounts', now()->addMonth(), function () {
            return Setting::where('key', 'release_of_order_amounts')->first();
        });
        $releaseOfOrderAmounts = $releaseOfOrderAmounts != null ? (int)$releaseOfOrderAmounts->value : 7;

        $createdAt = Carbon::parse($this->order->sent_at);
        $targetDate = $createdAt->copy()->addDays((int) $releaseOfOrderAmounts);
        $now = Carbon::now();

        if ($this->status == 'paid' && $this->paid_at != null) {
            return Jalalian::forge($this->paid_at)->format('H:i - Y/m/d');
        } else if ($this->order->status == 'delivered') {
            return null;
        } else {
            // calculate remaining date
            $diff = $now->diff($targetDate);
            $parts = [];
            if ($diff->d > 0) {
                $parts[] = $diff->d . ' روز';
            }
            if ($diff->h > 0) {
                $parts[] = $diff->h . ' ساعت';
            }
            if ($diff->i > 0 && count($parts) < 2) {
                $parts[] = $diff->i . ' دقیقه';
            }

            return implode(' و ', $parts) . ' مانده';
        }
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }
    /** end append */

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }
}
