<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class PaymentMeta extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'store_id',

        'reception_centers_addresses',
        'post_ways',
        'post_period',
        'free_post_min_price',
        'description',
    ];

    protected $casts = [
        'store_id' => 'integer',
        'free_post_min_price' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_reception_centers_addresses',
        'get_post_ways',
        'get_formatted_free_post_min_price',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getGetReceptionCentersAddressesAttribute()
    {
        return $this->reception_centers_addresses != null ? unserialize($this->reception_centers_addresses) : [];
    }

    public function getGetPostWaysAttribute()
    {
        return $this->post_ways != null ? unserialize($this->post_ways) : [];
    }

    public function getGetFormattedFreePostMinPriceAttribute()
    {
        return $this->free_post_min_price != null ? number_format($this->free_post_min_price) : null;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'store_id' => $this->store_id,
            'post_ways' => $this->post_ways,
            'post_period' => $this->post_period,
            'free_post_min_price' => $this->free_post_min_price,
            'get_post_ways' => $this->get_post_ways,
            'get_formatted_free_post_min_price' => $this->get_formatted_free_post_min_price,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
            'store' => $this->store != null ? $this->store->safe : null,
        ];
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function () {
            Cache::forget('payment_metas');
        });

        static::deleted(function () {
            Cache::forget('payment_metas');
        });
    }

    /* relationships **************/
    public function store(): BelongsTo
    {
        return $this->belongsTo(Store::class);
    }
}
