<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Morilog\Jalali\Jalalian;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;

class Order extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'id',
        'user_id',
        'order_group_id',

        'uuid',
        'send_by',
        'discount',
        'address',
        'send_method',
        'delivery_date',
        'post_tracking_code',
        'description',
        'abandoned_cart',
        'is_locked',
        'status',
        'sent_at',
    ];

    protected $casts = [
        'id' => 'integer',
        'user_id' => 'integer',
        'order_group_id' => 'integer',
        'send_by' => 'integer',
        'is_locked' => 'boolean',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_send_by',
        'get_discount',
        'get_address',
        'get_send_method',
        'is_awaiting_request_from_seller',
        'is_awaiting_seller',
        'is_awaiting_receipt_from_seller',
        'is_ready_to_send',
        'is_all_item_cancelled',
        'get_shares',
        'get_consolidated_shares',

        'get_total_bill',

        'jalali_sent_at',
        'jalali_created_at',
        'jalali_updated_at',

        'safe'
    ];

    public function getGetSendByAttribute()
    {
        if ($this->send_by != null) {
            $store = Store::find($this->send_by);
            return $store != null ? [
                'id' => $store->id,
                'uuid' => $store->uuid,
                'name' => $store->name,
            ] : __('messages.word.seller');
        } else {
            return null;
        }
    }

    public function getGetDiscountAttribute()
    {
        return $this->discount != null ? unserialize($this->discount) : null;
    }

    public function getGetAddressAttribute()
    {
        return $this->address != null ? unserialize($this->address) : null;
    }

    public function getGetSendMethodAttribute()
    {
        return $this->send_method != null ? unserialize($this->send_method) : null;
    }

    public function getIsAwaitingRequestFromSellerAttribute()
    {
        $orderId = $this->id;

        $arfs = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', 'awaiting_request_from_seller')->count();

        return $arfs > 0;
    }

    public function getIsAwaitingSellerAttribute()
    {
        $orderId = $this->id;

        $as = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', 'awaiting_seller')->count();

        return $as > 0;
    }

    public function getIsAwaitingReceiptFromSellerAttribute()
    {
        $orderId = $this->id;

        $arfs = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', 'awaiting_receipt_from_seller')->count();

        return $arfs > 0;
    }

    public function getIsReadyToSendAttribute()
    {
        $orderId = $this->id;

        $readyToSendCount = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', 'ready_to_send')->count();

        $allNotCancelledCount = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', '!=', 'cancelled')->count();

        return $readyToSendCount == $allNotCancelledCount;
    }

    public function getIsAllItemCancelledAttribute()
    {
        $orderId = $this->id;

        $allItems = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->count();

        $allCancelledCount = ConsignmentItem::whereHas('consignment', function ($query) use ($orderId) {
            $query->where('order_id', $orderId);
        })->where('status', 'cancelled')->count();

        return $allItems == $allCancelledCount;
    }

    public function getGetSharesAttribute()
    {
        // get order consignment
        $consignments = $this->consignments()->get()->toArray();
        $consignments = collect($consignments);

        return [
            'shop' => $consignments->sum('get_shares.shop'),
            'seller' => $consignments->sum('get_shares.seller'),
            'affiliate' => $consignments->sum('get_shares.affiliate'),
        ];
    }

    public function getGetConsolidatedSharesAttribute()
    {
        // get order consignment
        $consignments = $this->consignments()->get();

        // get seller and affilate information
        $sellerInfo = [];
        $sellerInfoTemp = [];
        $affiliateInfo = [];
        $affiliateInfoTemp = [];
        foreach ($consignments as $consignment) {
            foreach ($consignment->consignmentItems()->where('status', '!=', 'cancelled')->where('status', '!=', 'returned')->get() as $item) {
                if ($item->get_shares['seller_information'] != null) {
                    $store = Store::find($item->get_shares['seller_information']);
                    $sellerInfoTemp[] = [
                        'share' => $item->get_shares['seller'],
                        'consignment' => $consignment->uuid,
                        'information' => ['id' => $store->id, 'uuid' => $store->uuid, 'name' => $store->name],
                    ];
                }
                if ($item->get_shares['affiliate_information'] != null) {
                    $affiliateInfoTemp[] = [
                        'share' => $item->get_shares['affiliate'],
                        'product' => [
                            'id' => $item->product->id,
                            'uuid' => $item->product->uuid,
                            'title' => $item->product->title,
                            'slug' => $item->product->slug,
                        ],
                        'information' => $item->get_shares['affiliate_information']['user'],
                    ];
                }
            }
        }

        // group and validate seller information
        $sellerInfoTemp = new Collection($sellerInfoTemp);
        $sellerInfoTemp = $sellerInfoTemp->groupBy('information.uuid');
        foreach ($sellerInfoTemp as $value) {
            $dataTemp = [
                'share' => $value->sum('share'),
                'consignment' => $value[0]['consignment'],
                'information' => $value[0]['information'],
            ];

            $sellerInfo[] = $dataTemp;
        }

        // group and validate affiliate information
        $affiliateInfoTemp = new Collection($affiliateInfoTemp);
        $affiliateInfoTemp = $affiliateInfoTemp->groupBy('information.id');
        foreach ($affiliateInfoTemp as $value) {
            $dataTemp = [
                'share' => $value->sum('share'),
                'product' => $value[0]['product'],
                'information' => $value[0]['information'],
            ];

            $affiliateInfo[] = $dataTemp;
        }

        return [
            'seller' => $sellerInfo,
            'affiliate' => $affiliateInfo,
        ];
    }

    public function getGetTotalBillAttribute()
    {
        // get order consignment
        $consignments = $this->consignments()->get()->toArray();
        $consignments = collect($consignments);

        $sendPrice = 0;
        if ($this->get_send_method != null && $this->get_send_method['type'] == 'post' && array_key_exists('price', $this->get_send_method) && $this->get_send_method['price'] != null && !$this->get_send_method['free']) {
            $sendPrice = (int)str_replace(',', '', $this->get_send_method['price']);
        }

        return [
            'price' => $consignments->sum('get_total_bill.price'),
            'discount' => $consignments->sum('get_total_bill.discount'),
            'count' => $consignments->sum('get_total_bill.count'),
            'send_price' => $sendPrice,

            'cancel' => [
                'price' => $consignments->sum('get_total_cancel_bill.price'),
            ],
            'final' => $this->status == 'sent' || $this->status == 'delivered' ? [
                'price' => $consignments->sum('get_total_final_bill.price'),
                'discount' => $consignments->sum('get_total_final_bill.discount'),
                'send_price' => $sendPrice,
                'count' => $consignments->sum('get_total_final_bill.count'),
            ] : [],
            'awaiting' => $this->status != 'cancelled' && $this->status != 'returned' ? [
                'price' => $consignments->sum('get_total_final_bill.price'),
                'discount' => $consignments->sum('get_total_final_bill.discount'),
                'send_price' => $sendPrice,
                'count' => $consignments->sum('get_total_final_bill.count'),
            ] : []
        ];
    }

    public function getJalaliSentAtAttribute()
    {
        $date = [
            Jalalian::forge($this->sent_at)->format('%d %B %Y'),
            Jalalian::forge($this->sent_at)->format('Y/m/d'),
            Jalalian::forge($this->sent_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->sent_at)->ago(),
            Jalalian::forge($this->sent_at)->getTimestamp(),
        ];
        return $date != null ? $date : null;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'get_discount' => $this->get_discount,
            'get_address' => $this->get_address,
            'get_send_method' => $this->get_send_method,
            'get_total_bill' => $this->get_total_bill,
            'is_all_item_cancelled' => $this->is_all_item_cancelled,
            'post_tracking_code' => $this->post_tracking_code,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
            'user' => [
                'id' => $this->user->id,
                'get_name' => $this->user->get_name,
                'username' => $this->user->username,
                'get_avatar' => $this->user->get_avatar,
            ]
        ];
    }
    /** end append */

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function orderGroup(): BelongsTo
    {
        return $this->belongsTo(OrderGroup::class);
    }

    public function consignments(): HasMany
    {
        return $this->hasMany(Consignment::class);
    }

    public function pendingPayments(): HasMany
    {
        return $this->hasMany(PendingPayment::class);
    }
}
