<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class File extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'name',
        'url',
        'description',
        'extension',
        'size',
        'resolution',
        'type',
        'visibility',
    ];

    protected $casts = [
        'user_id' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'jalali_created_at',
        'jalali_updated_at',
    ];

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::deleted(function ($file) {
            // delete file cache
            Cache::forget('files_' . $file->id);

            // delete settings cache
            $settingsCaches = [
                'settings',
                ...array_map(function ($setting) {
                    return 'setting_' . $setting['key'];
                }, Setting::select(['key'])->get()->makeHidden(['get_image', 'jalali_created_at', 'jalali_updated_at'])->toArray())
            ];
            foreach ($settingsCaches as $cache)
                Cache::forget($cache);

            // delete template cache
            Cache::forget('template');
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }
}
