<?php

namespace App\Models;

use Cviebrock\EloquentSluggable\Sluggable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class ArticleTag extends Model
{
    use HasFactory, Sluggable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'title',
        'slug',

        'view_count',
        'status',
    ];

    protected $casts = [
        'view_count' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'articles_count',
        'get_formatted_view_count',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getArticlesCountAttribute()
    {
        return $this->articles()->count();
    }

    public function getGetFormattedViewCountAttribute()
    {
        return number_format($this->view_count);
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            'slug' => $this->slug,
            'view_count' => $this->view_count,
            'get_formatted_view_count' => $this->get_formatted_view_count,
            'status' => $this->status,
        ];
    }
    /** end append */

    /**
     * Return the sluggable configuration array for this model.
     *
     * @return array
     */
    public function sluggable(): array
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function () {
            Cache::forget('get_hashtags');
        });

        static::deleted(function () {
            Cache::forget('get_hashtags');
        });
    }

    /* relationships **************/
    public function articles(): BelongsToMany
    {
        return $this->belongsToMany(Article::class);
    }
}
