<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class ArticleComment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'article_id',
        'content',
        'reply_to',
        'vote_up',
        'vote_down',
        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'article_id' => 'integer',
        'reply_to' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_user',
        'answers',
        'get_reply_to',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getGetUserAttribute()
    {
        return Cache::remember('comment_user_' . $this->user_id, now()->addMonth(), function () {
            return [
                'id' => $this->user->id,
                'get_name' => $this->user->get_name,
                'username' => $this->user->username,
                'get_avatar' => $this->user->get_avatar,
                'role' => $this->user->role != null ? $this->user->role['label'] : null,
                'type' => $this->user->type,
                'seller' => $this->user->seller,
                'store' => $this->user->store != null ? [
                    'id' => $this->user->store->id,
                    'get_logo' => $this->user->store->get_logo,
                    'name' => $this->user->store->name,
                    'uuid' => $this->user->store->uuid,
                ] : null,
            ];
        });
    }

    public function getAnswersAttribute()
    {
        return Cache::remember('comment_answers_' . $this->id, now()->addMonth(), function () {
            return ArticleComment::where('status', 'publish')->where('reply_to', $this->id)->orderBy('created_at', 'asc')->get();
        })->map(fn($comment) => [
            'safe' => $comment->safe
        ]);
    }

    public function getGetReplyToAttribute()
    {
        if ($this->reply_to != null) {
            $replyToComment = Cache::remember('get_reply_to_' . $this->id, now()->addMonth(), function () {
                return ArticleComment::where('id', $this->reply_to)->first();
            });
            return $replyToComment != null ? [
                'user' => $replyToComment->get_user,
            ] : null;
        }

        return null;
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'content' => $this->content,
            'reply_to' => $this->reply_to,
            'vote_up' => $this->vote_up,
            'vote_down' => $this->vote_down,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'get_reply_to' => $this->get_reply_to,
            'answers' => $this->answers,
            'user' => $this->get_user,
            'article' => $this->article->safe,
        ];
    }
    /** end append */

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($comment) {
            // article appends
            Cache::forget('article_comments_info_' . $comment->article_id);
            Cache::forget('get_reply_to_' . $comment->id);
            Cache::forget('comment_answers_' . $comment->id);
            Cache::forget('comment_answers_' . $comment->reply_to);

            Cache::forget('comment_user_' . $comment->user_id);
        });

        static::deleted(function ($comment) {
            // article appends
            Cache::forget('article_comments_info_' . $comment->article_id);
            Cache::forget('get_reply_to_' . $comment->id);
            Cache::forget('comment_answers_' . $comment->id);
            Cache::forget('comment_answers_' . $comment->reply_to);

            Cache::forget('comment_user_' . $comment->user_id);
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function article(): BelongsTo
    {
        return $this->belongsTo(Article::class);
    }
}
