<?php

namespace App\Models;

use Cviebrock\EloquentSluggable\Sluggable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class ArticleCategory extends Model
{
    use HasFactory, Sluggable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'title',
        'slug',
        'image',

        'seo_title',
        'seo_description',
        'seo_keywords',
        'seo_canonical',

        'view_count',
        'status',
    ];

    protected $casts = [
        'view_count' => 'integer',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_image',
        'get_last_article_image',
        'get_formatted_view_count',
        'get_seo_keywords',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getGetLastArticleImageAttribute()
    {
        $image = [
            'url' => asset('img/temp/pre-load-article.png'),
            'alt' => __('messages.word.article')
        ];
        $article = $this->articles()->first();
        if ($article != null) {
            $file = Cache::remember('files_' . $article->image, now()->addMonth(), function () use ($article) {
                return File::find($article->image);
            });
            if ($file != null && $file->type == "image") {
                $image = [
                    'url' => asset($file->url),
                    'alt' => $article->title,
                ];
            }
        }
        return $image;
    }

    public function getGetImageAttribute()
    {
        $image = asset('img/temp/pre-load-category.png');
        if ($this->image != null) {
            $file = Cache::remember('files_' . $this->image, now()->addMonth(), function () {
                return File::find($this->image);
            });
            if ($file != null && $file->type == "image") {
                $image = asset($file->url);
            }
        }
        return $image;
    }

    public function getGetFormattedViewCountAttribute()
    {
        return number_format($this->view_count);
    }

    public function getGetSeoKeywordsAttribute()
    {
        $seo_keywords = [];
        foreach (($this->seo_keywords != null ? json_decode($this->seo_keywords) : []) as $keywords) {
            $seo_keywords[] = $keywords->value;
        }

        return implode(',', $seo_keywords);
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'title' => $this->title,
            'slug' => $this->slug,
            'get_image' => $this->get_image,
            'get_last_article_image' => $this->get_last_article_image,
            'seo_title' => $this->seo_title,
            'seo_description' => $this->seo_description,
            'seo_keywords' => $this->seo_keywords,
            'get_seo_keywords' => $this->get_seo_keywords,
            'seo_canonical' => $this->seo_canonical,
            'get_formatted_view_count' => $this->get_formatted_view_count,
            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,
        ];
    }
    /** end append */

    /**
     * Return the sluggable configuration array for this model.
     *
     * @return array
     */
    public function sluggable(): array
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($category) {
            if (array_key_exists('view_count', $category->getDirty())) return;

            Cache::forget('article_category_' . $category->id);

            // blog categories list
            Cache::forget('blog_category_simple_list');
        });

        static::deleted(function ($category) {
            Cache::forget('article_category_' . $category->id);

            // blog categories list
            Cache::forget('blog_category_simple_list');
        });
    }

    /* relationships **************/
    public function articles(): HasMany
    {
        return $this->hasMany(Article::class, 'category_id');
    }
}
