<?php

namespace App\Models;

use Cviebrock\EloquentSluggable\Sluggable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Cache;
use Morilog\Jalali\Jalalian;

class Article extends Model
{
    use HasFactory, Sluggable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'edited_by',
        'brand_id',
        'category_id',

        'uuid',
        'title',
        'slug',
        'content',
        'source',
        'read_time',
        'type',

        'image',
        'video',
        'podcast',

        'seo_title',
        'seo_description',
        'seo_keywords',
        'seo_canonical',

        'view_count',
        'comment_status',
        'editor_choice',
        'vote',

        'reason',
        'status',
    ];

    protected $casts = [
        'user_id' => 'integer',
        'edited_by' => 'integer',
        'brand_id' => 'integer',
        'category_id' => 'integer',

        'view_count' => 'integer',
        'vote' => 'integer',

        'comment_status' => 'boolean',
        'editor_choice' => 'boolean',
    ];

    /**
     * additional data
     */
    protected $appends = [
        'get_user',
        'get_content',
        'get_image',
        'get_video',
        'get_podcast',
        'get_formatted_view_count',
        'get_category',
        'get_tags_string',
        'comments_info',
        'is_voted',
        'get_seo_keywords',
        'jalali_created_at',
        'jalali_updated_at',

        'safe',
    ];

    public function getGetUserAttribute()
    {
        return Cache::remember('article_user_' . $this->user_id, now()->addMonth(), function () {
            return [
                'id' => $this->user->id,
                'get_name' => $this->user->get_name,
                'get_avatar' => $this->user->get_avatar,
                'username' => $this->user->username,
                'bio' => $this->user->bio,
                'get_social' => $this->user->get_social,
                'type' => $this->user->type,
                'role' => $this->user->role != null ? $this->user->role['label'] : null,
                'articles_count' => $this->user->articles_count,
                'jalali_created_at' => $this->user->jalali_created_at,
                'status' => $this->user->status,
                'seller' => $this->user->seller,
                'store' => $this->user->store != null ? [
                    'id' => $this->user->store->id,
                    'get_logo' => $this->user->store->get_logo,
                    'name' => $this->user->store->name,
                    'uuid' => $this->user->store->uuid,
                    'bio' => $this->user->store->bio,
                ] : null,
            ];
        });
    }

    public function getGetContentAttribute()
    {
        $content = $this->content;

        $pattern = '/\[(.*?)\]/';
        $matches = array();

        preg_match_all($pattern, $this->content, $matches);

        // dd($matches);

        foreach ($matches[1] as $uuid) {
            $product = Cache::remember('article_product_' . $uuid, now()->addMonth(), function () use ($uuid) {
                return Product::where('uuid', $uuid)->first();
            });

            if ($product != null && $product->status == 'publish') {
                $mention = '
                <div class="w-100 d-flex">
                <div class="card product-horizontal go-to-product" route="' . route('main.products.show', $product->slug) . '">
                    <div class="card-body">
                        <div class="product-horizontal--content d-flex flex-row align-items-center">
                            <img src="' . $product->get_images[0]['link'] . '" alt="' . $product->title . '" />
                            <div class="product-horizontal--information">
                                <h2>
                                    <a href="javascript:void(0)">' . $product->title . '</a>
                                </h2>
                                <div class="product-horizontal--footer d-flex flex-row align-items-center justify-content-between">
                                    <ul class="product-colors el-center">';
                foreach ($product->unique_color as $color) {
                    $mention .= '
                                        <li
                                            style="background-color: ' . $color['value']['color'] . '; ' . ($color['value']['color'] == '#ffffff' ? 'border: solid 1px #ddd' : '') . '"
                                            data-bs-toggle="tooltip"
                                            data-bs-placement="top"
                                            title="' . $color['value']['label'] . '"></li>';
                }
                $mention .= '
                                    </ul>
                                    <div class="product-footer-meta d-inline-flex align-items-end">
                                        <span class="product-rating-average">' . $product->comments_info['rating'] . ' <i class="ri-star-fill"></i></span>
                                        <span class="divider">|</span>';

                if (!$product->in_stock_status) {
                    $mention .=         '<span class="product-price-now fs-9 lts-05">' . __('messages.sentence.inventory_inquiry') . '</span>';
                } else if ($product->inventory) {
                    $mention .=         '<span class="product-price-now lts-05">
                                            ' . number_format($product->best_price) . '
                                            <span class="unit unit-sm"></span>
                                        </span>';
                } else {
                    $mention .=         '<span class="product-price-now fs-8 lts-05">' . __('messages.word.non_existent') . '</span>';
                }

                $mention .= '
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                </div>
                ';

                $content = str_replace('[' . $uuid . ']', $mention, $content);
            } else {
                $content = str_replace('[' . $uuid . ']', '', $content);
            }
        }

        $content = str_replace('<p>&nbsp;</p>', '<p style="line-height: 0;">&nbsp;</p>', $content);
        $content = str_replace('<p></p>', '', $content);

        return $content;
    }

    public function getGetImageAttribute()
    {
        $image = [
            'url' => asset('img/temp/pre-load-article.png'),
            'alt' => __('messages.word.no_image'),
        ];
        if ($this->image != null) {
            $file = Cache::remember('files_' . $this->image, now()->addMonth(), function () {
                return File::find($this->image);
            });
            if ($file != null && $file->type == "image") {
                $image = [
                    'url' => asset($file->url),
                    'alt' => asset($file->description),
                ];
            }
        }
        return $image;
    }

    public function getGetVideoAttribute()
    {
        return $this->video;
    }

    public function getGetPodcastAttribute()
    {
        return $this->podcast;
    }

    public function getGetFormattedViewCountAttribute()
    {
        return number_format($this->view_count);
    }

    public function getGetCategoryAttribute()
    {
        return Cache::remember('article_category_' . $this->category_id, now()->addMonth(), function () {
            return [
                'id' => $this->articleCategory->id,
                'title' => $this->articleCategory->title,
                'slug' => $this->articleCategory->slug,
                'get_image' => $this->articleCategory->get_image,
            ];
        });
    }

    public function getGetTagsStringAttribute()
    {
        $tags = [];
        foreach ($this->articleTags()->pluck('title')->toArray() as $tag) {
            $tags[] = [
                'value' => $tag
            ];
        }
        return json_encode($tags, \JSON_UNESCAPED_UNICODE);
    }

    public function getCommentsInfoAttribute()
    {
        $info = Cache::remember('article_comments_info_' . $this->id, now()->addMonth(), function () {
            return [
                'count' => number_format($this->articleComments()->count()),
            ];
        });

        return $info;
    }

    public function getIsVotedAttribute()
    {
        $votedArticles = session()->get('voted_articles', []);
        if (array_key_exists($this->id, $votedArticles)) {
            return true;
        }
        return false;
    }

    public function getGetSeoKeywordsAttribute()
    {
        $seo_keywords = [];
        foreach (($this->seo_keywords != null ? json_decode($this->seo_keywords) : []) as $keywords) {
            $seo_keywords[] = $keywords->value;
        }

        return implode(',', $seo_keywords);
    }

    public function getJalaliCreatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->created_at)->format('%d %B %Y'),
            Jalalian::forge($this->created_at)->format('Y/m/d'),
            Jalalian::forge($this->created_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->created_at)->ago(),
            Jalalian::forge($this->created_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getJalaliUpdatedAtAttribute()
    {
        $date = [
            Jalalian::forge($this->updated_at)->format('%d %B %Y'),
            Jalalian::forge($this->updated_at)->format('Y/m/d'),
            Jalalian::forge($this->updated_at)->format('H:i - Y/m/d'),
            Jalalian::forge($this->updated_at)->ago(),
            Jalalian::forge($this->updated_at)->getTimestamp(),
        ];
        return $date;
    }

    public function getSafeAttribute()
    {
        return [
            'id' => $this->id,
            'uuid' => $this->uuid,
            'title' => $this->title,
            'slug' => $this->slug,
            'get_content' => $this->get_content,
            'content' => $this->content,
            'source' => $this->source,
            'read_time' => $this->read_time,
            'type' => $this->type,

            'image' => $this->image,
            'get_image' => $this->get_image,
            'video' => $this->video,
            'get_video' => $this->get_video,
            'podcast' => $this->podcast,
            'get_podcast' => $this->get_podcast,

            'seo_title' => $this->seo_title,
            'seo_description' => $this->seo_description,
            'seo_keywords' => $this->seo_keywords,
            'get_seo_keywords' => $this->get_seo_keywords,
            'seo_canonical' => $this->seo_canonical,

            'view_count' => $this->view_count,
            'get_formatted_view_count' => $this->get_formatted_view_count,
            'get_category' => $this->get_category,
            'comment_status' => $this->comment_status,
            'comments_info' => $this->comments_info,
            'editor_choice' => $this->editor_choice,
            'vote' => $this->vote,
            'is_voted' => $this->is_voted,

            'status' => $this->status,
            'jalali_created_at' => $this->jalali_created_at,
            'jalali_updated_at' => $this->jalali_updated_at,

            'get_user' => $this->get_user,
            'user' => $this->user->safe,
            'brand' => $this->brand != null ? $this->brand->safe : null,
            'category' => $this->articleCategory->safe,
            'tags' => $this->articleTags()->where('status', 'publish')->get()->map(fn($tag) => [
                'safe' => $tag->safe
            ])
        ];
    }
    /** end append */

    /**
     * Return the sluggable configuration array for this model.
     *
     * @return array
     */
    public function sluggable(): array
    {
        return [
            'slug' => [
                'source' => 'title'
            ]
        ];
    }

    /**
     * Get the route key for the model.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }

    /**
     * remove cache data in update
     */
    protected static function boot()
    {
        parent::boot();

        static::saved(function ($article) {
            if (array_key_exists('view_count', $article->getDirty())) return;

            // article all data
            Cache::forget('article_' . $article->slug);

            Cache::forget('articles_count_user_' . $article->user_id);
            Cache::forget('article_user_' . $article->user_id);

            Cache::forget('blog_category_simple_list');
        });

        static::deleted(function ($article) {
            // article all data
            Cache::forget('article_' . $article->slug);

            Cache::forget('articles_count_user_' . $article->user_id);
            Cache::forget('article_user_' . $article->user_id);

            Cache::forget('blog_category_simple_list');
        });
    }

    /* relationships **************/
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function editedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'edited_by');
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function articleCategory(): BelongsTo
    {
        return $this->belongsTo(ArticleCategory::class, 'category_id');
    }

    public function articleTags(): BelongsToMany
    {
        return $this->belongsToMany(ArticleTag::class);
    }

    public function articleComments(): HasMany
    {
        return $this->hasMany(ArticleComment::class);
    }
}
