<?php

namespace App\Imports;

use App\Helpers\UUIDHelper;
use App\Models\Brand;
use App\Models\File;
use App\Models\Product;
use App\Models\ProductCategory;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File as FacadesFile;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\Drivers\Gd\Encoders\WebpEncoder;
use Intervention\Image\ImageManager;
use Maatwebsite\Excel\Concerns\ToCollection;

class ProductsImport implements ToCollection
{
    use UUIDHelper;

    public function collection(Collection $collection)
    {
        // کاربر فعلی
        $user = auth()->user();

        foreach ($collection as $key => $row) {
            // رد کردن اولین حلقه
            if ($key == 0) continue;

            // دریافت اطلاعات ردیف اکسل
            $actionType = $row[0];
            $uuid = $row[1] != null ? $row[1] : $this->generateUniqueRandomNumber(8, \App\Models\Product::class, 'uuid', 'p-');
            $title = $row[2];
            $titleEn = $row[3];
            $slug = $row[4];
            $category = $this->getCategoryId($row[5]);
            $brand = $this->getBrandId($row[6]);
            $warnings = $row[7];
            $content = $row[8];
            $guide = $row[9];
            $images = $this->jsonToArray($row[10]);
            $attrs = $this->jsonToArray($row[11]);
            $seoTitle = $row[12];
            $seoDescription = $row[13];
            $viewCount = $row[14];
            $commission = $row[15];
            $isVip = $row[16];
            $priceModels = $this->jsonToArray($row[17]);
            $status = $row[18];

            // بررسی فیلد های ضروری
            if ($title == null || $category == null) continue;

            // تنظیم تصاویر
            $resultIds = [];
            if ($images != null) {
                foreach ($images as $item) {
                    $link = $item['link'];
                    $alt = $item['alt'];

                    // جدا کردن نام فایل از لینک
                    $fileName = basename(parse_url($link, PHP_URL_PATH));

                    // جستجو توی مدل File بر اساس نام فایل
                    $file = File::where('name', $fileName)->first();

                    if ($file) {
                        // اگر فایل پیدا شد، id را اضافه کن
                        $resultIds[] = $file->id;
                    } else {
                        // اگر فایل پیدا نشد، متد uploadFromUrl را صدا بزن
                        $uploaded = $this->uploadFromUrl($link, $alt, $user);

                        if ($uploaded && isset($uploaded['id'])) {
                            $resultIds[] = $uploaded['id'];
                        }
                    }
                }
                $images = $resultIds;
            } else {
                $images = [];
            }

            // اطلاعات پایه
            $data = [
                'edited_by'        => $user->id,
                'title'            => $title,
                'title_en'         => $titleEn,
                'slug'             => $slug,
                'category_id'      => $category,
                'brand_id'         => $brand,
                'warnings'         => $warnings,
                'content'          => $content != null ? $content : '',
                'guide'            => $guide,
                'images'           => serialize($images),
                'attrs'            => $attrs,
                'seo_title'        => $seoTitle,
                'seo_description'  => $seoDescription,
                'in_stock_status'  => true,
                'inquiry_options'  => 'a:2:{s:14:"landline_phone";N;s:14:"cellular_phone";N;}',
                'view_count'       => $viewCount != null ? $viewCount : 0,
                'commission'       => $commission != null ? $commission : 0,
                'is_vip'           => $isVip == '1' ? true : false,
                'reason'           => '',
                'status'           => $status == 'publish' ? 'publish' : 'draft',
            ];

            // اگر وجود دارد، آپدیت کن، در غیر این صورت بساز
            $product = Product::where('uuid', $uuid)->first();
            if ($product && $actionType == 'update') {
                // اعتبارسنجی ویژگی های برجسته
                $attrs = $attrs != null ? serialize($attrs) : $product->attrs;
                $data['attrs'] = $attrs;

                $product->update($data);
            } else {
                if ($product) continue;

                // افزودن شناسه به داده ها
                $data['uuid'] = $uuid;

                // اعتبارسنجی ویژگی های برجسته
                $attrs = $attrs != null ? serialize($attrs) : null;
                $data['attrs'] = $attrs;

                $product = $user->products()->create($data);
            }

            // افزودن متغییر های فروش
            if ($product->inventories()->count() == 0) {
                if ($priceModels != null && is_array($priceModels)) {
                    $pmID = 0;
                    $product->productProps()->delete();
                    foreach ($priceModels as $pm) {
                        if ($pm['active'] && $pm['name'] != null) {
                            $pmModel = $product->productProps()->create([
                                'child' => $pmID,
                                'name' => $pm['name'],
                                'type' => $pm['type'],
                            ]);

                            $pmID = $pmModel->id;
                        }
                    }
                }
            }
        }

        try {
            // delete cache
            FacadesFile::deleteDirectory(storage_path('framework/cache/data'));
            Cache::flush();

            // delete views
            $views = FacadesFile::files(storage_path('framework/views'));
            foreach ($views as $view) {
                FacadesFile::delete($view);
            }
        } catch (\Exception $e) {
        }
    }

    /** تبدیل جیسون به ارایه */
    protected function jsonToArray($value)
    {
        if (is_null($value) || $value === '') {
            return null;
        }

        $decoded = json_decode($value, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            return $decoded;
        }

        return null;
    }

    /** یافتن آیدی دسته بندی از روی نام دسته بندی */
    protected function getCategoryId($categoryTitle)
    {
        if (!$categoryTitle) return null;
        $category = ProductCategory::where('title', $categoryTitle)->first();
        return $category ? $category->id : null;
    }

    /** یافتن آیدی برند از روی نام برند */
    protected function getBrandId($brandTitle)
    {
        if (!$brandTitle) return null;
        $brand = Brand::where('title', $brandTitle)->first();
        return $brand ? $brand->id : null;
    }

    /** اپلود تصویر از لینک به حافظه */
    private function uploadFromUrl($url, $alt, $user)
    {
        try {
            // دانلود فایل با timeout
            $response = Http::timeout(2)->get($url);

            $imageContent = $response->body();

            // ذخیره موقت
            $tmpFile = tempnam(sys_get_temp_dir(), 'img');
            file_put_contents($tmpFile, $imageContent);

            // گرفتن پسوند اصلی
            $fileInfo = pathinfo(parse_url($url, PHP_URL_PATH));
            $originalExtension = strtolower($fileInfo['extension'] ?? 'jpg');
            $fileType = match ($originalExtension) {
                'jpeg', 'jpg', 'png', 'webp', 'gif', 'svg', 'avif' => 'image',
                default => null,
            };

            if ($fileType !== 'image') {
                return null;
            }

            // پردازش تصویر
            $manager = ImageManager::withDriver(new Driver());
            $imageTemp = $manager->read($tmpFile);
            $imageResolution = $imageTemp->width() . '×' . $imageTemp->height();

            // تبدیل به webp و فشرده‌سازی
            $optimizationPercent = 30;
            $finalImageTemp = $imageTemp->encode(new WebpEncoder($optimizationPercent))->__toString();
            $fileFormat = 'webp';

            // مسیر ذخیره‌سازی
            $fileName = 'file.' . time() . '.' . rand(10000, 99999) . '.' . $fileFormat;
            $fileSaveAddressNoName = 'library/' . date('Y-m-d') . '/' . $user->id;
            $fileSaveAddressImage = $fileSaveAddressNoName . '/' . $fileName;

            $uploadResult = Storage::disk('media')->put($fileSaveAddressImage, $finalImageTemp, 'public');

            if ($uploadResult) {
                return $user->files()->create([
                    'name'        => $fileName,
                    'description' => $alt,
                    'url'         => '/media/' . $fileSaveAddressImage,
                    'extension'   => $fileFormat,
                    'size'        => strlen($finalImageTemp) / 1000,
                    'resolution'  => $imageResolution,
                    'type'        => $fileType,
                ])->toArray();
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }
}
