<?php

namespace App\Http\Controllers\User\Orders;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\OrderGroup;
use Illuminate\Http\Request;
use Inertia\Inertia;

class OrdersController extends Controller
{
    /**
     * orders list
     */
    public function index(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** @var LengthAwarePaginator */
        $orderGroups = $user->orderGroups()->when($request->input('search'), function ($query, $search) {
            $query->where('uuid', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->paginate(20);

        $orderGroups = $orderGroups->through(fn($og) => [
            'id' => $og->id,
            'transaction_id' => $og->transaction_id,
            'uuid' => $og->uuid,
            'status' => $og->status,
            'jalali_created_at' => $og->jalali_created_at,
            'jalali_updated_at' => $og->jalali_updated_at,
            'user' => [
                'id' => $og->user->id,
                'get_name' => $og->user->get_name,
            ],
            'orders' => $og->orders->map(fn($order) => [
                'id' => $order->id,
                'uuid' => $order->uuid,
                'send_by' => $order->send_by,
                'get_send_by' => $order->get_send_by,
                'status' => $order->status,
                'jalali_created_at' => $order->jalali_created_at,
                'jalali_updated_at' => $order->jalali_updated_at,
            ]),
        ]);

        $orderGroups->withQueryString();

        return Inertia::render('User/Orders/List', [
            'orderGroups' => $orderGroups,
        ]);
    }

    /**
     * show order details
     */
    public function show(Order $order)
    {
        if ($order->status == 'awaiting') {
            abort(404);
        }

        if ($order->status == 'prefactor') {
            abort(404);
        }

        if ($order->user_id != auth()->user()->id) {
            abort(403);
        }

        return Inertia::render('User/Orders/Show', [
            'order' => [
                'id' => $order->id,
                'uuid' => $order->uuid,
                'get_address' => $order->get_address,
                'post_tracking_code' => $order->post_tracking_code,
                'get_total_bill' => $order->get_total_bill,
                'get_discount' => $order->get_discount,
                'get_send_method' => $order->get_send_method,
                'is_all_item_cancelled' => $order->is_all_item_cancelled,
                'description' => $order->description,
                'jalali_created_at' => $order->jalali_created_at,
                'status' => $order->status,

                'consignments' => $order->consignments()->get()->map(fn($consignment) => [
                    'id' => $consignment->id,
                    'status' => $consignment->status,

                    'consignment_items' => $consignment->consignmentItems()->get()->map(fn($item) => [
                        'id' => $item->id,
                        'title' => $item->title,
                        'get_props' => $item->get_props,
                        'get_full_count_price' => $item->get_full_count_price,
                        'count' => $item->count,
                        'reason' => $item->reason,
                        'status' => $item->status,

                        'product' => $item->product != null ? [
                            'id' => $item->product->id,
                            'title' => $item->product->title,
                            'slug' => $item->product->slug,
                            'get_images' => $item->product->get_images,
                        ] : null,
                    ]),
                    'store' => $consignment->store != null ? [
                        'id' => $consignment->store->id,
                        'uuid' => $consignment->store->uuid,
                        'name' => $consignment->store->name,
                    ] : null,
                ]),
                'user' => [
                    'id' => $order->user->id,
                    'get_name' => $order->user->get_name,
                ],
            ],
        ]);
    }

    /**
     * order sales factor
     */
    public function sales_factor(Order $order)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($order->user_id != $user->id) {
            abort(403);
        }

        $orderConsignmentItems = [];

        $consignments = $order->consignments()->where(function ($query) {
            $query->where('status', 'sent_to_customer')->orWhere('status', 'direct_sent_to_customer')->orWhere('status', 'delivered');
        })->get();

        foreach ($consignments as $consignment) {
            $orderConsignmentItems = array_merge($orderConsignmentItems, $consignment->consignmentItems()->where(function ($query) {
                $query->where('status', 'sent')->orWhere('status', 'delivered');
            })->get()->map(fn($item) => [
                'id' => $item->id,
                'title' => $item->title,
                'get_props' => $item->get_props,
                'price' => $item->price,
                'discount' => $item->discount,
                'count' => $item->count,
                'product' => $item->product != null ? [
                    'id' => $item->product->id,
                    'uuid' => $item->product->uuid,
                    'title' => $item->product->title,
                ] : null,
            ])->toArray());
        }

        return Inertia::render('User/Orders/SalesFactor', [
            'order' => $order->safe,
            'consignmentItems' => $orderConsignmentItems,
            'user_national_number' => $order->user->national_number
        ]);
    }
}
