<?php

namespace App\Http\Controllers\User\Addresses;

use App\Helpers\Places;
use App\Models\Address;
use App\Http\Controllers\Controller;
use App\Http\Requests\User\AddressRequest;
use Illuminate\Http\Request;
use Inertia\Inertia;

class AddressesController extends Controller
{
    use Places;
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        /** user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        $addresses = $user->addresses()->latest()->get()->map(fn($address) => [
            'id' => $address->id,
            'user_id' => $address->user_id,
            'first_name' => $address->first_name,
            'last_name' => $address->last_name,
            'state' => $address->state,
            'city' => $address->city,
            'phone' => $address->phone,
            'postal_code' => $address->postal_code,
            'full_address' => $address->full_address,
            'active' => $address->active,
            'get_map' => $address->get_map,
            'jalali_created_at' => $address->jalali_created_at,
            'jalali_updated_at' => $address->jalali_updated_at,
        ]);

        return Inertia::render('User/Addresses/List', [
            'places' => $this->places(),
            'addresses' => $addresses
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(AddressRequest $request)
    {
        /** user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        /** postal code validation */
        if (strlen(\request('postal_code')) != 10) {
            return redirect()->back()->withErrors([__('messages.postal_code_wrong')]);
        }

        /** map longitude & latitude */
        $map = [];
        if ($request->exists('map') && is_array($request->map)) {
            $map = $request->map;
        }

        /** check active status */
        $active = false;
        if ($user->addresses()->count() == 0) {
            $active = true;
        }

        $user->addresses()->create([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'state' => $request->state,
            'city' => $request->city,
            'phone' => $request->phone,
            'postal_code' => $request->postal_code,
            'full_address' => $request->full_address,
            'map' => serialize($map),
            'active' => $active
        ]);

        return redirect()->back()->with('message', [__('messages.address_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Address $address)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Address $address)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(AddressRequest $request, Address $address)
    {
        /** user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        /** check user validation */
        if ($address->user_id != $user->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** postal code validation */
        if (strlen(\request('postal_code')) != 10) {
            return redirect()->back()->withErrors([__('messages.postal_code_wrong')]);
        }

        /** map longitude & latitude */
        $map = [];
        if ($request->exists('map') && is_array($request->map)) {
            $map = $request->map;
        }

        $address->update([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'state' => $request->state,
            'city' => $request->city,
            'phone' => $request->phone,
            'postal_code' => $request->postal_code,
            'full_address' => $request->full_address,
            'map' => serialize($map)
        ]);

        return redirect()->back()->with('message', [__('messages.address_edited')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Address $address)
    {
        /** user
         * @var App\Models\User $user
         */
        $user = auth()->user();

        /** check user validation */
        if ($address->user_id != $user->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        $address->delete();

        /** if this is active address, set other active */
        if ($user->addresses()->where('active', true)->first() == null) {
            if ($user->addresses()->count() > 0) {
                $user->addresses()->first()->update([
                    'active' => true
                ]);
            }
        }

        return redirect()->back()->with('message', [__('messages.address_deleted')]);
    }

    /**
     * change active address
     */
    public function change(Address $address, Request $request)
    {
        /** user
         * @var App\Models\User $user
         */
        $user = auth()->user();

        /** check user validation */
        if ($address->user_id != $user->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        foreach ($user->addresses()->where('id', '!=', $address->id)->get() as $adrs) {
            $adrs->update([
                'active' => false
            ]);
        }

        $address->update([
            'active' => true
        ]);

        return redirect()->back()->with('message', [__('messages.address_activated')]);
    }
}
