<?php

namespace App\Http\Controllers\Store\Tickets;

use App\Helpers\DropdownListHelper;
use App\Models\Ticket;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TicketsController extends Controller
{
    use DropdownListHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** @var LengthAwarePaginator */
        $tickets = $user->tickets()->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->when($request->input('waiting_admin'), function ($query, $waiting_admin) {
            if ($waiting_admin) {
                $query->where('status', 'waiting_admin');
            }
        })->when($request->input('waiting_user'), function ($query, $waiting_user) {
            if ($waiting_user) {
                $query->where('status', 'waiting_user');
            }
        })->when($request->input('closed'), function ($query, $closed) {
            if ($closed) {
                $query->where('status', 'closed');
            }
        })->orderByRaw("FIELD(status , 'waiting_user', 'waiting_admin', 'closed') ASC")
            ->orderByRaw("FIELD(priority , 'high', 'normal', 'low') ASC")
            ->orderBy('updated_at', 'asc')->paginate(20)->through(fn($ticket) => [
                'id' => $ticket->id,
                'jalali_created_at' => $ticket->jalali_created_at,
                'jalali_updated_at' => $ticket->jalali_updated_at,
                'priority' => $ticket->priority,
                'status' => $ticket->status,
                'title' => $ticket->title,
                'user_id' => $ticket->user_id,
            ]);

        $tickets->withQueryString();

        return Inertia::render('Store/Tickets/List', [
            'tickets' => $tickets
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Store/Tickets/Create', [
            'priorityList' => $this->ticketPriorityList()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        if ($user->type == "admin" || $user->type == "owner") {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.admin_can_not_create_ticket')]);
        }

        $request->validate([
            'title' => 'required',
            'priority' => 'required',
            'content' => 'required',
        ]);

        $ticket = $user->tickets()->create([
            'title' => $request->title,
            'priority' => $request->priority,
            'status' => 'waiting_admin',
        ]);

        if ($ticket != null) {
            $ticket->ticketAnswers()->create([
                'user_id' => $user->id,
                'type' => 'text',
                'content' => $request->content,
                'status' => 'publish',
            ]);

            if ($request->exists('attachment') && $request->attachment != null &&  $request->attachment != '') {
                $ticket->ticketAnswers()->create([
                    'user_id' => $user->id,
                    'type' => 'file',
                    'file' => $request->attachment,
                    'status' => 'publish',
                ]);
            }
        } else {
            return redirect()->back()->withErrors([__('messages.ticket_create_error')]);
        }

        if ($ticket != null) {
            return redirect()->route('seller.tickets.show', $ticket->id)->with('message', [__('messages.ticket_created')]);
        } else {
            return redirect()->back()->withErrors([__('messages.ticket_create_error')]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Ticket $ticket)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        if ($user->id != $ticket->user_id) {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.unauthorized')]);
        }

        $ticket = [
            'id' => $ticket->id,
            'user_id' => $ticket->user_id,
            'title' => $ticket->title,
            'priority' => $ticket->priority,
            'status' => $ticket->status,
            'jalali_created_at' => $ticket->jalali_created_at,
            'jalali_updated_at' => $ticket->jalali_updated_at,
            'ticket_answers' => $ticket->ticket_answers->map(fn($answer) => [
                'id' => $answer['id'],
                'user_id' => $answer['user_id'],
                'content' => $answer['content'],
                'type' => $answer['type'],
                'get_file' => $answer['get_file'],
                'jalali_created_at' => $answer['jalali_created_at'],
                'user' => [
                    'id' => $answer['user']['id'],
                    'get_name' => $answer['user']['get_name'],
                    'get_avatar' => $answer['user']['get_avatar'],
                    'type' => $answer['user']['type'],
                    'seller' => $answer['user']['seller'],
                ],
            ])
        ];

        return Inertia::render('Store/Tickets/Show', [
            'ticket' => $ticket
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Ticket $ticket)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Ticket $ticket)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Ticket $ticket)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();

        if ($user->id != $ticket->user_id) {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($ticket->status != "closed") {
            $ticket->update([
                'status' => 'closed'
            ]);

            return redirect()->back()->with('message', [__('messages.ticket_closed')]);
        }
    }

    /** send answer */
    public function answer(Request $request, Ticket $ticket)
    {
        if ($ticket->status == "closed") {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.ticket_is_closed')]);
        }

        /** @var App\Models\User $user */
        $user = auth()->user();

        $request->validate([
            'content' => 'required',
        ]);

        if ($user->id != $ticket->user_id) {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.unauthorized')]);
        }

        $answer = $ticket->ticketAnswers()->create([
            'user_id' => $user->id,
            'type' => 'text',
            'content' => $request->content,
            'status' => 'publish',
        ]);

        if ($answer != null) {
            $ticket->update([
                'status' => 'waiting_admin'
            ]);
        } else {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        return redirect()->back()->with('message', [__('messages.ticket_answer_created')]);
    }

    /** send attachment */
    public function attachment(Request $request, Ticket $ticket)
    {
        if ($ticket->status == "closed") {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.ticket_is_closed')]);
        }

        /** @var App\Models\User $user */
        $user = auth()->user();

        $request->validate([
            'attachment' => 'required',
        ]);

        if ($user->id != $ticket->user_id) {
            return redirect()->route('seller.tickets.index')->withErrors([__('messages.unauthorized')]);
        }

        $attachment = $ticket->ticketAnswers()->create([
            'user_id' => $user->id,
            'type' => 'file',
            'file' => $request->attachment,
            'status' => 'publish',
        ]);

        if ($attachment != null) {
            $ticket->update([
                'status' => 'waiting_admin'
            ]);
        } else {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        return redirect()->back()->with('message', [__('messages.ticket_attachment_created')]);
    }
}
