<?php

namespace App\Http\Controllers\Store\Storage;

use App\Exports\StorageExport;
use App\Models\Storage;
use App\Http\Controllers\Controller;
use App\Imports\StorageImport;
use App\Models\Brand;
use App\Models\Product;
use App\Models\ProductCategory;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Maatwebsite\Excel\Facades\Excel;

class StorageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** storages list */
        $storages = Storage::query()->where('store_id', $store->id)->when($request->input('search'), function ($query, $search) {
            $query->where('name', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->get()->map(fn($storage) => [
            'id' => $storage->id,
            'store_id' => $storage->store_id,
            'uuid' => $storage->uuid,
            'name' => $storage->name,
            'address' => $storage->address,
            'product_count' => $storage->product_count,
        ]);

        return Inertia::render('Store/Storage/List', [
            'storages' => $storages
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        /** check payment meta */
        /*if ($store->paymentMeta == null) {
            return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);
        } else {
            if (count($store->paymentMeta->get_post_ways) == 0 || $store->paymentMeta->post_period == null || $store->paymentMeta->free_post_min_price == null) {
                return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);
            }
        }*/

        $request->validate([
            'name' => 'required',
            'address' => 'required',
        ]);

        $store->storages()->create([
            'uuid' => 'storage-' . rand(1000000, 9999999),
            'name' => $request->name,
            'address' => $request->address,
        ]);

        /** Report Action */
        $reportMessage = __("messages.report_storage_created", ['storage' => $request->name]);
        $this->report($reportMessage);
        /******************/

        return redirect()->route('seller.storages.index')->with('message', [__('messages.storage_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, Storage $storage)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->id != $storage->store_id) {
            return redirect()->route('seller.storages.index')->withErrors(['messages.unauthorized']);
        }

        /** @var LengthAwarePaginator */
        $products = Product::with('inventories')->where('status', '!=', 'waiting')->where('status', '!=', 'deleted')->where('status', '!=', 'deep_deleted')->where('status', '!=', 'rejected')
            ->whereHas('inventories', function ($query) use ($storage) {
                $query->where('storage_id', $storage->id);
            })->when($request->input('search'), function ($query, $search) {
                $query->where('title', 'like', "%{$search}%");
            })->when($request->input('publish'), function ($query, $publish) {
                if ($publish) {
                    $query->where('status', 'publish');
                }
            })->when($request->input('awaiting'), function ($query, $awaiting) {
                if ($awaiting) {
                    $query->where('status', 'awaiting');
                }
            })->when($request->input('draft'), function ($query, $draft) {
                if ($draft) {
                    $query->where('status', 'draft');
                }
            })->when($request->input('hasPrice'), function ($query, $hasPrice) use ($storage) {
                if ($hasPrice) {
                    $query->whereHas('inventories', function ($query) use ($storage) {
                        $query->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0);
                    });
                }
            })->when($request->input('outofstock'), function ($query, $outofstock) use ($storage) {
                if ($outofstock) {
                    $query->whereHas('inventories', function ($query) use ($storage) {
                        $query->where('storage_id', $storage->id)->where('status', 'publish');
                    })->whereDoesntHave('inventories', function ($query) use ($storage) {
                        $query->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0);
                    });
                }
            })->orderBy('created_at', 'desc')->paginate(20)->through(fn($product) => [
                'id' => $product->id,
                'user_id' => $product->user_id,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'is_vip' => $product->is_vip,
                'inventory_in_storage' => $product->inventories()->where('storage_id', $storage->id)->get()->count(),
                'inventory_count_in_storage' => $product->inventories()->where('storage_id', $storage->id)->get()->sum('count'),
                'inventory_status' => $this->inventoryStatus($product, $storage),
                'status' => $product->status,
                'user' => [
                    'id' => $product->user->id,
                    'get_name' => $product->user->get_name,
                    'type' => $product->user->type,
                    'seller' => $product->user->seller,
                    'store' => $product->user->store != null ? $product->user->store->safe : null
                ]
            ]);
        $products->withQueryString();

        return Inertia::render('Store/Storage/Show', [
            'storage' => $storage,
            'products' => $products,
            'categories' => $this->getCategoriesBranch(),
            'brands' => Brand::where('status', 'publish')->get()->map(fn($brand) => [
                'id' => $brand->id,
                'title' => $brand->title,
            ]),
        ]);
    }

    /** get category branch */
    public function getCategoriesBranch()
    {
        $categoriesWithChilds = [];
        $categories = ProductCategory::where('status', 'publish')->where('parent_id', 0)->select('id', 'title', 'slug', 'parent_id')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'specifications' => $category->specifications()->get()->map(fn($specification) => [
                'id' => $specification->id,
                'title' => $specification->title,
                'keys' => $specification->specificationKeys()->get()->map(fn($specificationKey) => [
                    'id' => $specificationKey->id,
                    'key' => $specificationKey->key,
                    'value' => $specificationKey->default_value,
                ])->toArray()
            ])->toArray(),
            'childs' => $category->childs,
            'level' => 'one',
        ])->toArray();

        foreach ($categories as $category) {
            $categoriesWithChilds[] = $category;
            foreach ($category['childs'] as $childLevelTwo) {
                $categoriesWithChilds[] = $childLevelTwo;
                foreach ($childLevelTwo['childs'] as $childLevelThree) {
                    $categoriesWithChilds[] = $childLevelThree;
                }
            }
        }

        return $categoriesWithChilds;
    }

    private function inventoryStatus(Product $product, Storage $storage)
    {
        return [
            'inventory' => $product->inventories()->where('storage_id', $storage->id)->where('status', 'publish')->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->exists(),
            'instock' => $product->inventories()->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0)->where(function ($query) {
                $query->doesntHave('store')->orWhereHas('store', function ($query) {
                    $query->where('status', 'active');
                });
            })->exists(),
            'inquery' => !$product->in_stock_status
        ];
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Storage $storage)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Storage $storage)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->id != $storage->store_id) {
            return redirect()->route('seller.storages.index')->withErrors(['messages.unauthorized']);
        }

        $request->validate([
            'name' => 'required',
            'address' => 'required',
        ]);

        $storage->update([
            'name' => $request->name,
            'address' => $request->address,
        ]);

        /** Report Action */
        $reportMessage = __("messages.report_storage_updated", ['storage' => $storage->name]);
        $this->report($reportMessage);
        /******************/

        return redirect()->route('seller.storages.index')->with('message', [__('messages.storage_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Storage $storage)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->id != $storage->store_id) {
            return redirect()->route('seller.storages.index')->withErrors(['messages.unauthorized']);
        }

        $storageName = $storage->name;

        if ($storage->product_count == 0) {
            $storage->delete();

            /** Report Action */
            $reportMessage = __("messages.report_storage_deleted", ['storage' => $storageName]);
            $this->report($reportMessage);
            /******************/

            return redirect()->route('seller.storages.index')->with('message', [__('messages.storage_deleted')]);
        }

        return redirect()->route('seller.storages.index')->withErrors([__('messages.storage_has_product_can_not_delete')]);
    }

    /**
     * storage inventories list
     */
    public function storageList(Storage $storage, Product $product)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($storage->store_id != $store->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        // inventories
        $inventories = $product->inventories()->where('storage_id', $storage->id)
            ->orderByRaw("FIELD(status , 'awaiting', 'publish', 'deactive') ASC")
            ->orderBy('created_at', 'desc')->get()->map(fn($inventory) => [
                'id' => $inventory->id,
                'storage' => $inventory->storage->name,
                'storage_id' => $inventory->storage_id,
                'store' => $inventory->store != null ? $inventory->store->safe : null,
                'store_id' => $inventory->store_id,
                'uuid' => $inventory->uuid,
                'price' => $inventory->price,
                'get_formatted_price' => $inventory->get_formatted_price,
                'get_final_price' => $inventory->get_final_price,
                'discount_price' => $inventory->discount_price,
                'get_formatted_discount_price' => $inventory->get_formatted_discount_price,
                'jalali_discount_expire' => $inventory->jalali_discount_expire,
                'is_discount_valid' => $inventory->is_discount_valid,
                'get_image' => $inventory->get_image,
                'description' => $inventory->description,
                'get_discount_tree' => $inventory->get_discount_tree,
                'count' => $inventory->count,
                'get_formatted_count' => $inventory->get_formatted_count,
                'min_sale' => $inventory->min_sale,
                'get_formatted_min_sale' => $inventory->get_formatted_min_sale,
                'max_sale' => $inventory->max_sale,
                'get_formatted_max_sale' => $inventory->get_formatted_max_sale,
                'weight' => $inventory->weight,
                'send_time' => $inventory->send_time,
                'purchase_price' => $inventory->purchase_price,
                'status' => $inventory->status,
                'props' => $inventory->props,
            ]);

        // other inventories count
        $inventories_count = [
            'you' => $product->inventories()->where('store_id', $store->id)->get()->count() - count($inventories),
            'other' => $product->inventories()->get()->count() - $product->inventories()->where('store_id', $store->id)->get()->count(),
        ];

        // product
        $product = [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'title' => $product->title,
            'slug' => $product->slug,
            'get_images' => $product->get_images,
            'is_vip' => $product->is_vip,
            'price_model' => $product->price_model,
            'commission' => $product->commission,
        ];

        return Inertia::render('Store/Storage/Inventory/StorageList', [
            'storage' => $storage,
            'product' => $product,
            'inventories' => $inventories,
            'inventories_count' => $inventories_count
        ]);
    }

    /**
     * export storage
     */
    public function export(Storage $storage, Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($storage->store_id != $store->id) {
            return redirect()->route('seller.storages.index')->withErrors(['messages.unauthorized']);
        }

        $request->validate([
            'type' => 'required',
            'category' => 'required',
            'brand' => 'required',
        ]);



        /** Report Action */
        $this->report(__("messages.report_storage_exported", ['storage' => $storage->name]));

        return Excel::download(new StorageExport($storage, $request->type, $request->category, $request->brand), 'خروجی انبار ' . $storage->name . '.xlsx');
    }

    /**
     * import storage
     */
    public function import(Storage $storage, Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($storage->store_id != $store->id) {
            return redirect()->route('seller.storages.index')->withErrors(['messages.unauthorized']);
        }

        $request->validate([
            'file' => 'required',
        ]);

        /** Report Action */
        $this->report(__("messages.report_storage_imported", ['storage' => $storage->name]));

        Excel::import(new StorageImport($storage), $request->file('file'));

        return redirect()->back()->with('message', [__('messages.excel_imported_to_storage')]);
    }
}
