<?php

namespace App\Http\Controllers\Store\Storage;

use App\Models\Inventory;
use App\Http\Controllers\Controller;
use App\Models\Color;
use App\Models\Product;
use App\Notifications\ProductNotification;
use App\Models\Storage;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Morilog\Jalali\Jalalian;

class InventoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Product $product, Request $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        // main inventories
        $mainInventories = $product->inventories()->where('store_id', $store->id)
            ->orderByRaw("FIELD(status , 'awaiting', 'publish', 'deactive') ASC")
            ->orderBy('created_at', 'desc')->get()->map(fn($inventory) => [
                'id' => $inventory->id,
                'storage' => $inventory->storage->name,
                'storage_id' => $inventory->storage_id,
                'uuid' => $inventory->uuid,
                'store' => $inventory->store != null ? $inventory->store->safe : null,
                'store_id' => $inventory->store_id,
                'price' => $inventory->price,
                'get_formatted_price' => $inventory->get_formatted_price,
                'get_final_price' => $inventory->get_final_price,
                'discount_price' => $inventory->discount_price,
                'get_formatted_discount_price' => $inventory->get_formatted_discount_price,
                'jalali_discount_expire' => $inventory->jalali_discount_expire,
                'is_discount_valid' => $inventory->is_discount_valid,
                'get_image' => $inventory->get_image,
                'description' => $inventory->description,
                'get_discount_tree' => $inventory->get_discount_tree,
                'count' => $inventory->count,
                'get_formatted_count' => $inventory->get_formatted_count,
                'min_sale' => $inventory->min_sale,
                'get_formatted_min_sale' => $inventory->get_formatted_min_sale,
                'max_sale' => $inventory->max_sale,
                'get_formatted_max_sale' => $inventory->get_formatted_max_sale,
                'original' => $inventory->original,
                'used' => $inventory->used,
                'weight' => $inventory->weight,
                'send_time' => $inventory->send_time,
                'purchase_price' => $inventory->purchase_price,
                'status' => $inventory->status,

                'props' => $inventory->props,
            ]);

        // stores inventories
        $storesInventories = $product->inventories()->where(function ($query) use ($store) {
            return $query
                ->where('store_id', '!=', $store->id)
                ->orWhere('store_id', null);
        })->orderBy('created_at', 'desc')->get()->map(fn($inventory) => [
            'id' => $inventory->id,
            'storage' => $inventory->storage->name,
            'storage_id' => $inventory->storage_id,
            'uuid' => $inventory->uuid,
            'store' => $inventory->store != null ? $inventory->store->safe : null,
            'store_id' => $inventory->store_id,
            'price' => $inventory->price,
            'get_formatted_price' => $inventory->get_formatted_price,
            'get_final_price' => $inventory->get_final_price,
            'discount_price' => $inventory->discount_price,
            'get_formatted_discount_price' => $inventory->get_formatted_discount_price,
            'jalali_discount_expire' => $inventory->jalali_discount_expire,
            'is_discount_valid' => $inventory->is_discount_valid,
            'get_image' => $inventory->get_image,
            'get_discount_tree' => $inventory->get_discount_tree,
            'count' => $inventory->count,
            'get_formatted_count' => $inventory->get_formatted_count,
            'min_sale' => $inventory->min_sale,
            'get_formatted_min_sale' => $inventory->get_formatted_min_sale,
            'max_sale' => $inventory->max_sale,
            'get_formatted_max_sale' => $inventory->get_formatted_max_sale,
            'original' => $inventory->original,
            'used' => $inventory->used,
            'weight' => $inventory->weight,
            'send_time' => $inventory->send_time,
            'status' => $inventory->status,

            'props' => $inventory->props,
        ]);

        $inventories = [
            'main' => $mainInventories,
            'stores' => $storesInventories,
        ];

        // product
        $product = [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'title' => $product->title,
            'slug' => $product->slug,
            'get_images' => $product->get_images,
            'is_vip' => $product->is_vip,
            'price_model' => $product->price_model,
            'commission' => $product->commission,
        ];

        return Inertia::render('Store/Storage/Inventory/List', [
            'product' => $product,
            'inventories' => $inventories,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        $request->validate([
            'uuid' => 'required'
        ]);

        // product
        $product = Product::where('uuid', $request->uuid)->first();

        if ($product == null) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.product_not_found')]);
        }

        // validate single price
        if (count($product->price_model) == 0) {
            if ($product->inventories()->where('store_id', $store->id)->get()->count() > 0) {
                return redirect()->back()->withErrors([__('messages.just_can_add_one_price')]);
            }
        }

        // storage list
        $storages = Storage::where('store_id', $store->id)->get()->map(fn($storage) => [
            'id' => $storage->id,
            'name' => $storage->name,
        ]);

        // current storage
        $storage = Storage::where('store_id', $store->id)->where('uuid', $request->stg)->first();

        // product
        $product = [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'title' => $product->title,
            'slug' => $product->slug,
            'get_images' => $product->get_images,
            'is_vip' => $product->is_vip,
            'price_model' => $product->price_model,
        ];

        // color list
        $colors = Color::all();

        return Inertia::render('Store/Storage/Inventory/Create', [
            'product' => $product,
            'storages' => $storages,
            'storage' => $storage != null ? [
                'id' => $storage->id,
                'name' => $storage->name,
            ] : null,
            'storage_uuid' => $request->stg,
            'colors' => $colors,
            'originalList' => [
                ["id" => true, "label" => __('messages.sentence.original_product')],
                ["id" => false, "label" => __('messages.sentence.unoriginal_product')],
            ],
            'usedList' => [
                ["id" => false, "label" => __('messages.sentence.unused_product')],
                ["id" => true, "label" => __('messages.sentence.used_product')],
            ],
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        $request->validate([
            'uuid' => 'required',
            'storage' => 'required'
        ]);

        /** check payment meta */
        /*if ($store->paymentMeta == null) {
            return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);
        } else {
            if (count($store->paymentMeta->get_post_ways) == 0 || $store->paymentMeta->post_period == null || $store->paymentMeta->free_post_min_price == null) {
                return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);
            }
        }*/

        // product
        $product = Product::where('uuid', $request->uuid)->first();

        if ($product == null) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.product_not_found')]);
        }

        $request->validate([
            'price' => 'required',
            'count' => 'required',
        ]);

        // init fields
        $props = [];
        $price = (int)str_replace(',', '', $request->price);
        $purchasePrice = $request->has('purchase_price') ? (int)str_replace(',', '', $request->purchase_price) : null;
        $discount_price = null;
        $discount_expire = null;
        $count = (int)str_replace(',', '', $request->count);
        $minSale = $request->exists('min_sale') && $request->min_sale != null && $request->min_sale > 0 ? (int)str_replace(',', '', $request->min_sale) : 1;
        $maxSale = $request->exists('max_sale') && $request->max_sale != null && $request->max_sale != 0 ? (int)str_replace(',', '', $request->max_sale) : null;
        $discountTree = [];

        // validate single price
        if (count($product->price_model) == 0) {
            if ($product->inventories()->where('store_id', $store->id)->get()->count() > 0) {
                return redirect()->back()->withErrors([__('messages.just_can_add_one_price')]);
            }
        }

        // validate props
        if (count($product->price_model) > 0) {
            $request->validate([
                'props' => ['required', 'array'],
            ]);

            $props = $request->props;
        }
        foreach ($product->price_model as $key => $pm) {
            if (array_key_exists($key, $props)) {
                if ($props[$key]['type'] == "warranty" && (!array_key_exists('value', $props[$key]) || $props[$key]['value'] == null || $props[$key]['value'] == '')) {
                    $props[$key]['value'] = __('messages.default_warranty');
                }
                if (array_key_exists('value', $props[$key])) {
                    if ($props[$key]['value'] == null || $props[$key]['value'] == '') {
                        return redirect()->back()->withErrors([__('messages.product_props_required')]);
                    }
                } else {
                    return redirect()->back()->withErrors([__('messages.product_props_required')]);
                }
            } else {
                return redirect()->back()->withErrors([__('messages.product_props_required')]);
            }
        }


        // validate duplicated props
        foreach ($product->inventories()->where('store_id', $store->id)->get() as $invetory) {
            // prev prop
            $prevProp = [];
            $inProps = $invetory->inventoryProps->map(fn($inp) => [
                $inp->name => $inp->type == 'color' ? unserialize($inp->value)['color'] : $inp->value,
            ])->toArray();
            foreach ($inProps as $value) {
                foreach ($value as $k => $v) {
                    $prevProp[$k] = $v;
                }
            }

            // current prop
            $currentProp = [];
            foreach ($props as $p) {
                $currentProp[$p['name']] = $p['type'] == 'color' ? $p['value']['color'] : $p['value'];
            }

            if (count($prevProp) > 0) {
                if (count(array_diff($prevProp, $currentProp)) == 0) {
                    return redirect()->back()->withErrors([__('messages.price_is_duplicated')]);
                }
            }
        }

        // validate discount fields
        if ($request->discount_price != null && $request->discount_price != 0) {
            // compare discount_price and main price
            $discount_price = (int)str_replace(',', '', $request->discount_price);
            if ($discount_price > $price) {
                return redirect()->back()->withErrors([__('messages.discount_price_larger_than_price')]);
            }

            // check discount_expire required field
            if ($request->discount_expire != null) {
                $discount_expire = Jalalian::fromFormat('Y/m/d', $request->discount_expire)->getTimestamp();
                $discount_expire = Jalalian::forge($discount_expire)->toCarbon()->setTimezone('Asia/Tehran');
                $discount_expire = $discount_expire->format('Y-m-d H:i:s');
            }
        }

        // validate discount tree
        if ($request->exists('discount_tree') && is_array($request->discount_tree) && count($request->discount_tree) > 0) {
            if ($request->discount_price == null) {
                return redirect()->back()->withErrors([__('messages.to_add_discount_tree_first_fill_main_discount')]);
            }

            foreach ($request->discount_tree as $dt) {
                // check for null
                if ($dt['count'] != null && $dt['count'] != '' && $dt['price'] != null && $dt['price'] != '') {
                    // discount_tree fields
                    $dtCount = (int)str_replace(',', '', $dt['count']);
                    $dtPrice = (int)str_replace(',', '', $dt['price']);
                    // compare discount_tree price and main price
                    if ($dtPrice > $price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_price')]);
                    }
                    if ($discount_price != null && $dtPrice > $discount_price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_discount_price')]);
                    }

                    $discountTree[] = [
                        'count' => $dtCount,
                        'price' => $dtPrice,
                    ];
                }
            }

            // sort discount tree
            sort($discountTree);

            // validate discount tree counts
            $discountTreeCounts = [];
            foreach ($discountTree as $dt) {
                $discountTreeCounts[$dt['count']][] = $dt['count'];
            }
            if (count($discountTreeCounts) != count($discountTree)) {
                return redirect()->back()->withErrors([__('messages.discount_tree_count_wrong')]);
            }

            // validate discount tree prices
            $dtPrevPrice = null;
            foreach ($discountTree as $dt) {
                if ($dtPrevPrice != null) {
                    if ($dtPrevPrice <= $dt['price']) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_wrong')]);
                    }
                }
                $dtPrevPrice = $dt['price'];
            }
        }

        // check min and max sale value
        if ($maxSale != null && $maxSale != '' && $minSale > $maxSale) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_max_sale')]);
        }
        if ($minSale != null && $count != 0 && $minSale > $count) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_count')]);
        }

        // price changes
        $priceChanges[Carbon::now()->timestamp] = $discount_price != null ? $discount_price : $price;

        // validate weight
        $weight = null;
        if ($request->exists('weight') && $request->weight != null && is_numeric(str_replace(',', '', $request->weight))) {
            $weight = str_replace(',', '', $request->weight);
        }

        $inventory = $product->inventories()->create([
            'storage_id' => $request->storage,
            'store_id' => $store->id,
            'uuid' => 'inv-' . rand(1000000, 99999999),
            'price' => $price,
            'discount_price' => $discount_price,
            'discount_expire' => $discount_expire,
            'count' => $count,
            'min_sale' => $minSale,
            'max_sale' => $maxSale,
            'original' => $request->original,
            'used' => $request->used,
            'weight' => $weight,
            'purchase_price' => $purchasePrice,
            'discount_tree' => is_array($discountTree) && count($discountTree) > 0 ? serialize($discountTree) : null,
            'price_changes' => $priceChanges != null ? serialize($priceChanges) : null,
            'image' => $request->exists('image') ? $request->image : null,
            'description' => $request->exists('description') ? $request->description : null,
            'send_time' => $request->exists('send_time') ? $request->send_time : null,
            'status' => 'awaiting',
        ]);

        $pmID = 0;
        foreach ($product->price_model as $key => $pm) {
            if (array_key_exists($key, $props)) {
                if (array_key_exists('value', $props[$key])) {
                    if ($props[$key]['value'] == null || $props[$key]['value'] == '') {
                        return redirect()->back()->withErrors([__('messages.product_props_required')]);
                    }

                    $pmModel = $inventory->inventoryProps()->create([
                        'child' => $pmID,
                        'name' => $pm['name'],
                        'type' => $pm['type'],
                        'value' => $pm['type'] == 'color' ? serialize($props[$key]['value']) : $props[$key]['value'],
                    ]);
                    $pmID = $pmModel->id;
                } else {
                    return redirect()->back()->withErrors([__('messages.product_props_required')]);
                }
            } else {
                return redirect()->back()->withErrors([__('messages.product_props_required')]);
            }
        }

        /** product notifications */
        // discount notification
        if ($product->inventory) {
            $productInventoryNotifications = $product->productNotifications()->where('type', 'inventory')->get();
            foreach ($productInventoryNotifications as $pn) {
                // notification
                $pn->user->notify(new ProductNotification($product, 'inventory'), true);
                $product->productNotifications()->where('user_id', $pn->user_id)->where('type', 'inventory')->delete();
            }
        }

        // discount notification
        if ($product->has_discount) {
            $productDiscountNotifications = $product->productNotifications()->where('type', 'discount')->get();
            foreach ($productDiscountNotifications as $pn) {
                // notification
                $pn->user->notify(new ProductNotification($product, 'discount'), true);
                $product->productNotifications()->where('user_id', $pn->user_id)->where('type', 'discount')->delete();
            }
        }
        /**************************/

        /** Report Action */
        $reportMessage = __("messages.report_inventory_added");
        $this->report($reportMessage, 'product', $product);
        /******************/

        if ($request->exists('stg') && $request->stg != null && $request->stg != '') {
            return redirect()->route('seller.storages.storageList', ['storage' => $request->stg, 'product' => $product->slug])->with('message', [__('messages.inventory_added')]);
        }
        return redirect()->route('seller.inventories.index', $product->slug)->with('message', [__('messages.inventory_added')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Inventory $inventory)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Inventory $inventory, Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($inventory->store_id != $store->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        $request->validate([
            'uuid' => 'required'
        ]);

        // product
        $product = Product::where('uuid', $request->uuid)->first();

        if ($product == null) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.product_not_found')]);
        }

        // inventory props
        $priceModel = [];
        $inventoryProps = $inventory->inventoryProps()->get();
        foreach ($inventoryProps as $key => $inventoryProp) {
            $priceModel[$inventoryProp['id']] = [
                "child" => $inventoryProp['child'],
                "name" => $inventoryProp['name'],
                "type" => $inventoryProp['type'],
                "value" => $inventoryProp['type'] == 'color' ? unserialize($inventoryProp['value']) : $inventoryProp['value'],
            ];
        }

        // storage list
        $storages = Storage::where('store_id', $store->id)->get()->map(fn($storage) => [
            'id' => $storage->id,
            'name' => $storage->name,
        ]);

        // product
        $product = [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'title' => $product->title,
            'slug' => $product->slug,
            'get_images' => $product->get_images,
            'is_vip' => $product->is_vip,
            'price_model' => $product->price_model,
        ];

        // color list
        $colors = Color::all();

        return Inertia::render('Store/Storage/Inventory/Edit', [
            'inventory' => $inventory,
            'product' => $product,
            'priceModel' => $priceModel,
            'storages' => $storages,
            'colors' => $colors,
            'originalList' => [
                ["id" => true, "label" => __('messages.sentence.original_product')],
                ["id" => false, "label" => __('messages.sentence.unoriginal_product')],
            ],
            'usedList' => [
                ["id" => false, "label" => __('messages.sentence.unused_product')],
                ["id" => true, "label" => __('messages.sentence.used_product')],
            ],
            'storage_uuid' => $request->stg,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Inventory $inventory)
    {
        if ($inventory->status != 'publish' && $inventory->status != 'awaiting') {
            return redirect()->back();
        }

        /** @var \App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($inventory->store_id != $store->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        $request->validate([
            'uuid' => 'required',
            'storage' => 'required'
        ]);

        /** check payment meta */
        /*if ($store->paymentMeta == null) {
            return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);
        } else {
            if (count($store->paymentMeta->get_post_ways) == 0 || $store->paymentMeta->post_period == null || $store->paymentMeta->free_post_min_price == null) {
                return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);
            }
        }*/

        // product
        $product = Product::where('uuid', $request->uuid)->first();

        if ($product == null) {
            return redirect()->route('admin.products.index')->withErrors([__('messages.product_not_found')]);
        }

        $request->validate([
            'price' => 'required',
            'count' => 'required',
        ]);

        // init fields
        $props = [];
        $price = (int)str_replace(',', '', $request->price);
        $purchasePrice = $request->has('purchase_price') ? (int)str_replace(',', '', $request->purchase_price) : null;
        $discount_price = null;
        $discount_expire = null;
        $count = (int)str_replace(',', '', $request->count);
        $minSale = $request->exists('min_sale') && $request->min_sale != null && $request->min_sale > 0 ? (int)str_replace(',', '', $request->min_sale) : 1;
        $maxSale = $request->exists('max_sale') && $request->max_sale != null && $request->max_sale != 0 ? (int)str_replace(',', '', $request->max_sale) : null;
        $discountTree = [];

        // validate props
        if (count($product->price_model) > 0) {
            $request->validate([
                'props' => ['required', 'array'],
            ]);

            $props = $request->props;
        }
        $inventoryProps = $inventory->inventoryProps()->get();
        foreach ($inventoryProps as $inventoryProp) {
            if (array_key_exists($inventoryProp->id, $props)) {
                if ($props[$inventoryProp->id]['type'] == "warranty" && (!array_key_exists('value', $props[$inventoryProp->id]) || $props[$inventoryProp->id]['value'] == null || $props[$inventoryProp->id]['value'] == '')) {
                    $props[$inventoryProp->id]['value'] = __('messages.default_warranty');
                }
                if (array_key_exists('value', $props[$inventoryProp->id])) {
                    if ($props[$inventoryProp->id]['value'] == null || $props[$inventoryProp->id]['value'] == '') {
                        return redirect()->back()->withErrors([__('messages.product_props_required')]);
                    }
                } else {
                    return redirect()->back()->withErrors([__('messages.product_props_required')]);
                }
            } else {
                return redirect()->back()->withErrors([__('messages.product_props_required')]);
            }
        }

        // validate duplicated props
        foreach ($product->inventories()->where('store_id', $store->id)->get() as $inv) {
            if ($inventory->id == $inv->id) {
                continue;
            }

            // prev prop
            $prevProp = [];
            $inProps = $inv->inventoryProps->map(fn($inp) => [
                $inp->name => $inp->type == 'color' ? unserialize($inp->value)['color'] : $inp->value,
            ])->toArray();
            foreach ($inProps as $value) {
                foreach ($value as $k => $v) {
                    $prevProp[$k] = $v;
                }
            }

            // current prop
            $currentProp = [];
            foreach ($props as $p) {
                $currentProp[$p['name']] = $p['type'] == 'color' ? $p['value']['color'] : $p['value'];
            }

            if (count(array_diff($prevProp, $currentProp)) == 0) {
                return redirect()->back()->withErrors([__('messages.price_is_duplicated')]);
            }
        }

        // validate discount fields
        if ($request->discount_price != null && $request->discount_price != 0) {
            // compare discount_price and main price
            $discount_price = (int)str_replace(',', '', $request->discount_price);
            if ($discount_price > $price) {
                return redirect()->back()->withErrors([__('messages.discount_price_larger_than_price')]);
            }

            // check discount_expire required field
            if ($request->discount_expire != null) {
                $discount_expire = Jalalian::fromFormat('Y/m/d', $request->discount_expire)->getTimestamp();
                $discount_expire = Jalalian::forge($discount_expire)->toCarbon()->setTimezone('Asia/Tehran');
                $discount_expire = $discount_expire->format('Y-m-d H:i:s');
            }
        }

        // validate discount tree
        if ($request->exists('discount_tree') && is_array($request->discount_tree) && count($request->discount_tree) > 0) {
            if ($request->discount_price == null) {
                return redirect()->back()->withErrors([__('messages.to_add_discount_tree_first_fill_main_discount')]);
            }

            foreach ($request->discount_tree as $dt) {
                // check for null
                if ($dt['count'] != null && $dt['count'] != '' && $dt['price'] != null && $dt['price'] != '') {
                    // discount_tree fields
                    $dtCount = (int)str_replace(',', '', $dt['count']);
                    $dtPrice = (int)str_replace(',', '', $dt['price']);
                    // compare discount_tree price and main price
                    if ($dtPrice > $price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_price')]);
                    }
                    if ($discount_price != null && $dtPrice > $discount_price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_discount_price')]);
                    }

                    $discountTree[] = [
                        'count' => $dtCount,
                        'price' => $dtPrice,
                    ];
                }
            }

            // sort discount tree
            sort($discountTree);

            // validate discount tree counts
            $discountTreeCounts = [];
            foreach ($discountTree as $dt) {
                $discountTreeCounts[$dt['count']][] = $dt['count'];
            }
            if (count($discountTreeCounts) != count($discountTree)) {
                return redirect()->back()->withErrors([__('messages.discount_tree_count_wrong')]);
            }

            // validate discount tree prices
            $dtPrevPrice = null;
            foreach ($discountTree as $dt) {
                if ($dtPrevPrice != null) {
                    if ($dtPrevPrice <= $dt['price']) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_wrong')]);
                    }
                }
                $dtPrevPrice = $dt['price'];
            }
        }

        // check min and max sale value
        if ($maxSale != null && $maxSale != '' && $minSale > $maxSale) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_max_sale')]);
        }
        if ($minSale != null && $count != 0 && $minSale > $count) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_count')]);
        }

        // handle price changes
        $priceChanges = $inventory->get_price_changes;
        if (is_array($priceChanges)) {
            foreach ($priceChanges as $key => $value) {
                // delete all current day prices
                if ((Carbon::now()->timestamp - $key) < 86400) {
                    unset($priceChanges[$key]);
                    continue;
                }

                // delete all old price than 1month ago
                if ((Carbon::now()->timestamp - $key) > 2592000) {
                    unset($priceChanges[$key]);
                }
            }
        }
        // add to price changes
        $priceChanges[Carbon::now()->timestamp] = $discount_price != null ? $discount_price : $price;

        // validate weight
        $weight = $inventory->weight;
        if ($request->exists('weight') && $request->weight != null && is_numeric(str_replace(',', '', $request->weight))) {
            $weight = str_replace(',', '', $request->weight);
        }

        $inventory->update([
            'storage_id' => $request->storage,
            'store_id' => $store->id,
            'price' => $price,
            'discount_price' => $discount_price,
            'discount_expire' => $discount_expire,
            'count' => $count,
            'min_sale' => $minSale,
            'max_sale' => $maxSale,
            'original' => $request->original,
            'used' => $request->used,
            'weight' => $weight,
            'purchase_price' => $purchasePrice,
            'discount_tree' => is_array($discountTree) && count($discountTree) > 0 ? serialize($discountTree) : null,
            'image' => $request->exists('image') && $request->image != null ? $request->image : $inventory->image,
            'description' => $request->exists('description') && $request->description != null ? $request->description : $inventory->description,
            'send_time' => $request->exists('send_time') ? $request->send_time : $inventory->send_time,
            'price_changes' => serialize($priceChanges),
        ]);

        $inventoryProps = $inventory->inventoryProps()->get();
        foreach ($inventoryProps as $inventoryProp) {
            if (array_key_exists($inventoryProp->id, $props)) {
                if (array_key_exists('value', $props[$inventoryProp->id])) {
                    if ($props[$inventoryProp->id]['value'] == null || $props[$inventoryProp->id]['value'] == '') {
                        return redirect()->back()->withErrors([__('messages.product_props_required')]);
                    }

                    $inventoryProp->update([
                        'value' => $inventoryProp->type == 'color' ? serialize($props[$inventoryProp->id]['value']) : $props[$inventoryProp->id]['value'],
                    ]);
                } else {
                    return redirect()->back()->withErrors([__('messages.product_props_required')]);
                }
            } else {
                return redirect()->back()->withErrors([__('messages.product_props_required')]);
            }
        }

        /** product notifications */
        // discount notification
        if ($product->inventory) {
            $productInventoryNotifications = $product->productNotifications()->where('type', 'inventory')->get();
            foreach ($productInventoryNotifications as $pn) {
                // notification
                $pn->user->notify(new ProductNotification($product, 'inventory'), true);
                $product->productNotifications()->where('user_id', $pn->user_id)->where('type', 'inventory')->delete();
            }
        }

        // discount notification
        if ($product->has_discount) {
            $productDiscountNotifications = $product->productNotifications()->where('type', 'discount')->get();
            foreach ($productDiscountNotifications as $pn) {
                // notification
                $pn->user->notify(new ProductNotification($product, 'discount'), true);
                $product->productNotifications()->where('user_id', $pn->user_id)->where('type', 'discount')->delete();
            }
        }
        /**************************/

        /** Report Action */
        $reportMessage = __("messages.report_inventory_updated");
        $this->report($reportMessage, 'product', $product);
        /******************/

        if ($request->exists('stg') && $request->stg != null && $request->stg != '') {
            return redirect()->route('seller.storages.storageList', ['storage' => $request->stg, 'product' => $product->slug])->with('message', [__('messages.inventory_updated')]);
        }
        return redirect()->route('seller.inventories.index', $product->slug)->with('message', [__('messages.inventory_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Inventory $inventory)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($inventory->store_id != $store->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        $product = $inventory->product;
        $inventory->delete();

        /** Report Action */
        $reportMessage = __("messages.report_inventory_deleted");
        $this->report($reportMessage, 'product', $product);
        /******************/

        return redirect()->back()->with('message', [__('messages.inventory_deleted')]);
    }

    /**
     * decrease inventory
     */
    public function decrease(Inventory $inventory)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($inventory->store_id != $store->id) {
            return redirect()->route('seller.products.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($inventory->store_id == $store->id) {
            if ($inventory->count == 0) {
                return redirect()->back()->withErrors([__('messages.inventory_count_is_zero')]);
            }

            $inventory->update([
                'count' => $inventory->count - 1
            ]);
            return redirect()->back()->with('message', [__('messages.inventory_count_decrease')]);
        }

        return redirect()->back()->withErrors([__('messages.unauthorized')]);
    }
}
