<?php

namespace App\Http\Controllers\Store\Orders;

use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Models\Consignment;
use App\Models\ConsignmentItem;
use App\Models\Inventory;
use App\Models\Setting;
use App\Models\Store;
use App\Models\Transaction;
use App\Models\User;
use App\Notifications\OrderCancellation;
use App\Notifications\OrderSent;
use App\Notifications\SellerDirectSendOrderStatus;
use App\Notifications\SellerMainStorageSendOrderStatus;
use App\Services\Cart\CartService;
use App\Services\UPG\DigipayService;
use App\Services\UPG\SnapppayService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class OrdersController extends Controller
{
    use UUIDHelper;

    // properties
    protected SnapppayService $snapppayService;
    protected DigipayService $digipayService;

    public function __construct(SnapppayService $snapppayService, DigipayService $digipayService)
    {
        $this->snapppayService = $snapppayService;
        $this->digipayService = $digipayService;
    }

    /**
     * show order details
     */
    public function show(Consignment $consignment)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        $consignment = $store->consignments()->where('id', $consignment->id)->first();

        if ($consignment == null) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        if ($consignment->send_type != 'direct') {
            if ($consignment->address == null) {
                return redirect()->back()->withErrors([__('messages.unauthorized')]);
            }
        }

        if ($consignment->consignmentItems()->count() == 0) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        if ($consignment->consignmentItems()->where('status', 'awaiting_request_from_seller')->count() > 0) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        // order payment transaction
        $paymentTransaction = Transaction::where('transaction_id', $consignment->order->orderGroup->payment_transaction_uuid)->first();
        $returnResponse = [];
        if ($paymentTransaction) {
            $returnResponse = json_decode($paymentTransaction->return_response, true);
            $returnResponse = is_array($returnResponse) ? $returnResponse : [];
        }

        // check snapppay
        $isSnapppay = false;
        if ($paymentTransaction && $paymentTransaction->payment_gateway == 'snapppay') {
            $isSnapppay = true;
        }

        // check digipay upg
        $isDigipayUpg = false;
        if ($paymentTransaction && $paymentTransaction->payment_gateway == 'digipay' && array_key_exists('type', $returnResponse) && ((int)$returnResponse['type'] == 5 || (int)$returnResponse['type'] == 13)) {
            $isDigipayUpg = true;
        }

        // check azki
        $isAzki = false;
        if ($paymentTransaction && $paymentTransaction->payment_gateway == 'azki') {
            $isAzki = true;
        }

        return Inertia::render('Store/Orders/Show', [
            'consignment' => [
                'id' => $consignment->id,
                'uuid' => $consignment->uuid,
                'send_type' => $consignment->send_type,
                'send_method' => $consignment->send_type == 'direct' ? $consignment->order->get_send_method : null,
                'send_price' => $consignment->order->status != 'cancelled' && $consignment->order->status != 'returned' && $consignment->send_type == 'direct' ? $consignment->order->get_send_method['price'] : 0,
                'get_address' => $consignment->send_type != 'direct' ? $consignment->get_address : $consignment->order->get_address,
                'get_total_bill' => $consignment->get_total_bill,
                'get_total_cancel_bill' => $consignment->get_total_cancel_bill,
                'get_shares' => $consignment->get_shares,
                'is_all_product_sent' => $consignment->is_all_product_sent,
                'is_products_awaiting_receipt_from_seller' => $consignment->is_products_awaiting_receipt_from_seller,
                'tracking_code' => $consignment->tracking_code,
                'jalali_created_at' => $consignment->jalali_created_at,
                'status' => $consignment->status,

                'order' => [
                    'id' => $consignment->order->id,
                    'status' => $consignment->order->status,
                ],

                'consignment_items' => $consignment->consignmentItems->map(fn($item) => [
                    'id' => $item->id,
                    'get_props' => $item->get_props,
                    'get_full_count_price' => $item->get_full_count_price,
                    'count' => $item->count,
                    'price' => $item->price,
                    'discount' => $item->discount,
                    'get_shares' => $item->get_shares,
                    'reason' => $item->reason,
                    'status' => $item->status,

                    'storage' => [
                        'id' => $item->storage->id,
                        'name' => $item->storage->name,
                    ],
                    'product' => [
                        'id' => $item->product->id,
                        'title' => $item->product->title,
                        'slug' => $item->product->slug,
                        'get_images' => $item->product->get_images,
                        'commission' => $item->product->commission,
                    ],
                    'store' => $item->store != null ? [
                        'id' => $item->store->id,
                        'uuid' => $item->store->uuid,
                        'name' => $item->store->name,
                    ] : null,
                ]),
                'store' => $consignment->store != null ? [
                    'id' => $consignment->id,
                    'uuid' => $consignment->uuid,
                    'name' => $consignment->name,
                ] : null,
            ],
            'payment_transaction' => $paymentTransaction,
            'is_pay_with_wallet' => $paymentTransaction ? false : true,
            'is_snapppay' => $isSnapppay,
            'is_digipay_upg' => $isDigipayUpg,
            'is_azki' => $isAzki,
        ]);
    }

    /**
     * awaiting orders
     */
    public function awaiting(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** @var LengthAwarePaginator */
        $consignments = $store->consignments()->when($request->input('search'), function ($query, $search) {
            $query->where('uuid', 'like', "%{$search}%");
        })->whereHas('consignmentItems', function ($query) {
            $query->where(function ($query) {
                $query->where('status', 'awaiting_seller')->orWhere('status', 'awaiting_seller_send');
            });
        })->with('store')->paginate(10);

        $consignments = $consignments->through(fn($consignment) => [
            'id' => $consignment->id,
            'uuid' => $consignment->uuid,
            'order_uuid' => $consignment->order->uuid,
            'send_type' => $consignment->send_type,
            'status' => $consignment->status,
            'jalali_created_at' => $consignment->jalali_created_at,
            'jalali_updated_at' => $consignment->jalali_updated_at,
        ]);

        $consignments->withQueryString();

        return Inertia::render('Store/Orders/Awaiting', [
            'consignments' => $consignments,
        ]);
    }

    /**
     * sent orders
     */
    public function sent(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** @var LengthAwarePaginator */
        $consignments = $store->consignments()->where(function ($query) {
            $query->where('status', 'sent_to_main_storage')->orWhere('status', 'direct_sent_to_customer');
        })->when($request->input('search'), function ($query, $search) {
            $query->where('uuid', 'like', "%{$search}%");
        })->orderBy('updated_at', 'desc')->with('store')->paginate(10);

        $consignments = $consignments->through(fn($consignment) => [
            'id' => $consignment->id,
            'uuid' => $consignment->uuid,
            'order_uuid' => $consignment->order->uuid,
            'send_type' => $consignment->send_type,
            'status' => $consignment->status,
            'jalali_created_at' => $consignment->jalali_created_at,
            'jalali_updated_at' => $consignment->jalali_updated_at,
        ]);

        $consignments->withQueryString();

        return Inertia::render('Store/Orders/Sent', [
            'consignments' => $consignments,
        ]);
    }

    /**
     * sent orders
     */
    public function delivered(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** @var LengthAwarePaginator */
        $consignments = $store->consignments()->where(function ($query) {
            $query->where('status', 'delivered_to_main_storage')->orWhere('status', 'sent_to_customer')->orWhere('status', 'delivered');
        })->when($request->input('search'), function ($query, $search) {
            $query->where('uuid', 'like', "%{$search}%");
        })->orderBy('updated_at', 'desc')->with('store')->paginate(10);

        $consignments = $consignments->through(fn($consignment) => [
            'id' => $consignment->id,
            'uuid' => $consignment->uuid,
            'order_uuid' => $consignment->order->uuid,
            'send_type' => $consignment->send_type,
            'status' => $consignment->status,
            'jalali_created_at' => $consignment->jalali_created_at,
            'jalali_updated_at' => $consignment->jalali_updated_at,
        ]);

        $consignments->withQueryString();

        return Inertia::render('Store/Orders/Delivered', [
            'consignments' => $consignments,
        ]);
    }

    /**
     * awaiting orders
     */
    public function cancelled(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        /** @var LengthAwarePaginator */
        $consignments = $store->consignments()->where(function ($query) {
            $query->where('status', 'cancelled')->orWhere('status', 'returned');
        })->when($request->input('search'), function ($query, $search) {
            $query->where('uuid', 'like', "%{$search}%");
        })->orderBy('updated_at', 'desc')->with('store')->paginate(10);

        $consignments = $consignments->through(fn($consignment) => [
            'id' => $consignment->id,
            'uuid' => $consignment->uuid,
            'order_uuid' => $consignment->order->uuid,
            'send_type' => $consignment->send_type,
            'status' => $consignment->status,
            'jalali_created_at' => $consignment->jalali_created_at,
            'jalali_updated_at' => $consignment->jalali_updated_at,
        ]);

        $consignments->withQueryString();

        return Inertia::render('Store/Orders/Cancelled', [
            'consignments' => $consignments,
        ]);
    }

    /**
     * cancel consignment item
     */
    public function cancel_item(Request $request, ConsignmentItem $consignmentItem, CartService $cartService)
    {
        if (!$request->has('count')) {
            $request->validate([
                'reason' => 'required',
            ]);
        }

        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        // check consignment item status
        if (in_array($consignmentItem->status, ['delivered', 'sent', 'cancelled', 'returned'])) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        if ($store->direct_post && !$store->transportation_status) {
            return redirect()->route('seller.transportation.index')->withErrors([__('messages.post_ways_required')]);
        }

        // check seller consignment item
        if ($consignmentItem->store_id != $store->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        // check consignment item status
        if ($consignmentItem->consignment->send_type != 'direct') {
            if ($consignmentItem->status != 'awaiting_seller') {
                return redirect()->back()->withErrors([__('messages.unauthorized')]);
            }
        }

        // order payment transaction
        $transactionId = $consignmentItem->consignment?->order?->orderGroup?->payment_transaction_uuid;
        $paymentTransaction = $transactionId ? Transaction::where('transaction_id', $transactionId)->first() : null;

        // payment gateway return response
        $returnResponse = [];
        if ($paymentTransaction) {
            $returnResponse = json_decode($paymentTransaction->return_response, true);
            $returnResponse = is_array($returnResponse) ? $returnResponse : [];
        }

        // check snapppay
        $isSnapppay = false;
        if ($paymentTransaction && $paymentTransaction->payment_gateway == 'snapppay') {
            $isSnapppay = true;
        }

        // check digipay upg
        $isDigipayUpg = false;
        if ($paymentTransaction && $paymentTransaction->payment_gateway == 'digipay' && array_key_exists('type', $returnResponse) && ((int)$returnResponse['type'] == 5 || (int)$returnResponse['type'] == 13)) {
            $isDigipayUpg = true;
        }

        // check azki
        $isAzki = false;
        if ($paymentTransaction && $paymentTransaction->payment_gateway == 'azki') {
            $isAzki = true;
        }

        // check for UPG gateways
        if ($isDigipayUpg) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        try {
            DB::transaction(function () use ($consignmentItem, $request, $user, $cartService, $paymentTransaction, $isSnapppay) {
                /** count or all cancel */
                if ($isSnapppay && $request->has('count') && ctype_digit((string) $request->count)) {
                    if ((int)$request->count <= 0) {
                        throw new \Exception(__('messages.count_cant_zero'));
                    }

                    if ((int)$request->count < $consignmentItem->count) {
                        // lock inventory row
                        $inventory = Inventory::where('id', $consignmentItem->inventory_id)->lockForUpdate()->first();
                        if ($inventory != null) {
                            $inventory->update([
                                'count' => $inventory->count + (int)$request->count,
                            ]);
                        }

                        // cancelling item
                        $consignmentItem->update([
                            'count' => $consignmentItem->count - (int)$request->count,
                        ]);
                    } else {
                        throw new \Exception(__('messages.consignment_item_count_for_cancel_is_more_than_real_count'));
                    }
                } else {
                    // lock inventory row
                    $inventory = Inventory::where('id', $consignmentItem->inventory_id)->lockForUpdate()->first();
                    if ($inventory != null) {
                        $inventory->update([
                            'count' => $inventory->count + $consignmentItem->count,
                        ]);
                    }

                    // cancelling item
                    $consignmentItem->update([
                        'reason' => $request->reason,
                        'status' => 'cancelled'
                    ]);
                }
                /** end */

                /** check for all item cancelled or not */
                $consignment = $consignmentItem->consignment;
                $order = $consignment->order;
                $cancelledItemsCount = $consignment->consignmentItems()->where('status', 'cancelled')->count();
                $totalItemsCount = $consignment->consignmentItems()->count();
                if ($cancelledItemsCount === $totalItemsCount) {
                    $consignment->update(['status' => 'cancelled']);

                    // if it's a direct send, cancel the whole order
                    if ($consignment->send_type == 'direct') {
                        // get customer
                        $customer = $consignment->order->user;

                        // get customer paid price
                        $getDiscountCodeAmount = $consignment->order->get_discount != null ? $consignment->order->get_discount['amount'] : 0;
                        $paidPrice = $consignment->order->get_total_bill['price'] + $consignment->order->get_total_bill['send_price'] - $getDiscountCodeAmount;

                        // refund to customer wallet
                        $this->refundToWallet($customer, $paidPrice, $consignment->order);

                        // cancel order
                        $order->update(['status' => 'cancelled']);
                    }

                    /** Report Action */
                    $this->report(
                        __("messages.report_consignment_cancelled", [
                            'consignment_uuid' => $consignment->uuid,
                            'order_uuid' => $order->uuid
                        ]),
                        'user',
                        $user
                    );
                } else {
                    /** Report Action */
                    $this->report(
                        __("messages.report_consignment_item_cancelled", [
                            'consignment_uuid' => $consignment->uuid,
                            'order_uuid' => $order->uuid
                        ]),
                        'user',
                        $user
                    );
                }
                /** end */

                /** update discount coupon */
                if ($order->discount != null) {
                    // discount
                    $discount = $order->discount ? unserialize($order->discount) : null;

                    // delete old discount
                    $order->update([
                        'discount' => null
                    ]);

                    // refresh order model
                    $order->refresh();

                    // selected address
                    $selectedAddress = $order->address != null ? unserialize($order->address) : null;
                    // consignments information array
                    $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress, true);

                    // calculate discount amount
                    $amount = 0;
                    $discountPrice = ($consignmentsInformation['financial']['totalFinalPrice'] * $discount['information']['percent']) / 100;
                    if ($discountPrice <= $discount['information']['max_amount']) {
                        $amount = $discountPrice;
                    } else {
                        $amount = $discount['information']['max_amount'];
                    }

                    // add discount to order
                    $discount = [
                        'information' => [
                            'id' => $discount['information']['id'],
                            'title' => $discount['information']['title'],
                            'code' => $discount['information']['code'],
                            'percent' => $discount['information']['percent'],
                            'max_amount' => $discount['information']['max_amount'],
                        ],
                        'amount' => $amount,
                    ];

                    $order->update([
                        'discount' => serialize($discount)
                    ]);
                }
                /** end */

                /** upg gateways */
                $order->refresh();
                $this->snapppayService->snapppay($order->orderGroup);
                /** end */
            });
        } catch (\Exception $e) {
            return redirect()->back()->withErrors([$e->getMessage()]);
        }

        return redirect()->back()->with('message', [__('messages.consignment_item_cancelled')]);
    }

    /**
     * change consignment status
     */
    public function change_status(Request $request, Consignment $consignment)
    {
        $request->validate([
            'status' => 'required',
        ]);

        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        // base condition
        if ($consignment == null) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }
        if ($consignment->send_type != 'direct') {
            if ($consignment->status != 'awaiting_seller') {
                return redirect()->back()->withErrors([__('messages.unauthorized')]);
            }
        }
        if ($consignment->store_id != $store->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }
        if ($consignment->consignmentItems()->count() == 0) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }
        if ($consignment->consignmentItems()->where('status', 'awaiting_request_from_seller')->count() > 0) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        // check status
        if ($consignment->status === 'direct_sent_to_customer' || $consignment->status === 'sent_to_main_storage') {
            return redirect()->back()->withErrors([__('messages.already_sent')]);
        }
        if ($consignment->status === 'cancelled') {
            return redirect()->back()->withErrors([__('messages.already_cancelled')]);
        }

        // order
        $order = $consignment->order;

        // check for UPG gateways
        $paymentTransaction = Transaction::where('transaction_id', $order->orderGroup->payment_transaction_uuid)->first();

        // get customer
        $customer = $consignment->order->user;

        try {
            DB::transaction(function () use ($customer, $request, $user, $consignment, $order, $paymentTransaction) {
                // sent
                if ($request->status == 'sent') {
                    $request->validate([
                        'tracking_code' => 'required',
                    ]);

                    // lock consignment for update
                    $consignment = Consignment::where('id', $consignment->id)->lockForUpdate()->first();

                    if ($consignment->send_type == 'direct' && $consignment->send_method != null) {
                        // direct send

                        /** update order and consignment and consignmentItems status and shares / add tracking_code */
                        $consignmentItems = $consignment->consignmentItems()->where('status', 'awaiting_seller_send')->get();
                        foreach ($consignmentItems as $item) {
                            $item->update([
                                'shop_share' => $item->get_shares['shop'],
                                'seller_share' => $item->get_shares['seller'],
                                'affiliate_share' => $item->get_shares['affiliate'],
                                'status' => 'sent',
                            ]);
                        }
                        $consignment->update([
                            'tracking_code' => $request->tracking_code,
                            'status' => 'direct_sent_to_customer',
                        ]);
                        $order->update([
                            'post_tracking_code' => $request->tracking_code,
                            'status' => 'sent',
                            'sent_at' => now()
                        ]);
                        /** end */

                        /** handle consolidated shares */
                        // deposit sellers shares
                        foreach ($order->get_consolidated_shares['seller'] as $share) {
                            $seller = Store::find($share['information']['id'])->user;

                            // lock seller for update
                            $seller = User::where('id', $seller->id)->lockForUpdate()->first();

                            // Prevent duplicate construction
                            $exists = $seller->pendingPayments()
                                ->where('order_id', $order->id)
                                ->where('type', 'seller')
                                ->where('data', serialize(['consignment_uuid' => $share['consignment']]))
                                ->exists();

                            if (! $exists) {
                                $seller->pendingPayments()->create([
                                    'order_id' => $order->id,
                                    'amount' => $share['share'] + $order->get_total_bill['send_price'],
                                    'data' => serialize([
                                        'consignment_uuid' => $share['consignment']
                                    ]),
                                    'type' => 'seller',
                                    'status' => 'awaiting',
                                ]);
                            }
                        }
                        // deposit affiliates shares
                        foreach ($order->get_consolidated_shares['affiliate'] as $share) {
                            $affiliate = User::find($share['information']['id']);
                            if ($affiliate != null) {
                                // lock affiliate for update
                                $affiliate = User::where('id', $affiliate->id)->lockForUpdate()->first();

                                $exists = $affiliate->pendingPayments()
                                    ->where('order_id', $order->id)
                                    ->where('type', 'affiliate')
                                    ->where('data', serialize(['product' => $share['product']]))
                                    ->exists();

                                if (! $exists) {
                                    $affiliate->pendingPayments()->create([
                                        'order_id' => $order->id,
                                        'amount' => $share['share'],
                                        'data' => serialize([
                                            'product' => $share['product']
                                        ]),
                                        'type' => 'affiliate',
                                        'status' => 'awaiting',
                                    ]);
                                }
                            }
                        }
                        /** end */

                        /** refund cancelled items to customer wallet */
                        if ($paymentTransaction && $paymentTransaction->payment_gateway == "snapppay") {
                            // dont refund (UPG Direct cancellation)
                        } else {
                            if ($order->get_total_bill['cancel']['price'] > 0) {
                                if ($customer->wallet) {
                                    $customer->wallet->update([
                                        'amount' => $customer->wallet->amount + $order->get_total_bill['cancel']['price'],
                                    ]);
                                }

                                $customer->transactions()->create([
                                    'amount' => $order->get_total_bill['cancel']['price'],
                                    'transaction_id' => $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id'),
                                    'description' => __('messages.transaction_user_order_returned_some_item', ['order' => $order->uuid]),
                                    'type' => 'user_order_returned',
                                    'status' => 'accepted'
                                ]);
                            }
                        }
                        /** end */

                        /** delivery digipay upg */
                        $this->digipayService->digipayDelivery($order->orderGroup, $request->tracking_code);
                        /** end */
                    } else {
                        // send to main storage
                        $consignmentItems = $consignment->consignmentItems()->where('status', 'awaiting_seller')->get();
                        foreach ($consignmentItems as $item) {
                            $item->update([
                                'status' => 'awaiting_receipt_from_seller',
                            ]);
                        }
                        $consignment->update([
                            'tracking_code' => $request->tracking_code,
                            'status' => 'sent_to_main_storage',
                        ]);
                    }

                    // register report
                    $this->report(__("messages.report_consignment_sent", ['consignment_uuid' => $consignment->uuid, 'order_uuid' => $consignment->order->uuid]), 'user', $user);
                }

                // cancelled
                if ($request->status == 'cancelled') {
                    $request->validate([
                        'reason' => 'required',
                    ]);

                    /** calculate paid price */
                    $getDiscountCodeAmount = $consignment->order->get_discount != null ? $consignment->order->get_discount['amount'] : 0;
                    $paidPrice = $consignment->order->get_total_bill['price'] + $consignment->order->get_total_bill['send_price'] - $getDiscountCodeAmount;
                    /** end */

                    /** update consignment and consignmentItems inventory count and status / add cancellation reason */
                    $consignmentItems = $consignment->consignmentItems()->get();
                    foreach ($consignmentItems as $consignmentItem) {
                        // add inventory count
                        if ($consignmentItem->status != 'cancelled' && $consignmentItem->inventory != null) {
                            $consignmentItem->inventory->update([
                                'count' => $consignmentItem->inventory->count + $consignmentItem->count,
                            ]);
                        }
                        $consignmentItem->update([
                            'reason' => $request->reason,
                            'status' => 'cancelled',
                        ]);
                    }
                    if ($consignment->status != 'cancelled') {
                        $consignment->update([
                            'status' => 'cancelled',
                        ]);
                    }
                    if ($consignment->send_type == 'direct') {
                        $consignment->order->update([
                            'status' => 'cancelled',
                        ]);

                        // register report
                        $this->report(__("messages.report_order_cancelled", ['uuid' => $consignment->order->uuid]), 'user', $user);
                    } else {
                        // register report
                        $this->report(__("messages.report_consignment_cancelled", ['consignment_uuid' => $consignment->uuid, 'order_uuid' => $consignment->order->uuid]), 'user', $user);
                    }
                    /** end */

                    /** Customer Club Points Deduction */
                    try {
                        if ($consignment->send_type == 'direct') {
                            $clubSetting = Setting::where('key', 'customer_club')->first();
                            $club = $clubSetting ? @unserialize($clubSetting->value) : ['price' => 0, 'score' => 0];
                            $score = ($club['price'] > 0 && $club['score'] > 0)
                                ? round(($paidPrice / str_replace(',', '', $club['price'])) * str_replace(',', '', $club['score']))
                                : 0;

                            $customer->scores()->create([
                                'score' => $score,
                                'description' => __('messages.sentence.decrease_points_for_orders') . ' ' . $consignment->order->uuid,
                                'type' => 'decrease',
                            ]);
                        }
                    } catch (\Exception $e) {
                    }
                    /** end */

                    /** handle UPG gateways */
                    if ($paymentTransaction) {
                        if (in_array($paymentTransaction->payment_gateway, ['snapppay', 'digipay'])) {
                            /** snapppay update or cancel */
                            $order->refresh();
                            $this->snapppayService->snapppay($order->orderGroup);
                            /** end */

                            /** digipay refund */
                            $order->refresh();
                            $this->digipayService->digipayRefund($order->orderGroup, $paidPrice, $request->reason);
                            /** end */
                        } else {
                            if ($consignment->send_type == 'direct') {
                                $this->refundToWallet($customer, $paidPrice, $order);
                            }
                        }
                    } else {
                        if ($consignment->send_type == 'direct') {
                            $this->refundToWallet($customer, $paidPrice, $order);
                        }
                    }
                    /** end */
                }
            });
        } catch (\Exception $e) {
            return redirect()->back()->withErrors([$e->getMessage()]);
        }

        /** send notifications */
        // sent
        if ($request->status == 'sent') {
            $sendTypeStatus = 'main_storage';
            // send notification to customer
            if ($consignment->send_type == 'direct' && $consignment->send_method != null) {
                $customer->notify(new OrderSent(
                    $consignment->order->orderGroup->uuid,
                    $consignment->order->uuid,
                    $request->tracking_code ?? null,
                    $consignment->order->get_total_bill['cancel']['price'] > 0
                ));
                $sendTypeStatus = 'direct_send';
            }
            // notification to admin
            User::whereIn('type', ['owner', 'admin'])->get()->each(function ($admin) use ($consignment, $sendTypeStatus) {
                if ($admin->type === 'owner' || $admin->can('manage-orders')) {
                    if ($sendTypeStatus == 'direct_send')
                        $admin->notify(new SellerDirectSendOrderStatus($consignment->order->orderGroup->uuid, $consignment->order->uuid, 'ارسال'));
                    else if ($sendTypeStatus == 'main_storage')
                        $admin->notify(new SellerMainStorageSendOrderStatus($consignment->order->orderGroup->uuid, $consignment->order->uuid, 'ارسال'));
                }
            });
            return redirect()->back()->with('message', [__('messages.consignment_sent')]);
        }
        // cancel
        if ($request->status == 'cancelled') {
            $sendTypeStatus = 'main_storage';
            if ($consignment->send_type == 'direct' && $consignment->send_method != null) {
                // notification
                $customer->notify(new OrderCancellation($consignment->order->orderGroup->uuid, $consignment->order->uuid));
                $sendTypeStatus = 'direct_send';
            }

            // notification to admin
            User::whereIn('type', ['owner', 'admin'])->get()->each(function ($admin) use ($consignment, $sendTypeStatus) {
                if ($admin->type === 'owner' || $admin->can('manage-orders')) {
                    if ($sendTypeStatus == 'direct_send')
                        $admin->notify(new SellerDirectSendOrderStatus($consignment->order->orderGroup->uuid, $consignment->order->uuid, 'لغو'));
                    else if ($sendTypeStatus == 'main_storage')
                        $admin->notify(new SellerMainStorageSendOrderStatus($consignment->order->orderGroup->uuid, $consignment->order->uuid, 'لغو'));
                }
            });

            return redirect()->back()->with('message', [__('messages.consignment_cancelled')]);
        }
        /** end */

        // return result
        return redirect()->back()->withErrors([__('messages.unknown_error')]);
    }

    /**
     * refund to wallet
     */
    protected function refundToWallet($customer, $paidPrice, $order)
    {
        if ($customer->wallet) {
            $customer->wallet->update([
                'amount' => $customer->wallet->amount + $paidPrice,
            ]);
        }
        $customer->transactions()->create([
            'amount' => $paidPrice,
            'transaction_id' => $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id'),
            'description' => __('messages.transaction_user_order_returned', ['order' => $order->uuid]),
            'type' => 'user_order_returned',
            'status' => 'accepted'
        ]);
    }

    /**
     * order sales factor
     */
    public function sales_factor(Consignment $consignment)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($consignment->store_id != $store->id) {
            abort(403);
        }
        if ($consignment->consignmentItems()->count() == 0) {
            abort(403);
        }
        if ($consignment->status == 'cancelled' || $consignment->status == 'returned') {
            abort(403);
        }

        $orderConsignmentItems = $consignment->consignmentItems()->where(function ($query) {
            $query->where('status', '!=', 'cancelled')->where('status', '!=', 'returned');
        })->get()->map(fn($item) => [
            'id' => $item->id,
            'title' => $item->title,
            'get_props' => $item->get_props,
            'price' => $item->price,
            'discount' => $item->discount,
            'count' => $item->count,
            'product' => $item->product != null ? [
                'id' => $item->product->id,
                'uuid' => $item->product->uuid,
                'title' => $item->product->title,
            ] : null,
        ])->toArray();

        return Inertia::render('Store/Orders/SalesFactor', [
            'order' => $consignment->order->safe,
            'consignmentItems' => $orderConsignmentItems,
            'user_national_number' => $consignment->order->user->national_number
        ]);
    }

    /**
     * order thermal sales factor
     */
    public function thermal_sales_factor(Consignment $consignment)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($consignment->store_id != $store->id) {
            abort(403);
        }
        if ($consignment->consignmentItems()->count() == 0) {
            abort(403);
        }
        if ($consignment->status == 'cancelled' || $consignment->status == 'returned') {
            abort(403);
        }

        $orderConsignmentItems = $consignment->consignmentItems()
            ->whereNotIn('status', ['cancelled', 'returned'])
            ->get()
            ->map(fn($item) => [
                'id' => $item->id,
                'title' => $item->title,
                'get_props' => $item->get_props,
                'price' => $item->price,
                'discount' => $item->discount,
                'count' => $item->count,
                'product' => $item->product ? [
                    'id' => $item->product->id,
                    'uuid' => $item->product->uuid,
                    'title' => $item->product->title
                ] : null
            ])
            ->toArray();

        return Inertia::render('Store/Orders/ThermalSalesFactor', [
            'store' => [
                'id' => $store->id,
                'uuid' => $store->uuid,
                'name' => $store->name,
            ],
            'order' => $consignment->order->safe,
            'consignmentItems' => $orderConsignmentItems,
            'user_national_number' => $consignment->order->user->national_number,
        ]);
    }
}
