<?php

namespace App\Http\Controllers\Store\Blog;

use App\Models\Article;
use App\Http\Controllers\Controller;
use App\Models\ArticleCategory;
use App\Models\ArticleTag;
use App\Models\Brand;
use App\Models\Setting;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ArticlesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        /** @var LengthAwarePaginator */
        $articles = $user->articles()->where('status', '!=', 'deleted')->where('status', '!=', 'deep_deleted')->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->when($request->input('publish'), function ($query, $publish) {
            if ($publish) {
                $query->where('status', 'publish');
            }
        })->when($request->input('awaiting'), function ($query, $awaiting) {
            if ($awaiting) {
                $query->where('status', 'awaiting');
            }
        })->when($request->input('draft'), function ($query, $draft) {
            if ($draft) {
                $query->where('status', 'draft');
            }
        })->when($request->input('mostViewer'), function ($query, $mostViewer) {
            if ($mostViewer) {
                $query->orderBy('view_count', 'desc');
            }
        })->orderByRaw("FIELD(status , 'awaiting') DESC")
            ->orderBy('created_at', 'desc')->paginate(20)->through(fn($article) => [
                'id' => $article->id,
                'title' => $article->title,
                'slug' => $article->slug,
                'get_image' => $article->get_image,
                'editor_choice' => $article->editor_choice,
                'get_formatted_view_count' => $article->get_formatted_view_count,
                'status' => $article->status,
                'jalali_updated_at' => $article->jalali_updated_at,
                'user' => [
                    'id' => $article->user->id,
                    'get_name' => $article->user->get_name,
                    'type' => $article->user->type,
                    'seller' => $article->user->seller,
                    'store' => $article->user->store != null ? [
                        'id' => $article->user->store->id,
                        'uuid' => $article->user->store->uuid,
                        'name' => $article->user->store->name,
                    ] : null,
                ],
            ]);

        $articles->withQueryString();

        return Inertia::render('Store/Blog/Articles/List', [
            'articles' => $articles
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = ArticleCategory::where('status', 'publish')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'status' => $category->status,
        ]);

        return Inertia::render('Store/Blog/Articles/Create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        $request->validate([
            'title' => 'required',
            'category' => 'required',
            'comment_status' => 'required',
        ]);

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null) {
            $articleWithSlug = Article::where('slug', $request->slug)->first();
            if ($articleWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        /** validate source */
        $source = Setting::where('key', 'shop_title')->first() != null ? Setting::where('key', 'shop_title')->first()->value : null;
        if ($request->exists('source') && $request->source != null) {
            $source = $request->source;
        }

        /** validate brand */
        $brand = Brand::where('title', $request->brand)->first();

        $article = $user->articles()->create([
            'edited_by' => $user->id,
            'brand_id' => $brand != null ? $brand->id : null,
            'category_id' => $request->category['id'],
            'uuid' => 'article-' . rand(1000000, 9999999),
            'title' => $request->title,
            'slug' => $request->exists('slug') && $request->slug != null ? $request->slug : null,
            'content' => $request->exists('content') ? $request->content : null,
            'source' => $source,
            'read_time' => $request->exists('read_time') ? $request->read_time : null,
            'type' => "text",

            'image' => $request->exists('image') ? $request->image : null,
            'video' => null,
            'podcast' => null,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? $request->seo_canonical : null,

            'editor_choice' => false,
            'comment_status' => $request->comment_status,

            'reason' => 'create',
            'status' => "awaiting",
        ]);

        /** create and sync tags */
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            foreach (json_decode($request->tags) as $tag) {
                $findTag = ArticleTag::where('title', $tag->value)->get()->first();
                if ($findTag != null) {
                    $tags[] = $findTag->id;
                } else {
                    $createdTag = ArticleTag::create([
                        'title' => $tag->value,
                        'status' => 'publish',
                    ]);
                    $tags[] = $createdTag->id;
                }
            }
            $article->articleTags()->sync($tags);
        }

        /** Report Action */
        $reportMessage = __("messages.report_article_created");
        $this->report($reportMessage, 'article', $article);
        /******************/

        return redirect()->route('seller.articles.index')->with('message', [__('messages.article_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Article $article)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Article $article)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($article->status == 'publish' || $article->status == 'draft') {
            return redirect()->route('seller.articles.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($article->user_id != $user->id) {
            return redirect()->route('seller.articles.index')->withErrors([__('messages.unauthorized')]);
        }

        $categories = ArticleCategory::where('status', 'publish')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
            'status' => $category->status,
        ]);

        return Inertia::render('Store/Blog/Articles/Edit', [
            'article' => [
                'id' => $article->id,
                'title' => $article->title,
                'slug' => $article->slug,
                'get_tags_string' => $article->get_tags_string,
                'category_id' => $article->category_id,
                'get_image' => $article->get_image,
                'image' => $article->image,
                'content' => $article->content,
                'source' => $article->source,
                'read_time' => $article->read_time,
                'seo_title' => $article->seo_title,
                'seo_description' => $article->seo_description,
                'seo_keywords' => $article->seo_keywords,
                'seo_canonical' => $article->seo_canonical,
                'comment_status' => $article->comment_status,
                'reason' => $article->reason,
                'status' => $article->status,

                'brand' => $article->brand != null ? [
                    'id' => $article->brand->id,
                    'title' => $article->brand->title,
                ] : null,
            ],
            'categories' => $categories,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Article $article)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($article->status == 'publish' || $article->status == 'draft') {
            return redirect()->route('seller.articles.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($article->user_id != $user->id) {
            return redirect()->route('seller.articles.index')->withErrors([__('messages.unauthorized')]);
        }

        $request->validate([
            'title' => 'required',
            'category' => 'required',
            'comment_status' => 'required',
        ]);

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null && $request->slug != $article->slug) {
            $articleWithSlug = Article::where('slug', $request->slug)->first();
            if ($articleWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        /** validate source */
        $source = $article->source;
        if ($request->exists('source') && $request->source != null) {
            $source = $request->source;
        }

        /** validate brand */
        $brand = Brand::where('title', $request->brand)->first();

        $article->update([
            'edited_by' => $user->id,
            'brand_id' => $brand != null ? $brand->id : $article->brand_id,
            'category_id' => $request->category['id'],
            'title' => $request->title,
            'slug' => $request->exists('slug') ? $request->slug : $article->slug,
            'content' => $request->content,
            'source' => $source,
            'read_time' => $request->exists('read_time') ? $request->read_time : $article->read_time,

            'image' => $request->exists('image') ? $request->image : $article->image,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $article->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $article->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $article->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? $request->seo_canonical : $article->seo_canonical,

            'comment_status' => $request->comment_status,

            'reason' => 'update',
            'status' => "awaiting",
        ]);

        /** create and sync tags */
        if ($request->exists('tags') && $request->tags != null && $request->tags != '') {
            $tags = [];
            if (is_array($request->tags)) {
                foreach ($request->tags as $tag) {
                    $findTag = ArticleTag::where('title', $tag)->get()->first();
                    if ($findTag != null) {
                        $tags[] = $findTag->id;
                    } else {
                        $createdTag = ArticleTag::create([
                            'title' => $tag,
                            'status' => 'publish',
                        ]);
                        $tags[] = $createdTag->id;
                    }
                }
            } else {
                foreach (json_decode($request->tags) as $tag) {
                    $findTag = ArticleTag::where('title', $tag->value)->get()->first();
                    if ($findTag != null) {
                        $tags[] = $findTag->id;
                    } else {
                        $createdTag = ArticleTag::create([
                            'title' => $tag->value,
                            'status' => 'publish',
                        ]);
                        $tags[] = $createdTag->id;
                    }
                }
            }


            $article->articleTags()->sync($tags);
        }

        /** Report Action */
        $reportMessage = __("messages.report_article_updated");
        $this->report($reportMessage, 'article', $article);
        /******************/

        return redirect()->route('seller.articles.index')->with('message', [__('messages.article_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Article $article)
    {
        /** @var App\Models\User $user */
        $user = auth()->user();
        $store = $user->store;

        if ($article->status == 'publish' || $article->status == 'draft') {
            return redirect()->route('seller.articles.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($article->user_id != $user->id) {
            return redirect()->route('seller.articles.index')->withErrors([__('messages.unauthorized')]);
        }

        if ($article->status == "awaiting" && $article->reason == "create") {
            $article->delete();
        } else if ($article->status == "rejected") {
            $article->delete();
        } else {
            return redirect()->back()->with('message', [__('messages.can_not_delete_published_article')]);
        }

        /** Report Action */
        $reportMessage = __("messages.report_article_to_trash");
        $this->report($reportMessage, 'article', $article);
        /******************/

        return redirect()->route('seller.articles.index')->withErrors([__('messages.article_deleted')]);
    }

    /** search raw brand */
    public function searchBrand(Request $request)
    {
        $brands = Brand::query()->where('status', 'publish')->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->pluck('title')->take(50)->toArray();

        return response($brands);
    }
}
