<?php

namespace App\Http\Controllers\Main\Products;

use App\Http\Controllers\Controller;
use App\Http\Requests\Main\ProductCommentAddRequest;
use App\Models\Product;
use App\Models\ProductComment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CommentsController extends Controller
{
    /**
     * add product comment
     */
    public function add(ProductCommentAddRequest $request, Product $product)
    {
        // check user login
        if (!auth()->check())
            return redirect()->back()->withErrors([__('messages.login_required')]);

        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** validate suggation */
        $suggation = null;
        if ($request->exists('suggation') && $request->suggation != null) {
            if ($request->suggation == 'suggest') {
                $suggation =  1;
            } else if ($request->suggation == 'not_suggest') {
                $suggation =  0;
            }
        }

        /** images */
        $images = [];
        if ($request->exists('images') && $request->images != null) {
            $request->validate([
                'images' => ['required', 'array'],
                'images.*' => ['file', 'max:2048', 'mimes:jpeg,jpg,png'],
            ]);

            foreach ($request->file('images') as $file) {
                $fileFormat = $file->getClientOriginalExtension();
                $fileType = null;

                switch ($fileFormat) {
                    case "jpeg":
                    case "jpg":
                    case "png":
                        $fileType = "image";
                        break;
                    default:
                        return redirect()->back()->withErrors('messages.unknown_error');
                        break;
                }

                $fileName = 'file.' . time() . '.' . rand(10000, 99999) . '.' . $file->getClientOriginalExtension();
                $fileSize = floor($file->getSize() / 1000);
                $fileSaveAddressNoName = 'library/' . date('Y-m-d') . '/' . $user->id;
                $fileSaveAddress = $fileSaveAddressNoName . '/' . $fileName;
                $imageResolution = null;

                if ($fileType == "image") {
                    $imageResolution = getimagesize($file)[0] . '×' . getimagesize($file)[1];
                }

                $uploadPath = '/media/' . $fileSaveAddress;
                Storage::disk('media')->put($fileSaveAddress, file_get_contents($file));
                $imageFile = $user->files()->create([
                    'name' => $fileName,
                    'url' => $uploadPath,
                    'description' => $user->get_name,
                    'extension' => $fileFormat,
                    'size' => $fileSize,
                    'resolution' => $imageResolution,
                    'type' => $fileType,
                ]);

                $images[] = $imageFile->id;
            }
        }

        $product->productComments()->create([
            'user_id' => $user->id,
            'content' => $request->content,
            'strengths' => $request->exists('strengths') && $request->strengths != null ? serialize($request->strengths) : null,
            'weakness' => $request->exists('weakness') && $request->weakness != null ? serialize($request->weakness) : null,
            'buy_suggest' => $suggation,
            'rating' => $request->rating,
            'images' => serialize($images),
            'status' => $user->type == "owner" || $user->type == "admin" ? 'publish' : 'awaiting',
        ]);

        if ($user->type == "owner" || $user->type == "admin")
            return redirect()->back()->with('message', [__('messages.comment_published')]);
        else
            return redirect()->back()->with('message', [__('messages.comment_submited')]);
    }

    /**
     * like comment
     */
    public function like(Request $request, ProductComment $comment)
    {
        $liked_comments = $request->session()->get('liked_comments', []);
        $disliked_comments = $request->session()->get('disliked_comments', []);

        if (!in_array($comment->id, $liked_comments) && !in_array($comment->id, $disliked_comments)) {
            $comment->update([
                'vote_up' => $comment->vote_up + 1
            ]);

            $liked_comments[] = $comment->id;
            session(['liked_comments' => $liked_comments]);
        }

        return redirect()->back();
    }

    /**
     * dislike comment
     */
    public function dislike(Request $request, ProductComment $comment)
    {
        $liked_comments = $request->session()->get('liked_comments', []);
        $disliked_comments = $request->session()->get('disliked_comments', []);

        if (!in_array($comment->id, $liked_comments) && !in_array($comment->id, $disliked_comments)) {
            $comment->update([
                'vote_down' => $comment->vote_down + 1
            ]);

            $disliked_comments[] = $comment->id;
            session(['disliked_comments' => $disliked_comments]);
        }

        return redirect()->back();
    }

    /**
     * report comment
     */
    public function report_comment(ProductComment $comment)
    {
        // check user login
        if (!auth()->check())
            return redirect()->back()->withErrors([__('messages.login_required')]);

        if (!$comment->reported) {
            $comment->update([
                'reported' => true,
            ]);
        }

        return redirect()->back()->with('message', [__('messages.comment_reported')]);
    }
}
