<?php

namespace App\Http\Controllers\Main\Cart;

use App\Helpers\CommonFunctionsHelper;
use App\Helpers\DropdownListHelper;
use App\Helpers\Places;
use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Main\AddToCartRequest;
use App\Http\Requests\Main\ChangeCartCountRequest;
use App\Http\Requests\Main\DeleteFromCartRequest;
use App\Http\Requests\User\AddressRequest;
use App\Models\Address;
use App\Models\Consignment;
use App\Models\ConsignmentItem;
use App\Models\Discount;
use App\Models\Inventory;
use App\Models\Order;
use App\Models\PaymentMeta;
use App\Models\Setting;
use App\Models\User;
use App\Notifications\WalletCharge;
use App\Services\Cart\CartService;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;
use Morilog\Jalali\Jalalian;
use Shetabit\Multipay\Drivers\SnappPay\SnappPay;
use Illuminate\Support\Facades\Cookie;
use Shetabit\Multipay\Exceptions\PurchaseFailedException;
use Shetabit\Multipay\Invoice;
use Shetabit\Payment\Facade\Payment;

class CartController extends Controller
{
    use Places, UUIDHelper, CommonFunctionsHelper, DropdownListHelper;

    /**
     * cart page
     */
    public function index()
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        $consignmentItems = [];
        $totalPrice = 0;
        $totalDiscount = 0;
        $totalFinalPrice = 0;
        $customerClubScore = 0;
        if ($user != null) {
            /** @var \App\Models\Order $order */
            $order = $user->orders()->where('status', 'awaiting')->first();

            // reset discount code
            if ($order != null) {
                $order->update([
                    'discount' => null,
                ]);
            }

            // get order consignment
            $consignments = [];
            if ($order != null) {
                $consignments = $order->consignments()->with('consignmentItems')->get();
            }

            // add consignment items
            $consignmentItems = new Collection();
            foreach ($consignments as $consignment) {
                $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
            }

            // main store post ways status
            $mainStorePaymentMetaStatus = false;
            $mainStorePaymentMeta = PaymentMeta::where('store_id', null)->first();
            if ($mainStorePaymentMeta != null && $mainStorePaymentMeta->post_ways != null &&  count(unserialize($mainStorePaymentMeta->post_ways)) > 0) {
                $postWays = unserialize($mainStorePaymentMeta->post_ways);
                $hasActive = collect($postWays)->contains(function ($item) {
                    return $item['status'] === true;
                });
                $mainStorePaymentMetaStatus = $hasActive;
            }

            // sync consignmentItems count
            foreach ($consignmentItems as $consignmentItem) {
                if ($consignmentItem->store != null && $consignmentItem->store->direct_post && !$consignmentItem->store->transportation_status) {
                    $consignmentItem->delete();

                    // update consignment items
                    $consignmentItems = new Collection();
                    foreach ($consignments as $consignment) {
                        $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                    }
                } else if (!$mainStorePaymentMetaStatus) {
                    $consignmentItem->delete();

                    // update consignment items
                    $consignmentItems = new Collection();
                    foreach ($consignments as $consignment) {
                        $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                    }
                }

                $inventory = $consignmentItem->inventory;
                if ($inventory->count == 0) {
                    $consignmentItem->delete();

                    // update consignment items
                    $consignmentItems = new Collection();
                    foreach ($consignments as $consignment) {
                        $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                    }
                } else if ($consignmentItem->count > $inventory->count) {
                    $consignmentItem->update([
                        'count' => $inventory->count
                    ]);
                }
            }

            // prices
            $totalPrice = $consignmentItems->sum('get_full_count_price') + $consignmentItems->sum('get_full_count_discount');
            $totalDiscount = $consignmentItems->sum('get_full_count_discount');
            $totalFinalPrice = $consignmentItems->sum('get_full_count_price');

            // get customer club (price and score)
            $customerClub = Setting::where('key', 'customer_club')->first();
            $customerClub = $customerClub != null ? unserialize($customerClub->value) : ['price' => 0, 'score' => 0];
            $customerClubScore = $customerClub['price'] > 0 && $customerClub['score'] > 0 ? round(($consignmentItems->sum('get_full_count_price') / str_replace(',', '', $customerClub['price'])) * str_replace(',', '', $customerClub['score'])) : 0;
        }

        return Inertia::render('Main/Cart/Cart', [
            'items' => fn() => $consignmentItems != null ? $consignmentItems->map(fn($item) => [
                'id' => $item->id,
                'get_props' => $item->get_props,
                'inventory_id' => $item->inventory_id,
                'count' => $item->count,
                'discount' => $item->discount,
                'price' => $item->price,

                'inventory' => $item->inventory ? [
                    'id' => $item->inventory->id,
                    'send_time' => $item->inventory->send_time,
                ] : null,
                'product' => $item->product ? [
                    'id' => $item->product->id,
                    'title' => $item->product->title,
                    'slug' => $item->product->slug,
                    'get_images' => $item->product->get_images,
                ] : null,
                'store' => $item->store != null ? [
                    'id' => $item->store->id,
                    'uuid' => $item->store->uuid,
                    'name' => $item->store->name,
                ] : null,
            ]) : [],
            'totalPrice' => $totalPrice,
            'totalDiscount' => $totalDiscount,
            'totalFinalPrice' => $totalFinalPrice,
            'customerClubScore' => $customerClubScore,
        ])->title(__('messages.title.cart'));
    }

    /**
     * add to cart
     */
    public function add(AddToCartRequest $request, CartService $cartService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($user == null) {
            return redirect()->route('login');
        }

        // main store post ways status
        $mainStorePaymentMetaStatus = false;
        $mainStorePaymentMeta = PaymentMeta::where('store_id', null)->first();
        if ($mainStorePaymentMeta != null && $mainStorePaymentMeta->post_ways != null &&  count(unserialize($mainStorePaymentMeta->post_ways)) > 0) {
            $postWays = unserialize($mainStorePaymentMeta->post_ways);
            $hasActive = collect($postWays)->contains(function ($item) {
                return $item['status'] === true;
            });
            $mainStorePaymentMetaStatus = $hasActive;
        }
        if (!$mainStorePaymentMetaStatus) {
            return redirect()->back()->withErrors([__('messages.transportation_not_exist')]);
        }

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();

        // add new order
        if ($order == null) {
            $order = $user->orders()->create([
                'uuid' => $this->generateUniqueRandomNumber(9, \App\Models\Order::class, 'uuid'),
                'status' => 'awaiting',
            ]);
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get product
        $product = $inventory->product;
        if (!$product->in_stock_status) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get count
        $count = $request->count;

        // handle count
        if ($count > $inventory->count) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }
        if ($inventory->max_sale != null && $count > $inventory->max_sale) {
            return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
        }
        if ($inventory->min_sale != null && $count < $inventory->min_sale) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => $inventory->min_sale])]);
        }

        // get inventory props
        $props = $inventory->props->toArray();

        // get store
        $store = $inventory->store;

        /** @var \App\Models\Consignment $consignment */
        $consignment = $order->consignments()->where('store_id', $store != null ? $store->id : null)->first();

        // handle consignment
        if ($consignment == null) {
            $consignment = $order->consignments()->create([
                'user_id' => $user->id,
                'store_id' => $store != null ? $store->id : null,
                'uuid' => $this->generateUniqueRandomNumber(9, \App\Models\Consignment::class, 'uuid', 'cn-'),
                'status' => 'created',
            ]);
        }

        // handle discount tree
        $price = $inventory->get_final_price;
        if ($inventory->is_discount_valid) {
            foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                if ($count >= $dt['count']) {
                    $price = str_replace(',', '', $dt['price']);
                    break;
                }
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // handle payment gateway price limit
        $pgPriceLimit = $cartService->validatePaymentGatewayPrice($order, $count, $price);
        if ($pgPriceLimit) {
            return redirect()->back()->withErrors([__('messages.pg_price_limit_error', ['price' => number_format($pgPriceLimit)])]);
        }

        // check for exist consignment items
        $consignmentItem = $consignment->consignmentItems()->where('inventory_id', $inventory->id)->first();

        // add affiliate
        $affiliatesCookie = request()->cookie('affiliates');
        $affiliates = $affiliatesCookie != null ? unserialize($affiliatesCookie) : [];
        $affiliate = null;
        if (array_key_exists($product->uuid, $affiliates)) {
            $targetUser = User::where('id', $affiliates[$product->uuid])->first();
            if ($targetUser != null) {
                $affiliate = $targetUser->id;
            }
        }

        // add inventory to consignment item
        if ($consignmentItem == null) {
            $consignmentItem = $consignment->consignmentItems()->create([
                'store_id' => $store != null ? $store->id : null,
                'inventory_id' => $inventory->id,
                'product_id' => $inventory->product->id,
                'storage_id' => $inventory->storage->id,
                'title' => $inventory->product->title,
                'props' => serialize($props),
                'count' => $count,
                'price' => $price,
                'discount' => $discount,
                'affiliate' => $affiliate,
                'status' => $store != null ? 'awaiting_request_from_seller' : 'ready_to_send',
            ]);
        } else {
            // get new count
            $newCount = $consignmentItem->count + $count;

            // validate again max_sale and total inventory count
            if ($inventory->max_sale != null && $newCount > $inventory->max_sale) {
                return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
            }
            if ($newCount > $inventory->count) {
                return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
            }

            // get new price
            if ($inventory->is_discount_valid) {
                foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                    if ($newCount >= $dt['count']) {
                        $price = str_replace(',', '', $dt['price']);
                        break;
                    }
                }
            }

            // get new discount
            $discount = $inventory->price - $price;

            // update count and price and discount
            $consignmentItem->update([
                'count' => $newCount,
                'price' => $price,
                'discount' => $discount,
                'affiliate' => $affiliate,
            ]);
        }

        // get final count
        $finalCount = $consignment->consignmentItems()->where('inventory_id', $inventory->id)->first()->count;

        return redirect()->back()->with('add_to_cart', [
            'type' => 'added_to_cart',
            'data' => [
                'props' => $props,
                'count' => $finalCount,
                'price' => $price,
                'discount' => $discount,
            ],
        ]);
    }

    /**
     * delete from cart
     */
    public function delete(DeleteFromCartRequest $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($user == null) {
            return redirect()->route('login');
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // delete consignment item
        $consignmentItem->delete();

        return redirect()->back();
    }

    /**
     * increase 1 unit cart count
     */
    public function increase(ChangeCartCountRequest $request, CartService $cartService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($user == null) {
            return redirect()->route('login');
        }

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // check for inventory count
        if ($inventory->count == 0) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }

        // get product
        $product = $inventory->product;
        if (!$product->in_stock_status) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get final count
        $finalCount = $consignmentItem->count + 1;

        // handle count
        if ($finalCount > $inventory->count) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }
        if ($inventory->max_sale != null && $finalCount > $inventory->max_sale) {
            return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
        }

        // handle discount
        $price = $inventory->get_final_price;
        if ($inventory->is_discount_valid) {
            foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                if ($finalCount >= $dt['count']) {
                    $price = str_replace(',', '', $dt['price']);
                    break;
                }
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // handle payment gateway price limit
        $pgPriceLimit = $cartService->validatePaymentGatewayPrice($order, 1, $price);
        if ($pgPriceLimit) {
            return redirect()->back()->withErrors([__('messages.pg_price_limit_error', ['price' => number_format($pgPriceLimit)])]);
        }

        // increase consignment item count
        $consignmentItem->update([
            'count' => $finalCount,
            'price' => $price,
            'discount' => $discount,
        ]);

        return redirect()->back();
    }

    /**
     * decrease 1 unit cart count
     */
    public function decrease(ChangeCartCountRequest $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        if ($user == null) {
            return redirect()->route('login');
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get final count
        $finalCount = $consignmentItem->count - 1;

        // handle count
        if ($finalCount < 1) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => 1])]);
        }
        if ($inventory->min_sale != null && $finalCount < $inventory->min_sale) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => $inventory->min_sale])]);
        }

        // handle discount
        $price = $inventory->get_final_price;
        foreach (array_reverse($inventory->get_discount_tree) as $dt) {
            if ($finalCount >= $dt['count']) {
                $price = str_replace(',', '', $dt['price']);
                break;
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // reduce consignment item count
        $consignmentItem->update([
            'count' => $finalCount,
            'price' => $price,
            'discount' => $discount,
        ]);

        return redirect()->back();
    }

    /**
     * cart shiping page
     */
    public function shiping(CartService $cartService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        if ($order == null) {
            return redirect()->route('cart');
        }
        // get order consignment
        $consignments = [];
        $consignments = $order->consignments()->with('store')->with('consignmentItems')->get();
        // get consignment items
        $consignmentItems = new Collection();
        foreach ($consignments as $consignment) {
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
        }

        // reset discount code
        $order->update([
            'discount' => null,
        ]);

        // main store post ways status
        $mainStorePaymentMetaStatus = false;
        $mainStorePaymentMeta = PaymentMeta::where('store_id', null)->first();
        if ($mainStorePaymentMeta != null && $mainStorePaymentMeta->post_ways != null &&  count(unserialize($mainStorePaymentMeta->post_ways)) > 0) {
            $postWays = unserialize($mainStorePaymentMeta->post_ways);
            $hasActive = collect($postWays)->contains(function ($item) {
                return $item['status'] === true;
            });
            $mainStorePaymentMetaStatus = $hasActive;
        }

        // sync consignmentItems count
        foreach ($consignmentItems as $consignmentItem) {
            if ($consignmentItem->store != null && $consignmentItem->store->direct_post && !$consignmentItem->store->transportation_status) {
                $consignmentItem->delete();

                // update consignment items
                $consignmentItems = new Collection();
                foreach ($consignments as $consignment) {
                    $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                }
            } else if (!$mainStorePaymentMetaStatus) {
                $consignmentItem->delete();

                // update consignment items
                $consignmentItems = new Collection();
                foreach ($consignments as $consignment) {
                    $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                }
            }

            $inventory = $consignmentItem->inventory;
            if ($inventory->count == 0) {
                $consignmentItem->delete();

                // update consignment items
                $consignmentItems = new Collection();
                foreach ($consignments as $consignment) {
                    $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                }
            } else if ($consignmentItem->count > $inventory->count) {
                $consignmentItem->update([
                    'count' => $inventory->count
                ]);
            }
        }

        // validate empty cart
        if ($consignmentItems->count() == 0) {
            return redirect()->route('cart');
        }

        // get customer club (price and score)
        $customerClub = Setting::where('key', 'customer_club')->first();
        $customerClub = $customerClub != null ? unserialize($customerClub->value) : ['price' => 0, 'score' => 0];
        $customerClubScore = $customerClub['price'] > 0 && $customerClub['score'] > 0 ? round(($consignmentItems->sum('get_full_count_price') / str_replace(',', '', $customerClub['price'])) * str_replace(',', '', $customerClub['score'])) : 0;

        // user addresses
        $addresses = $user->addresses()->latest()->get()->map(fn($address) => [
            'id' => $address->id,
            'user_id' => $address->user_id,
            'first_name' => $address->first_name,
            'last_name' => $address->last_name,
            'state' => $address->state,
            'city' => $address->city,
            'phone' => $address->phone,
            'postal_code' => $address->postal_code,
            'full_address' => $address->full_address,
            'active' => $address->active,
            'get_map' => $address->get_map,
            'jalali_created_at' => $address->jalali_created_at,
            'jalali_updated_at' => $address->jalali_updated_at,
        ]);

        // main payment meta
        $paymentMeta = PaymentMeta::whereNull('store_id')->first();

        /********* address *********/
        $selectedAddress = $order->address != null ? unserialize($order->address) : null;
        if ($selectedAddress == null) {
            // delete prev send methods
            $order->update([
                'send_method' => null,
            ]);
            foreach ($consignments as $consignment) {
                $consignment->update([
                    'send_method' => null,
                    'send_type' => null,
                ]);
            }

            // select user active address for default
            $activeAddress = $user->addresses()->where('active', true)->first() != null ? $user->addresses()->where('active', true)->first() : null;
            if ($activeAddress != null) {
                $address = $activeAddress->toArray();
                if (isset($address['safe'])) unset($address['safe']);
                if (isset($address['user'])) unset($address['user']);
                $finalAddress = [
                    'type' => 'user',
                    'address' => $address
                ];
                $order->update([
                    'address' => serialize($finalAddress)
                ]);
                // resync
                $selectedAddress = $order->address != null ? unserialize($order->address) : null;
            }
        }

        $consignmentsInformation = [];
        if ($selectedAddress != null && $selectedAddress['type'] == 'shop') {
            $consignmentsInformation = $cartService->singleConsignmentInformation($order);
        } else {
            // seprated Consignments
            $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress);

            /********* send method *********/
            if ($selectedAddress != null && $selectedAddress['type'] == 'user') {
                // determine send_method from get_post_ways
                foreach ($consignmentsInformation['consignments'] as $key => &$consignment) {
                    $cs = null;
                    if ($key != 'main') {
                        $cs = Consignment::where('user_id', $user->id)->where('uuid', $consignment['uuid'])->first();
                    }

                    if (!empty($consignment['get_post_ways'])) {
                        if ($key == 'main') {
                            if ($order->send_method == null) {
                                $order->update([
                                    'send_method' => serialize([
                                        'type' => 'post',
                                        'item' => $consignment['get_post_ways'][0]
                                    ])
                                ]);
                            }
                        } else {
                            if ($cs != null && $cs->send_method == null) {
                                $cs->update([
                                    'send_method' => serialize([
                                        'type' => 'post',
                                        'item' => $consignment['get_post_ways'][0]
                                    ]),
                                    'send_type' => 'direct',
                                ]);
                            }
                        }
                    } else {
                        if ($cs != null) {
                            $cs->delete();
                        }
                    }
                }
                unset($consignment);
            }

            // resync seprated Consignments
            $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress);
        }

        return Inertia::render('Main/Cart/Shiping', [
            'places' => $this->places(),
            'addresses' => $addresses,

            'selectedAddress' => $selectedAddress,

            'sepratedConsignments' => $consignmentsInformation['consignments'],
            'itemsCount' => count($consignmentItems),
            'financial' => $consignmentsInformation['financial'],
            'customerClubScore' => $customerClubScore,

            'paymentMeta' => $paymentMeta,
        ])->title(__('messages.title.cart_shiping'));
    }

    /**
     * update shiping data
     */
    public function shiping_update(Request $request, CartService $cartService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();
        if (!$user) {
            return redirect()->route('login');
        }

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        if (!$order) {
            return redirect()->route('cart');
        }

        // get order consignment
        $consignments = [];
        $consignments = $order->consignments()->with('consignmentItems')->get();
        // add consignment items
        $consignmentItems = new Collection();
        foreach ($consignments as $consignment) {
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
        }

        // validate empty cart
        if ($consignmentItems->count() == 0) {
            return redirect()->route('cart');
        }

        // main payment meta
        $paymentMeta = PaymentMeta::where('store_id', null)->first();

        /*********** address ***********/
        $finalAddress = $order->address != null ? unserialize($order->address) : null;
        if ($request->exists('address_id') && $request->address_id != null && $request->exists('type') && $request->type != null) {
            if ($request->type == 'user') {
                // selected address
                $address = Address::where('id', $request->address_id)->where('user_id', $user->id)->first();
                if ($address == null) {
                    return redirect()->back()->withErrors('messages.address_not_found');
                }

                // activate user address
                $cartService->activateAddress($user, $address);

                $address = $address->toArray();
                if (isset($address['safe'])) unset($address['safe']);
                if (isset($address['user'])) unset($address['user']);
                $finalAddress = [
                    'type' => 'user',
                    'address' => $address
                ];
            } else if ($request->type == 'shop') {
                if (array_key_exists($request->address_id, $paymentMeta->get_reception_centers_addresses) && $paymentMeta->get_reception_centers_addresses[$request->address_id] != null) {
                    $finalAddress = [
                        'type' => 'shop',
                        'address' => [
                            'id' => $request->address_id,
                            ...$paymentMeta->get_reception_centers_addresses[$request->address_id]
                        ],
                    ];
                }
            }

            // validate address
            if ($finalAddress == null || !is_array($finalAddress)) {
                return redirect()->back()->withErrors([__('messages.address_not_selected_or_wrong')]);
            }

            // reset send_method
            $order->update([
                'send_method' => null
            ]);
            foreach ($consignments as $consignment) {
                $consignment->update([
                    'send_method' => null,
                    'send_type' => null,
                ]);
            }
        }
        // update address
        $order->update([
            'address' => serialize($finalAddress),
        ]);
        $finalAddress = $order->address != null ? unserialize($order->address) : null;

        /*********** send method ***********/
        $sendMethod = null;
        if ($finalAddress != null && $finalAddress['type'] == 'user') {
            if ($request->exists('consignment_uuid') && $request->consignment_uuid != null && $request->exists('send_method_uuid') && $request->send_method_uuid != null) {
                if ($request->exists('type') && $request->type == 'main') {
                    $order = Order::where('user_id', $user->id)->where('status', 'awaiting')->where('uuid', $request->consignment_uuid)->first();
                    if ($order == null) {
                        return redirect()->back()->withErrors([__('messages.unknown_error')]);
                    }

                    $defaultPostWays = $paymentMeta?->get_post_ways ?? [];
                    $defaultPostWays = array_values(array_filter($defaultPostWays, fn($way) => $way['status'] ?? false));
                    $selectedSendMethod = array_filter($defaultPostWays, fn($item) => $item['uuid'] === $request->send_method_uuid);
                    if (count($selectedSendMethod) > 0) {
                        $sendMethod = [
                            'type' => 'post',
                            'item' => reset($selectedSendMethod)
                        ];

                        $order->update([
                            'send_method' => serialize($sendMethod),
                        ]);
                    }
                } else if ($request->exists('type') && $request->type == 'direct') {
                    $consignment = Consignment::where('user_id', $user->id)->where('uuid', $request->consignment_uuid)->first();
                    if ($consignment == null) {
                        return redirect()->back()->withErrors([__('messages.unknown_error')]);
                    }

                    $consignmentStorePaymentMeta = $consignment->store->paymentMeta;
                    $consignmentStorePaymentMeta = $consignmentStorePaymentMeta?->get_post_ways ?? [];
                    $consignmentStorePaymentMeta = array_values(array_filter($consignmentStorePaymentMeta, fn($way) => $way['status'] ?? false));
                    $selectedSendMethod = array_filter($consignmentStorePaymentMeta, fn($item) => $item['uuid'] === $request->send_method_uuid);
                    if (count($selectedSendMethod) > 0) {
                        $sendMethod = [
                            'type' => 'post',
                            'item' => reset($selectedSendMethod)
                        ];

                        $consignment->update([
                            'send_method' => serialize($sendMethod),
                            'send_type' => 'direct',
                        ]);
                    }
                }

                // validate send method
                if ($sendMethod == null || !is_array($sendMethod)) {
                    return redirect()->back()->withErrors([__('messages.send_way_not_selected_or_wrong')]);
                }
            }
        } else if ($finalAddress != null && $finalAddress['type'] == 'shop') {
            $sendMethod = [
                'type' => 'inperson',
            ];

            $order->update([
                'send_method' => serialize($sendMethod),
            ]);
        }

        // reset discount code
        $order->update([
            'discount' => null,
        ]);

        // update and return back
        return redirect()->route('cart.shiping');
    }

    /**
     * cart payment page
     */
    public function payment(Request $request, CartService $cartService)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        if ($order == null) {
            return redirect()->route('cart');
        }

        // selected address
        $selectedAddress = $order->address != null ? unserialize($order->address) : null;

        // send method status
        $sendMethodValidation = $order->send_method != null ? true : false;

        // get order consignment
        $consignments = $order->consignments()->with('consignmentItems')->get();

        // add consignment items
        $consignmentItems = new Collection();
        foreach ($consignments as $consignment) {
            if ($consignment->send_method != null) {
                $sendMethodValidation = true;
            }
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
        }

        // validate address and send_method
        if ($selectedAddress == null || !$sendMethodValidation) {
            return redirect()->route('cart');
        }

        // main store post ways status
        $mainStorePaymentMetaStatus = false;
        $mainStorePaymentMeta = PaymentMeta::where('store_id', null)->first();
        if ($mainStorePaymentMeta != null && $mainStorePaymentMeta->post_ways != null &&  count(unserialize($mainStorePaymentMeta->post_ways)) > 0) {
            $postWays = unserialize($mainStorePaymentMeta->post_ways);
            $hasActive = collect($postWays)->contains(function ($item) {
                return $item['status'] === true;
            });
            $mainStorePaymentMetaStatus = $hasActive;
        }

        // sync consignmentItems count
        foreach ($consignmentItems as $consignmentItem) {
            if ($consignmentItem->store != null && $consignmentItem->store->direct_post && !$consignmentItem->store->transportation_status) {
                $consignmentItem->delete();

                // update consignment items
                $consignmentItems = new Collection();
                foreach ($consignments as $consignment) {
                    $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                }
            } else if (!$mainStorePaymentMetaStatus) {
                $consignmentItem->delete();

                // update consignment items
                $consignmentItems = new Collection();
                foreach ($consignments as $consignment) {
                    $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                }
            }

            $inventory = $consignmentItem->inventory;
            if ($inventory->count == 0) {
                $consignmentItem->delete();

                // update consignment items
                $consignmentItems = new Collection();
                foreach ($consignments as $consignment) {
                    $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
                }
            } else if ($consignmentItem->count > $inventory->count) {
                $consignmentItem->update([
                    'count' => $inventory->count
                ]);
            }
        }

        // validate empty cart
        if ($consignmentItems->count() == 0) {
            return redirect()->route('cart');
        }

        // consignments information array
        $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress);
        // validate send_method
        if ($selectedAddress['type'] == 'user') {
            foreach ($consignmentsInformation['consignments'] as &$cns) {
                if ($cns['send_method'] == null) {
                    return redirect()->route('cart');
                }
            }
            unset($cns);
        }

        // get customer club (price and score)
        $customerClub = Setting::where('key', 'customer_club')->first();
        $customerClub = $customerClub != null ? unserialize($customerClub->value) : ['price' => 0, 'score' => 0];
        $customerClubScore = $customerClub['price'] > 0 && $customerClub['score'] > 0 ? round(($consignmentItems->sum('get_full_count_price') / str_replace(',', '', $customerClub['price'])) * str_replace(',', '', $customerClub['score'])) : 0;

        // get payment gateways
        $allPaymentGateways = $this->paymentGatewayList();
        $paymentGatewaysList = Setting::where('key', 'payment_gateway')->first() != null
            ? unserialize(Setting::where('key', 'payment_gateway')->first()->value)
            : [];

        // حذف درگاه‌هایی که دیگر در allPaymentGateways وجود ندارند
        $validIds = array_column($allPaymentGateways, 'id');
        $paymentGatewaysList = array_filter($paymentGatewaysList, function ($pg) use ($validIds) {
            return in_array($pg['id'], $validIds);
        });

        // create payment gateway final result for active pg's
        $paymentGateways = array_filter(array_map(function ($pg) {
            return [
                'id' => $pg['id'],
                'label' => $pg['label'],
                'icon' => array_key_exists('icon', $pg) ? $pg['icon'] : null,
                'status' => $pg['status'],
            ];
        }, $paymentGatewaysList), function ($pg) {
            return $pg['status'];
        });

        // snapppay eligible
        if (in_array('snapppay', array_column($paymentGateways, 'id'))) {
            try {
                // set payment gateway driver configs
                foreach ($paymentGatewaysList as $pg) {
                    foreach ($pg['fields'] as $key => $field) {
                        Config::set('payment.drivers.' . $pg['id'] . '.' . $key, $field['value']);
                    }
                }

                $invoice = new Invoice;
                $invoice->amount($consignmentsInformation['financial']['totalFinalPrice']);

                // select driver and invoice
                $settings = config('payment.drivers.snapppay');
                $snappPayDriver = new SnappPay($invoice, $settings);

                $result = $snappPayDriver->eligible();

                if (isset($result['eligible']) && $result['eligible'] === true) {
                    foreach ($paymentGateways as $key => $gateway) {
                        if ($gateway['id'] === 'snapppay') {
                            $paymentGateways[$key] = [
                                'id' => 'snapppay',
                                'eligible' => $result['eligible'],
                                'label' => $result['title_message'] ?? null,
                                'description' => $result['description'] ?? null,
                                'status' => true,
                            ];
                            break;
                        }
                    }
                } else {
                    // delete snapppay
                    foreach ($paymentGateways as $key => $gateway) {
                        if ($gateway['id'] === 'snapppay') {
                            unset($paymentGateways[$key]);
                            break;
                        }
                    }
                }
            } catch (\Exception $e) {
                // delete snapppay
                foreach ($paymentGateways as $key => $gateway) {
                    if ($gateway['id'] === 'snapppay') {
                        unset($paymentGateways[$key]);
                        break;
                    }
                }
            }
        }

        // add gateways icon
        $iconsMap = [];
        foreach ($allPaymentGateways as $original) {
            if (!empty($original['icon'])) {
                $iconsMap[$original['id']] = $original['icon'];
            }
        }
        foreach ($paymentGateways as &$pg) {
            if (empty($pg['icon']) && isset($iconsMap[$pg['id']])) {
                $pg['icon'] = $iconsMap[$pg['id']];
            }
        }
        unset($pg);

        // vpn warning
        $vpnWarningStatus = false;
        $vpnWarning = Setting::where('key', 'vpn_warning')->get()->first();
        $vpnWarning = $vpnWarning != null ? (bool)$vpnWarning->value : false;
        if ($vpnWarning) {
            try {
                $userIP = $request->ip();
                $data = Cache::remember('ip-' . $userIP, now()->addHours(1), function () use ($userIP) {
                    $response = Http::get('http://ip-api.com/json/' . $userIP);
                    return $response->json();
                });

                if ($data['status'] === 'success' && $data['countryCode'] !== 'IR') {
                    $vpnWarningStatus = true;
                }
            } catch (\Exception $e) {
                Log::error('IP-API Error: ' . $e->getMessage());
            }
        }

        return Inertia::render('Main/Cart/Payment', [
            'selectedAddress' => $selectedAddress,

            'sepratedConsignments' => $consignmentsInformation['consignments'],
            'itemsCount' => count($consignmentItems),
            'financial' => $consignmentsInformation['financial'],
            'discount' => $order->discount != null ? unserialize($order->discount) : null,
            'customerClubScore' => $customerClubScore,

            'paymentGateways' => array_values($paymentGateways),
            'vpn_warning_status' => $vpnWarningStatus
        ])->title(__('messages.title.cart_payment'));
    }

    /**
     * check discount
     */
    public function check_discount(Request $request, CartService $cartService)
    {
        $request->validate([
            'code' => 'required'
        ]);

        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        if ($order == null) {
            return redirect()->route('cart');
        }
        if ($order->address == null) {
            return redirect()->route('cart');
        }

        // get order consignment
        $consignmentItems = [];
        $consignments = [];
        $consignments = $order->consignments()->with('consignmentItems')->get();

        // add consignment items
        $consignmentItems = new Collection();
        foreach ($consignments as $consignment) {
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
        }

        // validate empty cart
        if ($consignmentItems->count() == 0) {
            return redirect()->route('cart');
        }

        // clear prev discount
        $order->update([
            'discount' => null
        ]);

        // selected address
        $selectedAddress = $order->address != null ? unserialize($order->address) : null;
        // consignments information array
        $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress);

        // validate discount code
        $discount = Discount::where('code', $request->code)->where(function ($query) use ($user) {
            $query->where('for', $user->id)->orWhere('for', 'all');
        })->where('expire_at', '>=', Carbon::now()->toDateTimeString())->first();
        if ($discount == null) {
            return redirect()->back()->withErrors([__('messages.discount_code_is_invalid')]);
        }
        if ($discount->users()->where('user_id', $user->id)->count() >= $discount->max_use) {
            return redirect()->back()->withErrors([__('messages.discount_code_has_already_been_used')]);
        }

        // calculate discount amount
        $amount = 0;
        $discountPrice = ($consignmentsInformation['financial']['totalFinalPrice'] * $discount->percent) / 100;
        if ($discountPrice <= $discount->max_amount) {
            $amount = $discountPrice;
        } else {
            $amount = $discount->max_amount;
        }

        // add discount to order
        $discount = [
            'information' => [
                'id' => $discount->id,
                'title' => $discount->title,
                'code' => $discount->code,
                'percent' => $discount->percent,
                'max_amount' => $discount->max_amount,
            ],
            'amount' => $amount,
        ];
        $order->update([
            'discount' => serialize($discount)
        ]);

        return redirect()->back()->with('message', [__('messages.discount_applied')]);
    }

    /**
     * go to payment gateway
     */
    public function gateway(Request $request, CartService $cartService)
    {
        if ($request->method() === 'GET') {
            return redirect()->route('cart');
        }

        /** @var \App\Models\User $user */
        $user = auth()->user();

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        if ($order == null) {
            return redirect()->route('cart');
        }
        if ($order->address == null) {
            return redirect()->route('cart');
        }

        // add order description
        $order->update([
            'description' => $request->exists('description') ? $request->description : null,
        ]);

        // selected address
        $selectedAddress = $order->address != null ? unserialize($order->address) : null;

        // consignments information array
        $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress);

        // payment gateways
        $paymentGateways = Setting::where('key', 'payment_gateway')->first() != null ? unserialize(Setting::where('key', 'payment_gateway')->first()->value) : [];

        // validate existing payment gateways
        if (count($paymentGateways) == 0) {
            return redirect()->back()->withErrors([__('messages.no_any_payment_gateways')]);
        }

        // validate selected payment gateway
        $paymentGatewaysIDStatus = [];
        foreach ($paymentGateways as $pg) {
            $paymentGatewaysIDStatus[$pg['id']] = $pg['status'];
        }
        if (!$request->exists('payment_gateway') || $request->payment_gateway == null || $request->payment_gateway == '')
            return redirect()->back()->withErrors([__('messages.no_any_selected_payment_gateway')]);
        if (!array_key_exists($request->payment_gateway, $paymentGatewaysIDStatus) || !$paymentGatewaysIDStatus[$request->payment_gateway])
            return redirect()->back()->withErrors([__('messages.selected_payment_gateway_not_active')]);

        // set payment gateway driver configs
        foreach ($paymentGateways as $pg) {
            foreach ($pg['fields'] as $key => $field) {
                Config::set('payment.drivers.' . $pg['id'] . '.' . $key, $field['value']);
            }
        }

        // get order consignment
        $consignments = $order->consignments()->with('consignmentItems')->get();

        // add consignment items
        $consignmentItems = new Collection();
        $consignmentItemsForEdit = new Collection();
        foreach ($consignments as $consignment) {
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
            $consignmentItemsForEdit = $consignmentItemsForEdit->merge($consignment->consignmentItems()->get());
        }

        // recharge consignments time for 1 day if less than 10 minutes
        foreach ($consignmentItemsForEdit as $ci) {
            $updatedAt = Carbon::parse($ci->updated_at);
            $expireAt = $updatedAt->copy()->addDays(15);
            if ($expireAt->diffInMinutes(now()) <= 10) {
                $ci->update([
                    'updated_at' => $updatedAt->addDay()
                ]);
            }
        }

        // validate empty cart
        if ($consignmentItems->count() == 0) {
            return redirect()->route('cart');
        }

        // validate send_method
        if ($selectedAddress['type'] == 'user') {
            foreach ($consignmentsInformation['consignments'] as &$cns) {
                if ($cns['send_method'] == null) {
                    return redirect()->route('cart');
                }
            }
            unset($cns);
        }

        // check wallet
        $payablePrice = $consignmentsInformation['financial']['totalFinalPrice'];
        if ($request->payment_gateway != 'snapppay' && $request->payment_gateway != 'digipay' && $request->payment_gateway != 'azki') {
            if ($request->use_wallet) {
                if ($consignmentsInformation['financial']['totalFinalPrice'] - $user->wallet->amount <= 0) {
                    // pay just with wallet
                    $orderUUID = $cartService->registerOrder();
                    if ($orderUUID === false) {
                        return redirect()->route('cart', ['status' => 'error']);
                    } else if ($orderUUID) {
                        // delete affiliate cookies
                        Cookie::queue(Cookie::forget('affiliates'));

                        return redirect()->route('cart.order.status', ['order' => $orderUUID]);
                    }

                    // order registeration error
                    return redirect()->route('cart')->with('cart_error', __('messages.order_registeration_error'));
                }

                $payablePrice = $consignmentsInformation['financial']['totalFinalPrice'] - $user->wallet->amount;
            }
        }

        // add transaction
        $transaction = $user->transactions()->create([
            'amount' => $payablePrice,
            'transaction_id' => $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id'),
            'payment_gateway' => $request->payment_gateway,
            'payment_details' => serialize([
                'type' => 'order'
            ]),
            'description' => __('messages.transaction_user_charge'),
            'type' => 'user_charge',
            'status' => 'waiting_payment'
        ]);

        // create invoice and send to payment gateway
        $invoice = new Invoice;
        $invoice->amount($payablePrice);
        $invoice->detail(['mobile' => $this->convertToInternationalFormat($user->phone)]);
        $invoice->detail(['transaction_id' => $transaction->id]);

        // handle snapppay
        if ($request->payment_gateway === 'snapppay') {
            // set order id
            $snapppayOrderUUID = $this->generateUniqueRandomNumber(10, \App\Models\OrderGroup::class, 'uuid');
            $invoice->uuid($snapppayOrderUUID);

            // items
            $cartItems = [];
            $cartTotalAmount = 0;

            foreach ($consignmentsInformation['consignments'] as $cartData) {
                foreach ($cartData['items'] as $item) {
                    $cartItems[] = [
                        'amount' => $item['price'],
                        'category' => $item['product']['category'],
                        'count' => $item['count'],
                        'id' => $item['id'],
                        'name' => $item['product']['title'],
                        'commissionType' => 100,
                    ];

                    $cartTotalAmount += ($item['price']) * $item['count'];
                }
            }

            $cartList = [
                [
                    'cartId' => 1,
                    'cartItems' => $cartItems,
                    'isShipmentIncluded' => true,
                    'isTaxIncluded' => true,
                    'shippingAmount' => $consignmentsInformation['financial']['sendPrice'],
                    'taxAmount' => 0,
                    'totalAmount' => $cartTotalAmount + $consignmentsInformation['financial']['sendPrice'],
                ]
            ];

            $invoice->detail('cartList', $cartList);
            $invoice->detail('discountAmount', $consignmentsInformation['financial']['totalCouponDiscountPrice']);
            $invoice->detail('externalSourceAmount', 0);
        }

        // handle digipay
        if ($request->payment_gateway === 'digipay') {
            $cartItems = [];
            foreach ($consignmentsInformation['consignments'] as $cartData) {
                foreach ($cartData['items'] as $item) {
                    $cartItems[] = [
                        'productId' => $item['id'],
                        'title' => $item['product']['title'],
                        'price' => $item['price'],
                        'count' => $item['count'],
                    ];
                }
            }

            $invoice->detail([
                'phone' => $user->phone,
                'basketDetailsDto' => $cartItems, // آرایه محصولات
            ]);
        }

        // handle azki
        if ($request->payment_gateway === 'azki') {
            $cartItems = [];
            foreach ($consignmentsInformation['consignments'] as $cartData) {
                foreach ($cartData['items'] as $item) {
                    $cartItems[] = [
                        'name' => $item['product']['title'],
                        'count' => $item['count'],
                        'amount' => $item['price'],
                        'url' => route('main.products.show', $item['product']['slug']),
                    ];
                }
            }

            // add send price
            $cartItems[] = [
                'name' => __('messages.sentence.send_price'),
                'count' => 1,
                'amount' => $consignmentsInformation['financial']['sendPrice'],
                'url' => null,
            ];

            // subtract discount coupon price
            if ($consignmentsInformation['financial']['totalCouponDiscountPrice'] > 0) {
                $cartItems[] = [
                    'name' => __('messages.sentence.deduction_with_discount_code'),
                    'count' => 1,
                    'amount' => $consignmentsInformation['financial']['totalCouponDiscountPrice'] * -1,
                    'url' => null,
                ];
            }

            $invoice->detail([
                'items' => $cartItems, // آرایه محصولات
            ]);
        }

        // handle toman
        if ($request->payment_gateway === 'toman') {
            Config::set('payment.drivers.toman.data', [
                'res_number' => $transaction->uuid,
                'return_to' => route('cart.payment.callback_get'),
                'items' => [
                    [
                        'name' => 'شارژ کیف پول',
                        'price' => $payablePrice * 10,
                        'quantity' => 1
                    ]
                ]
            ]);
        }

        // handle payping
        if ($request->payment_gateway === 'payping') {
            //
        }

        // You can specify callbackUrl
        $pay = null;
        try {
            $pay = Payment::via($request->payment_gateway)->callbackUrl(route('cart.payment.callback_get'))->purchase(
                $invoice,
                function ($driver, $transactionId) use ($transaction) {
                    $transaction->update([
                        'gateway_transaction_id' => $transactionId,
                    ]);
                }
            );
        } catch (PurchaseFailedException $e) {
            return redirect()->back()->withErrors([$e->getMessage()]);
        }

        if ($pay != null) {
            $redirectionForm = $pay->pay();
            $data = [
                'action' => $redirectionForm->getAction(),
                'method' => $redirectionForm->getMethod(),
                'inputs' => $redirectionForm->getInputs(),
            ];

            return Inertia::render('Main/Cart/PaymentRedirect', $data);
        }

        return redirect()->back()->withErrors([__('messages.error_in_payment_gateway')]);
    }

    /**
     * payment callback get
     */
    public function callback_get(Request $request, CartService $cartService)
    {
        // validate requests
        $_POST = array_merge($request->query->all(), $request->post());

        /** @var \App\Models\User $user */
        $user = auth()->user();

        // user awaiting transactions
        $userAwaitingTransactions = $user->transactions()->where('type', 'user_charge')->where('status', 'waiting_payment')->orderBy('created_at', 'desc')->get();

        // payment gateways
        $paymentGatewaySetting = Setting::where('key', 'payment_gateway')->first();
        $paymentGateways = $paymentGatewaySetting ? unserialize($paymentGatewaySetting->value) : [];

        // set payment gateway driver configs
        foreach ($paymentGateways as $pg) {
            foreach ($pg['fields'] as $key => $field) {
                Config::set('payment.drivers.' . $pg['id'] . '.' . $key, $field['value']);
            }
        }

        // verify transactions
        $transactionStatus = false;
        $verifiedTransaction = null;
        $snapppayTransactionID = null;
        foreach ($userAwaitingTransactions as $transaction) {
            if ($transaction->gateway_transaction_id == null || $transaction->payment_gateway == null || $transaction->status === 'accepted') {
                // delete transaction
                $transaction->delete();
                continue;
            }

            try {
                // shetabit payment instance
                $paymentInstance = new \Shetabit\Multipay\Payment(config('payment'));

                if ($transaction->payment_gateway == 'asanpardakht') {
                    $receipt = $paymentInstance->via($transaction->payment_gateway)->amount($transaction->amount)->transactionId($request->invoice)->verify();
                } else if ($transaction->payment_gateway == 'snapppay') {
                    $driver = $paymentInstance->via('snapppay')
                        ->amount($transaction->amount)
                        ->transactionId($transaction->gateway_transaction_id);

                    $invoice = new Invoice;
                    $invoice->transactionId($transaction->gateway_transaction_id);
                    $snapppayDriver = new SnappPay($invoice, config('payment.drivers.snapppay'));

                    $verified = false;
                    try {
                        // verify
                        $receipt = $driver->verify();
                        $verified = true;
                    } catch (\Exception $e) {
                        $resultStatus = $snapppayDriver->status();

                        if ($resultStatus['status'] == 'VERIFY') {
                            // verified
                            $verified = true;
                        } elseif ($resultStatus['status'] == 'PENDING') {
                            // verify again
                            sleep(2);
                            $receipt = $driver->verify();
                            $verified = true;
                        } else {
                            throw new \Exception('Error verifying SnappPay transaction');
                        }
                    }

                    if ($verified) {
                        try {
                            $snapppayDriver->settle();
                        } catch (\Exception $e) {
                            $resultStatus = $snapppayDriver->status();

                            if ($resultStatus['status'] == 'VERIFY') {
                                // settle again
                                $snapppayDriver->settle();
                            } elseif ($resultStatus['status'] == 'SETTLE') {
                                // no problem
                            } else {
                                throw new \Exception('Error settling SnappPay transaction');
                            }
                        }
                    }

                    $snapppayTransactionID = $receipt->getReferenceId();
                } else if ($transaction->payment_gateway === 'digipay') {
                    $driver = $paymentInstance->via('digipay')
                        ->amount($transaction->amount)
                        ->transactionId($transaction->gateway_transaction_id);
                    $receipt = $driver->verify();
                } else if ($transaction->payment_gateway == 'kipaa') {
                    // this driver is for test and not active
                    $paymentToken = $request->input('payment_token', $transaction->gateway_transaction_id);
                    $receiptNumber = $request->input('reciept_number');
                    if (!$receiptNumber) {
                        Log::error('Kipaa callback missing reciept_number', $request->all());
                        throw new \Exception('Kipaa Unknown Error');
                    }
                    $receipt = $paymentInstance->via($transaction->payment_gateway)->amount($transaction->amount)->transactionId($paymentToken)->verify();
                } else if ($transaction->payment_gateway == 'zibal') {
                    $receipt = $paymentInstance->via($transaction->payment_gateway)->amount($transaction->amount)->transactionId($request->trackId)->verify();
                } else if ($transaction->payment_gateway == 'sep') {
                    $receipt = $paymentInstance->via($transaction->payment_gateway)->amount($transaction->amount)->transactionId($request->ResNum)->verify();
                } else {
                    $receipt = $paymentInstance->via($transaction->payment_gateway)->amount($transaction->amount)->transactionId($transaction->gateway_transaction_id)->verify();
                }

                // charge wallet
                DB::transaction(function () use ($transaction, $user, $receipt) {
                    // lock user wallet for update
                    $wallet = $user->wallet()->lockForUpdate()->first();

                    // update verified transaction status to accepted
                    $transaction->update([
                        'gateway_reference_id' => $receipt->getReferenceId(),
                        'return_response' => json_encode($_POST),
                        'status' => 'accepted',
                    ]);

                    // charge wallet
                    $wallet->update([
                        'amount' => $wallet->amount + $transaction->amount,
                    ]);
                });

                // final variables and actions
                $transactionStatus = true;
                $verifiedTransaction = $transaction;
                unset($paymentInstance, $receipt);
            } catch (Exception $exception) {
                // delete invalid transaction
                $transaction->delete();
            }
        }

        // check transaction status for redirect to result page
        if ($transactionStatus && $verifiedTransaction !== null && $verifiedTransaction->payment_details !== null) {
            $paymentDetails = unserialize($verifiedTransaction->payment_details);
            if ($paymentDetails['type'] == 'order') {
                // pay with wallet
                $orderUUID = $cartService->registerOrder($verifiedTransaction->transaction_id, $snapppayTransactionID);
                if ($orderUUID === false) {
                    return redirect()->route('cart', ['status' => 'error']);
                } else if ($orderUUID) {
                    // delete affiliate cookies
                    Cookie::queue(Cookie::forget('affiliates'));

                    return redirect()->route('cart.order.status', ['order' => $orderUUID]);
                }

                // order registeration error
                return redirect()->route('cart', ['status' => 'error']);
            } else if ($paymentDetails['type'] == 'prefactor') {
                // pre factor payment
                $orderUUID = $cartService->registerPrefactorOrder($paymentDetails['uuid'], $verifiedTransaction->transaction_id, $snapppayTransactionID);
                if ($orderUUID === false) {
                    return redirect()->route('prefactor.shiping', ['uuid' => $paymentDetails['uuid'], 'status' => 'error']);
                } else if ($orderUUID) {
                    // delete affiliate cookies
                    Cookie::queue(Cookie::forget('affiliates'));

                    return redirect()->route('cart.order.status', ['order' => $orderUUID, 'prefactor' => $paymentDetails['uuid']]);
                }

                // order registeration error
                return redirect()->route('prefactor.shiping', ['uuid' => $paymentDetails['uuid'], 'status' => 'error']);
            } else if ($paymentDetails['type'] == 'wallet_charge') {
                // notification
                $user->notify(new WalletCharge($verifiedTransaction));

                return redirect()->route('user.wallet.index', ['status' => 'success']);
            }
        }

        // order transaction error
        return redirect()->route('cart.transaction_error');
    }

    /**
     * payment callback post
     */
    public function callback_post(Request $request, CartService $cartService)
    {
        $params = http_build_query($request->all());
        $url = route('cart.payment.callback_get') . '?' . $params;
        return redirect()->to($url);
    }

    /**
     * payment status page
     */
    public function status(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        // check for existing order id
        if (!$request->exists('order') || $request->order == null || $request->order == "") {
            return redirect()->route('index');
        }

        /** @var \App\Models\Order $order */
        $orderGroup = $user->orderGroups()->where('uuid', $request->order)->first();
        if ($orderGroup == null) {
            return redirect()->route('index');
        }

        // check order group status
        if ($orderGroup->status != 'processing') {
            return redirect()->route('index');
        }

        return Inertia::render('Main/Cart/PaymentSuccessful', [
            'order' => [
                'id' => $orderGroup->id,
                'uuid' => $orderGroup->uuid,
                'jalali_created_at' => $orderGroup->jalali_created_at,
            ],
            'prefactor' => $request->exists('prefactor') && $request->prefactor != null ? $request->prefactor : null
        ])->title(__('messages.title.payment_factor'));
    }

    /**
     * transaction error
     */
    public function transaction_error()
    {
        return Inertia::render('Main/Cart/PaymentError', [
            'date' => Jalalian::forge(Carbon::now())->format('H:i - Y/m/d'),
        ])->title(__('messages.title.transaction_error'));
    }

    /**
     * update address
     */
    public function store_address(AddressRequest $request)
    {
        /** user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        /** postal code validation */
        if (strlen(\request('postal_code')) != 10) {
            return redirect()->back()->withErrors([__('messages.postal_code_wrong')]);
        }

        /** map longitude & latitude */
        $map = [];
        if ($request->exists('map') && is_array($request->map)) {
            $map = $request->map;
        }

        /** check active status */
        $active = false;
        if ($user->addresses()->count() == 0) {
            $active = true;
        }

        $address = $user->addresses()->create([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'state' => $request->state,
            'city' => $request->city,
            'phone' => $request->phone,
            'postal_code' => $request->postal_code,
            'full_address' => $request->full_address,
            'map' => serialize($map),
            'active' => $active
        ]);

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        if ($order != null) {
            if ($address != null) {
                $address = $address->toArray();
                unset($address['user']);
                unset($address['safe']);
                $order->update([
                    'address' => serialize([
                        'type' => 'user',
                        'address' => $address
                    ]),
                    'send_method' => null,
                ]);
            }
        }

        return redirect()->route('cart.shiping')->with('message', [__('messages.address_created')]);
    }

    /**
     * update address
     */
    public function update_address(AddressRequest $request, Address $address)
    {
        /** user
         * @var App\Models\User $user
         */
        $user = auth()->user();

        /** check user validation */
        if ($address->user_id != $user->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** postal code validation */
        if (strlen(\request('postal_code')) != 10) {
            return redirect()->back()->withErrors([__('messages.postal_code_wrong')]);
        }

        /** map longitude & latitude */
        $map = [];
        if ($request->exists('map') && is_array($request->map)) {
            $map = $request->map;
        }

        $address->update([
            'first_name' => $request->first_name,
            'last_name' => $request->last_name,
            'state' => $request->state,
            'city' => $request->city,
            'phone' => $request->phone,
            'postal_code' => $request->postal_code,
            'full_address' => $request->full_address,
            'map' => serialize($map)
        ]);

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        $selectedAddress = $order != null && $order->address != null ? unserialize($order->address) : null;
        if ($selectedAddress != null && $address->id == $selectedAddress['address']['id']) {
            $newAddress = $user->addresses()->where('id', $address->id)->first();
            if ($newAddress != null) {
                $newAddress = $newAddress->toArray();
                unset($newAddress['user']);
                unset($newAddress['safe']);
                $order->update([
                    'address' => serialize([
                        'type' => 'user',
                        'address' => $newAddress
                    ]),
                    'send_method' => null,
                ]);
            } else {
                $order->update([
                    'address' => null,
                    'send_method' => null,
                ]);
            }
        }

        return redirect()->route('cart.shiping')->with('message', [__('messages.address_edited')]);
    }

    /**
     * delete address
     */
    public function delete_address(Address $address)
    {
        /** user
         * @var App\Models\User $user
         */
        $user = auth()->user();

        /** check user validation */
        if ($address->user_id != $user->id) {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        $address->delete();

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();
        $order->update([
            'address' => null,
            'send_method' => null,
        ]);

        return redirect()->route('cart.shiping')->with('message', [__('messages.address_deleted')]);
    }
}
