<?php

namespace App\Http\Controllers\Api\v1\Products;

use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Models\File;
use App\Models\Product;
use App\Models\ProductCategory;
use App\Models\User;
use App\Services\Admin\Products\ProductService;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\Drivers\Gd\Encoders\WebpEncoder;
use Intervention\Image\ImageManager;

class ProductsController extends Controller
{
    use UUIDHelper;

    /**
     * products list
     */
    public function index(Request $request)
    {
        $perPage = $request->exists('per_page') && is_int((int)$request->per_page) && (int)$request->per_page > 0 ? (int)$request->per_page : 100;

        $products = Product::with([
            'brand',
            'productCategory',
            'inventories.store',
            'inventories.storage'
        ])->where('status', '!=', 'deep_deleted')->where('status', '!=', 'deleted')->when($request->input('status'), function ($query, $status) {
            $query->where('status', $status);
        })->orderByDesc('created_at')->paginate($perPage)->through(fn($product) => [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'user_id' => $product->user_id,
            'edited_by' => $product->edited_by,
            'brand' => $product->brand ? [
                'id' => $product->brand->id,
                'title' => $product->brand->title,
                'slug' => $product->brand->slug,
                'logo' => $product->brand->get_logo,
                'status' => $product->brand->status,
                'created_at' => $product->brand->created_at,
                'updated_at' => $product->brand->updated_at,
            ] : null,
            'category' => $product->productCategory ? [
                'id' => $product->productCategory->id,
                'title' => $product->productCategory->title,
                'slug' => $product->productCategory->slug,
                'image' => $product->productCategory->get_image,
                'status' => $product->productCategory->status,
                'created_at' => $product->productCategory->created_at,
                'updated_at' => $product->productCategory->updated_at,
            ] : null,
            'title' => $product->title,
            'title_en' => $product->title_en,
            'content' => $product->content,
            'warnings' => $product->warnings,
            'tabs' => $product->get_tabs,
            'guide' => $product->guide,
            'images' => $product->get_all_images,
            'videos' => $product->get_videos,
            'model_3d' => $product->model_3d,
            'attrs' => $product->get_attrs,
            'specifications' => $product->get_specifications,
            'seo_title' => $product->seo_title,
            'seo_description' => $product->seo_description,
            'seo_keywords' => json_decode($product->seo_keywords),
            'seo_canonical' => $product->seo_canonical,
            'view_count' => $product->view_count,
            'in_stock_status' => $product->in_stock_status,
            'inquiry_options' => $product->get_inquiry_options,
            'variables' => $product->raw_price_model,
            'inventories' => $product->inventories()->where('status', 'publish')->get()->map(fn($inventory) => [
                ...[
                    'id' => $inventory->id,
                    'uuid' => $inventory->uuid,
                    'store' => $inventory->store ? [
                        'id' => $inventory->store->id,
                        'uuid' => $inventory->store->uuid,
                        'user_id' => $inventory->store->user_id,
                        'name' => $inventory->store->name,
                        'logo' => $inventory->store->get_logo,
                        'type' => $inventory->store->type,
                        'direct_post' => $inventory->store->direct_post,
                        'meta' => $inventory->store->get_meta,
                        'status' => $inventory->store->status,
                        'created_at' => $inventory->store->created_at,
                        'updated_at' => $inventory->store->updated_at,
                    ] : null,
                    'storage' => $inventory->storage ? [
                        'id' => $inventory->storage->id,
                        'uuid' => $inventory->storage->uuid,
                        'name' => $inventory->storage->name,
                        'address' => $inventory->storage->address,
                        'product_count' => $inventory->storage->product_count,
                        'created_at' => $inventory->storage->created_at,
                        'updated_at' => $inventory->storage->updated_at,
                    ] : null,
                    'props' => $inventory->raw_props,
                    'price' => $inventory->price,
                    'discount_price' => $inventory->discount_price,
                    'discount_expire' => $inventory->discount_expire,
                    'discount_percent' => $inventory->get_discount_percent,
                    'count' => $inventory->count,
                    'min_sale' => $inventory->min_sale,
                    'max_sale' => $inventory->max_sale,
                    'price_changes' => $inventory->get_price_changes_chart,
                    'status' => $inventory->status,
                    'created_at' => $inventory->created_at,
                    'updated_at' => $inventory->updated_at,
                ]
            ]),
            'commission' => $product->commission,
            'is_vip' => $product->is_vip,
            'comment_status' => $product->comment_status,
            'question_status' => $product->question_status,
            'status' => $product->status,
            'created_at' => $product->created_at,
            'updated_at' => $product->updated_at,
        ]);

        return response()->json([
            'success' => true,
            'response' => $products,
            'code' => 200,
        ]);
    }

    /**
     * single product information
     */
    public function show(Request $request, $uuid)
    {
        $product = Product::where('uuid', $uuid)->first();

        if ($product == null || $product->status == 'deleted' || $product->status == 'deep_deleted') {
            return response()->json([
                'success' => false,
                'message' => __('messages.product_not_found'),
                'code' => 404,
            ]);
        }

        $product = [
            'id' => $product->id,
            'uuid' => $product->uuid,
            'user_id' => $product->user_id,
            'edited_by' => $product->edited_by,
            'brand' => $product->brand ? [
                'id' => $product->brand->id,
                'title' => $product->brand->title,
                'slug' => $product->brand->slug,
                'logo' => $product->brand->get_logo,
                'status' => $product->brand->status,
                'created_at' => $product->brand->created_at,
                'updated_at' => $product->brand->updated_at,
            ] : null,
            'category' => $product->productCategory ? [
                'id' => $product->productCategory->id,
                'title' => $product->productCategory->title,
                'slug' => $product->productCategory->slug,
                'image' => $product->productCategory->get_image,
                'status' => $product->productCategory->status,
                'created_at' => $product->productCategory->created_at,
                'updated_at' => $product->productCategory->updated_at,
            ] : null,
            'title' => $product->title,
            'title_en' => $product->title_en,
            'content' => $product->content,
            'warnings' => $product->warnings,
            'tabs' => $product->get_tabs,
            'guide' => $product->guide,
            'images' => $product->get_all_images,
            'videos' => $product->get_videos,
            'model_3d' => $product->model_3d,
            'attrs' => $product->get_attrs,
            'specifications' => $product->get_specifications,
            'seo_title' => $product->seo_title,
            'seo_description' => $product->seo_description,
            'seo_keywords' => json_decode($product->seo_keywords),
            'seo_canonical' => $product->seo_canonical,
            'view_count' => $product->view_count,
            'in_stock_status' => $product->in_stock_status,
            'inquiry_options' => $product->inquiry_options,
            'variables' => $product->price_model,
            'commission' => $product->commission,
            'is_vip' => $product->is_vip,
            'comment_status' => $product->comment_status,
            'question_status' => $product->question_status,
            'status' => $product->status,
            'created_at' => $product->created_at,
            'updated_at' => $product->updated_at,
            'inventories' => $product->inventories()->where('status', 'publish')->get()->map(fn($inventory) => [
                ...[
                    'id' => $inventory->id,
                    'uuid' => $inventory->uuid,
                    'store' => $inventory->store ? [
                        'id' => $inventory->store->id,
                        'uuid' => $inventory->store->uuid,
                        'user_id' => $inventory->store->user_id,
                        'name' => $inventory->store->name,
                        'logo' => $inventory->store->get_logo,
                        'type' => $inventory->store->type,
                        'direct_post' => $inventory->store->direct_post,
                        'meta' => $inventory->store->get_meta,
                        'status' => $inventory->store->status,
                        'created_at' => $inventory->store->created_at,
                        'updated_at' => $inventory->store->updated_at,
                    ] : null,
                    'storage' => $inventory->storage ? [
                        'id' => $inventory->storage->id,
                        'uuid' => $inventory->storage->uuid,
                        'name' => $inventory->storage->name,
                        'address' => $inventory->storage->address,
                        'product_count' => $inventory->storage->product_count,
                        'created_at' => $inventory->storage->created_at,
                        'updated_at' => $inventory->storage->updated_at,
                    ] : null,
                    'props' => $inventory->raw_props,
                    'price' => $inventory->price,
                    'discount_price' => $inventory->discount_price,
                    'discount_expire' => $inventory->discount_expire,
                    'discount_percent' => $inventory->get_discount_percent,
                    'count' => $inventory->count,
                    'min_sale' => $inventory->min_sale,
                    'max_sale' => $inventory->max_sale,
                    'price_changes' => $inventory->get_price_changes_chart,
                    'status' => $inventory->status,
                    'created_at' => $inventory->created_at,
                    'updated_at' => $inventory->updated_at,
                ]
            ]),
        ];

        return response()->json([
            'success' => true,
            'response' => $product,
            'code' => 200,
        ]);
    }

    /**
     * create product
     */
    public function create(Request $request, ProductService $productService)
    {
        $request->validate([
            'user_id' => 'required',
            'uuid' => ['nullable', 'regex:/^[A-Za-z0-9_-]+$/'],
            'title' => ['required', 'string', 'max:255'],
            'title_en' => ['max:255'],
            'slug' => ['max:255'],
            'category_id' => 'required',
            'tabs' => ['nullable', 'array'],
            'attrs' => ['nullable', 'array'],
            'price_model' => ['array'],
            'commission' => ['int'],
            'status' => 'required',
        ]);

        /** user */
        $user = User::find($request->user_id);
        if ($user == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_found'),
                'code' => 404,
            ]);
        } else if ($user->type != 'owner' && $user->type != 'admin') {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_admin'),
                'code' => 403,
            ]);
        }

        /** validate category */
        if (ProductCategory::find($request->category_id) == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.category_not_found'),
                'code' => 400,
            ]);
        }

        /** validate inquiry options */
        if (!$request->exists('inquiry_options') || $request->inquiry_options == null || !is_array($request->inquiry_options)) {
            $inquiryOptions = [
                'landline_phone' => null,
                'cellular_phone' => null,
            ];
        } else {
            $inquiryOptions = $request->inquiry_options;
        }

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null) {
            $productWithSlug = Product::where('slug', $request->slug)->first();
            if ($productWithSlug != null) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.slug_is_exist'),
                    'code' => 400,
                ]);
            }
        }

        /** validate duplicated uuid */
        if ($request->exists('uuid') && $request->uuid != null) {
            $productWithSlug = Product::where('uuid', $request->uuid)->first();
            if ($productWithSlug != null) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.product_with_this_uuid_is_exist'),
                    'code' => 400,
                ]);
            }
        }

        /** validate in stock status when is false */
        if ($request->exists('in_stock_status') && is_bool($request->in_stock_status) && !$request->in_stock_status) {
            if ($inquiryOptions['landline_phone'] == null || $inquiryOptions['cellular_phone'] == null) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.landline_phone_or_cellular_phone_is_null'),
                    'code' => 400,
                ]);
            }
        }

        /** validate seller commission */
        $commission = 0;
        if ($request->exists('commission') && $request->commission != null) {
            if ($request->commission > 100 || $request->commission < 0) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.seller_commission_incorrect'),
                    'code' => 400,
                ]);
            }
            $commission = $request->commission;
        }

        /** images */
        $images = null;
        if ($request->exists('images') && !empty($request->images) && is_array($request->images)) {
            foreach ($request->images as $image) {
                $imageUrl = $image['src'] ?? null;
                $imageAlt = $image['alt'] ?? null;
                if ($imageUrl) {
                    $fileName = basename(parse_url($imageUrl, PHP_URL_PATH));
                    $file = File::where('name', $fileName)->first();
                    if ($file) {
                        $images[] = $file->id;
                    } else {
                        $uploadedFile = $this->uploadFromUrl($imageUrl, $imageAlt, $user);
                        if ($uploadedFile && isset($uploadedFile['id'])) {
                            $images[] = $uploadedFile['id'];
                        }
                    }
                }
            }
        }

        $product = $user->products()->create([
            'edited_by' => $user->id,
            'brand_id' => $productService->initBrand($user, $request),
            'category_id' => $request->category_id,
            'uuid' => $request->exists('uuid') && $request->uuid != null && $request->uuid != '' ? $request->uuid : $this->generateUniqueRandomNumber(8, \App\Models\Product::class, 'uuid', 'p-'),
            'title' => $request->title,
            'title_en' => $request->exists('title_en') ? $request->title_en : null,
            'slug' => $request->exists('slug') && $request->slug != null ? $request->slug : null,
            'content' => $request->exists('content') ? $request->content : '',
            'warnings' => $request->exists('warnings') ? $request->warnings : null,
            'guide' => $request->exists('guide') ? $request->guide : null,
            'tabs' => $request->exists('tabs') && is_array($request->tabs) ? serialize($request->tabs) : null,

            'images' => $images != null ? serialize($images) : null,
            'videos' => $request->exists('videos') && is_array($request->videos) ? serialize($request->videos) : null,
            'model_3d' => $request->exists('model_3d') ? $request->model_3d : null,

            'attrs' => $request->exists('attrs') && is_array($request->attrs) ? serialize($request->attrs) : null,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : null,

            'in_stock_status' => $request->exists('in_stock_status') && is_bool($request->in_stock_status) ? $request->in_stock_status : true,
            'inquiry_options' => serialize($inquiryOptions),

            'commission' => $commission,

            'is_vip' => $request->exists('is_vip') && is_bool($request->is_vip) ? $request->is_vip : false,
            'comment_status' => $request->exists('comment_status') && is_bool($request->comment_status) ? $request->comment_status : true,
            'question_status' => $request->exists('question_status') && is_bool($request->question_status) ? $request->question_status : true,

            'reason' => 'create',
            'status' => $request->has('status') ? $request->status : 'publish',
        ]);

        /** add price model */
        $productService->addPriceModel($product, $request);

        return response()->json([
            'success' => true,
            'message' => __('messages.product_created'),
            'product_uuid' => $product->uuid,
            'code' => 200,
        ]);
    }

    /**
     * update product
     */
    public function update(Request $request, $uuid, ProductService $productService)
    {
        $request->validate([
            'user_id' => 'required',
            'title' => ['nullable', 'string', 'max:255'],
            'title_en' => ['nullable', 'max:255'],
            'slug' => ['nullable', 'max:255'],
            'tabs' => ['nullable', 'array'],
            'attrs' => ['nullable', 'array'],
            'price_model' => ['nullable', 'array'],
            'commission' => ['nullable', 'int'],
        ]);

        /** product */
        $product = Product::where('uuid', $uuid)->first();
        if ($product == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.product_not_found'),
                'code' => 404,
            ]);
        }

        /** user */
        $user = User::find($request->user_id);
        if ($user == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_found'),
                'code' => 404,
            ]);
        } else if ($user->type != 'owner' && $user->type != 'admin') {
            return response()->json([
                'success' => false,
                'response' => __('messages.user_not_admin'),
                'code' => 403,
            ]);
        }

        /** validate category */
        if (ProductCategory::find($request->category_id) == null) {
            return response()->json([
                'success' => false,
                'response' => __('messages.category_not_found'),
                'code' => 400,
            ]);
        }

        /** validate inquiry options */
        if (!$request->exists('inquiry_options') || $request->inquiry_options == null || !is_array($request->inquiry_options)) {
            $inquiryOptions = $product->inquiry_options;
        } else {
            $inquiryOptions = $request->inquiry_options;
        }

        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null && $request->slug != $product->slug) {
            $productWithSlug = Product::where('slug', $request->slug)->first();
            if ($productWithSlug != null) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.slug_is_exist'),
                    'code' => 400,
                ]);
            }
        }

        /** validate in stock status when is false */
        if ($request->exists('in_stock_status') && is_bool($request->in_stock_status) && !$request->in_stock_status) {
            if ($inquiryOptions['landline_phone'] == null || $inquiryOptions['cellular_phone'] == null) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.landline_phone_or_cellular_phone_is_null'),
                    'code' => 400,
                ]);
            }
        }

        /** validate seller commission */
        $commission = $product->commission;
        if ($request->exists('commission') && $request->commission != null) {
            if ($request->commission > 100 || $request->commission < 0) {
                return response()->json([
                    'success' => false,
                    'response' => __('messages.seller_commission_incorrect'),
                    'code' => 400,
                ]);
            }
            $commission = $request->commission;
        }

        /** images */
        $images = null;
        if ($request->exists('images') && !empty($request->images) && is_array($request->images)) {
            foreach ($request->images as $image) {
                $imageUrl = $image['src'] ?? null;
                $imageAlt = $image['alt'] ?? null;
                if ($imageUrl) {
                    $fileName = basename(parse_url($imageUrl, PHP_URL_PATH));
                    $file = File::where('name', $fileName)->first();
                    if ($file) {
                        $images[] = $file->id;
                    } else {
                        $uploadedFile = $this->uploadFromUrl($imageUrl, $imageAlt, $user);
                        if ($uploadedFile && isset($uploadedFile['id'])) {
                            $images[] = $uploadedFile['id'];
                        }
                    }
                }
            }
        }

        $product->update([
            'edited_by' => $user->id,
            'brand_id' => $productService->initBrand($user, $request),
            'category_id' => $request->has('category_id') ? $request->category_id : $product->category_id,
            'title' => $request->has('title') ? $request->title : $product->title,
            'title_en' => $request->has('title_en') ? $request->title_en : $product->title_en,
            'slug' => $request->has('slug') && $request->slug != null ? $request->slug : $product->slug,
            'content' => $request->has('content') ? $request->content : $product->content,
            'warnings' => $request->has('warnings') ? $request->warnings : $product->warnings,
            'guide' => $request->has('guide') ? $request->guide : $product->guide,
            'tabs' => $request->has('tabs') && is_array($request->tabs) ? serialize($request->tabs) : $product->tabs,

            'images' => $images != null ? serialize($images) : $product->images,
            'videos' => $request->has('videos') && is_array($request->videos) ? serialize($request->videos) : $product->videos,
            'model_3d' => $request->has('model_3d') ? $request->model_3d : $product->model_3d,

            'attrs' => $request->has('attrs') && is_array($request->attrs) ? serialize($request->attrs) : $product->attrs,

            'seo_title' => $request->has('seo_title') ? $request->seo_title : $product->seo_title,
            'seo_description' => $request->has('seo_description') ? $request->seo_description : $product->seo_description,
            'seo_canonical' => $request->has('seo_canonical') ? urldecode($request->seo_canonical) : $product->seo_canonical,

            'in_stock_status' => $request->has('in_stock_status') && is_bool($request->in_stock_status) ? $request->in_stock_status : $product->in_stock_status,
            'inquiry_options' => serialize($inquiryOptions),

            'commission' => $commission,

            'is_vip' => $request->has('is_vip') && is_bool($request->is_vip) ? $request->is_vip : $product->is_vip,
            'comment_status' => $request->has('comment_status') && is_bool($request->comment_status) ? $request->comment_status : $product->comment_status,
            'question_status' => $request->has('question_status') && is_bool($request->question_status) ? $request->question_status : $product->question_status,

            'reason' => 'update',
            'status' => $request->has('status') ? $request->status : $product->status,
        ]);

        /** add price model */
        if ($product->inventories()->count() == 0) {
            if ($request->exists('price_model') && is_array($request->price_model)) {
                $pmID = 0;
                $product->productProps()->delete();
                foreach ($request->price_model as $pm) {
                    if ($pm['active'] && $pm['name'] != null) {
                        $pmModel = $product->productProps()->create([
                            'child' => $pmID,
                            'name' => $pm['name'],
                            'type' => $pm['type'],
                        ]);

                        $pmID = $pmModel->id;
                    }
                }
            }
        }

        return response()->json([
            'success' => true,
            'message' => __('messages.product_updated'),
            'code' => 200,
        ]);
    }

    /**
     * delete product
     */
    public function delete(Request $request, $uuid)
    {
        $product = Product::where('uuid', $uuid)->first();

        if ($product == null || $product->status == 'deleted' || $product->status == 'deep_deleted') {
            return response()->json([
                'success' => false,
                'message' => __('messages.product_not_found'),
                'code' => 404,
            ]);
        }

        // delete product from users cart
        $product->consignmentItems()->whereHas('consignment', function ($query) {
            $query->whereHas('order', function ($query) {
                $query->where('status', '!=', 'prefactor')->whereDoesntHave('orderGroup');
            });
        })->delete();

        $product->update([
            'status' => 'deleted'
        ]);

        return response()->json([
            'success' => true,
            'message' => __('messages.product_deleted'),
            'code' => 200,
        ]);
    }

    /**
     * upload image from url
     */
    private function uploadFromUrl($url, $alt, $user)
    {
        try {
            // دانلود فایل با timeout
            $response = Http::timeout(2)->get($url);

            $imageContent = $response->body();

            // ذخیره موقت
            $tmpFile = tempnam(sys_get_temp_dir(), 'img');
            file_put_contents($tmpFile, $imageContent);

            // گرفتن پسوند اصلی
            $fileInfo = pathinfo(parse_url($url, PHP_URL_PATH));
            $originalExtension = strtolower($fileInfo['extension'] ?? 'jpg');
            $fileType = match ($originalExtension) {
                'jpeg', 'jpg', 'png', 'webp', 'gif', 'svg', 'avif' => 'image',
                default => null,
            };

            if ($fileType !== 'image') {
                return null;
            }

            // پردازش تصویر
            $manager = ImageManager::withDriver(new Driver());
            $imageTemp = $manager->read($tmpFile);
            $imageResolution = $imageTemp->width() . '×' . $imageTemp->height();

            // تبدیل به webp و فشرده‌سازی
            $optimizationPercent = 30;
            $finalImageTemp = $imageTemp->encode(new WebpEncoder($optimizationPercent))->__toString();
            $fileFormat = 'webp';

            // مسیر ذخیره‌سازی
            $fileName = 'file.' . time() . '.' . rand(10000, 99999) . '.' . $fileFormat;
            $fileSaveAddressNoName = 'library/' . date('Y-m-d') . '/' . $user->id;
            $fileSaveAddressImage = $fileSaveAddressNoName . '/' . $fileName;

            $uploadResult = Storage::disk('media')->put($fileSaveAddressImage, $finalImageTemp, 'public');

            if ($uploadResult) {
                return $user->files()->create([
                    'name'        => $fileName,
                    'description' => $alt,
                    'url'         => '/media/' . $fileSaveAddressImage,
                    'extension'   => $fileFormat,
                    'size'        => strlen($finalImageTemp) / 1000,
                    'resolution'  => $imageResolution,
                    'type'        => $fileType,
                ])->toArray();
            }

            return null;
        } catch (\Exception $e) {
            return null;
        }
    }
}
