<?php

namespace App\Http\Controllers\Admin\Users;

use App\Exports\UsersExport;
use App\Helpers\DropdownListHelper;
use App\Models\User;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\updateUserScoreRequest;
use App\Http\Requests\Main\AddToCartRequest;
use App\Http\Requests\Main\ChangeCartCountRequest;
use App\Http\Requests\Main\DeleteFromCartRequest;
use App\Models\ConsignmentItem;
use App\Models\Inventory;
use App\Models\Product;
use App\Models\Role;
use App\Notifications\Raw;
use App\Rules\NationalNumber;
use App\Rules\Phone;
use App\Rules\Username;
use App\Services\Admin\Account\ProfileService;
use App\Services\Admin\Users\UsersService;
use App\Services\Cart\CartService;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Maatwebsite\Excel\Facades\Excel;

class UsersController extends Controller
{
    use DropdownListHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, UsersService $usersService)
    {
        return Inertia::render('Admin/Users/List', [
            'users' => $usersService->getUsers($request),
            'roles' => Role::get(['id', 'label'])
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Request $request, User $user, UsersService $usersService)
    {
        // only owner can access to admin and owner users
        if (auth()->user()->type != "owner" && ($user->type == "admin" || $user->type == "owner"))
            return redirect()->back()->withErrors([__('messages.admin_not_editable')]);

        return Inertia::render('Admin/Users/Show', [
            'user' => [
                'id' => $user->id,
                'phone' => $user->phone,
                'avatar' => $user->avatar,
                'get_avatar' => $user->get_avatar,
                'name' => $user->name,
                'get_name' => $user->get_name,
                'username' => $user->username,
                'email' => $user->email,
                'email_verified_at' => $user->email_verified_at,
                'national_number' => $user->national_number,
                'jalali_birthday' => $user->jalali_birthday,
                'gender' => $user->gender,
                'bio' => $user->bio,
                'get_social' => $user->get_social,
                'get_level' => $user->get_level,
                'seller' => $user->seller,
                'type' => $user->type,
                'get_score' => $user->get_score,
                'role' => $user->role != null ? $user->role : null,
                'status' => $user->status,

                'store' => $user->store != null ? [
                    'id' => $user->store->id,
                    'uuid' => $user->store->uuid,
                    'name' => $user->store->name,
                ] : null,
                'wallet' => [
                    'id' => $user->wallet->id,
                    'get_formatted_amount' => $user->wallet->get_formatted_amount,
                ],
            ],
            'cart' => $usersService->getUserCart($user),
            'orders' => $usersService->getUserOrders($user),
            'transactions' => $usersService->getUserTransactions($user),
            'tickets' => $usersService->getUserTickets($user),
            'addresses' => $user->addresses()->latest()->get(),
            'productComments' => $usersService->getUserProductComments($user),
            'genderList' => $this->genderList(),
            'priorityList' => $this->ticketPriorityList(),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(User $user)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, User $user, ProfileService $profileService)
    {
        // only owner access
        if (auth()->user()->type != "owner")
            return redirect()->back()->withErrors([__('messages.just_owner')]);

        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'lowercase', 'max:255', Rule::unique(User::class)->ignore($user->id)],
            'username' => ['nullable', 'string', 'regex:/\w*$/', 'min:4', 'max:20', 'unique:users,username,' . $user->id, new Username()],
            'national_number' => ['nullable', 'string', new NationalNumber()],
        ]);

        $user->update([
            'name' => $request->name,
            'username' => $request->username,
            'email' => $user->email_verified_at == null ? $request->email : $user->email,
            'avatar' => $request->exists('avatar') && $request->avatar != null ? $request->avatar : null,
            'national_number' => $request->exists('national_number') ? $request->national_number : null,
            'birthday' => $profileService->getBirthday($request),
            'gender' => $profileService->getGender($request),
            'bio' => $request->exists('bio') ? $request->bio : null,
            'social' => $profileService->getSocial($request),
        ]);

        return redirect()->back()->with('message', [__('messages.profile_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(User $user)
    {
        // check user type
        if ($user->type == 'owner' || $user->type == 'admin')
            return redirect()->back()->withErrors([__('messages.user_has_role')]);

        // do ban or unban
        if ($user->status == 'active') {
            $user->update([
                'status' => 'banned'
            ]);

            /** Report Action */
            $this->report(__("messages.report_user_ban"), 'user', $user);

            return redirect()->back()->with('message', [__('messages.user_banned')]);
        } else if ($user->status == 'banned') {

            $user->update([
                'status' => 'active'
            ]);

            /** Report Action */
            $this->report(__("messages.report_user_unban"), 'user', $user);

            return redirect()->back()->with('message', [__('messages.user_unbanned')]);
        }
    }

    /**
     * update user wallet amount
     */
    public function walletUpdate(Request $request, User $user)
    {
        // only owner can update wallet
        if (auth()->user()->type != "owner")
            return redirect()->back()->withErrors([__('messages.just_owner')]);

        $request->validate([
            'amount' => 'required'
        ]);

        $oldAmount = $user->wallet->amount;
        $newAmount = (int)str_replace(',', '', $request->amount);
        $transactionId = rand(100000000, 999999999);
        $description = null;
        $type = null;

        // detect type of transaction
        if ($newAmount < $oldAmount) {
            $type = 'admin_decrease';
            $description = __('messages.transction_admin_decrease');
        } else if ($newAmount > $oldAmount) {
            $type = 'admin_increase';
            $description = __('messages.transction_admin_increase');
        } else {
            return redirect()->back();
        }

        // do wallet update
        $user->wallet->update([
            'amount' => $newAmount
        ]);

        // add transaction
        $user->transactions()->create([
            'amount' => $newAmount - $oldAmount,
            'transaction_id' => $transactionId,
            'description' => $description,
            'type' => $type,
            'status' => 'accepted'
        ]);

        /** Report Action */
        $this->report(__("messages.report_update_wallet", ['from' => number_format($oldAmount), 'to' => number_format($newAmount)]), 'user', $user);

        // notification
        $user->notify(new Raw(__("messages.notification_admin_update_user_wallet", ['from' => number_format($oldAmount), 'to' => number_format($newAmount)]), true));

        return redirect()->back()->with('message', [__('messages.user_wallet_updated')]);
    }

    /**
     * edit user score
     */
    public function scoreUpdate(updateUserScoreRequest $request, User $user)
    {
        $score = str_replace(',', '', $request->score);
        if (!is_numeric($score)) {
            return redirect()->back()->withErrors([__('messages.score_amount_invalid')]);
        }

        $type = $request->type;
        $type_translate = null;
        $description = null;
        if ($type == 'increase') {
            $type_translate = __('messages.word.increase');
            $description = __('messages.sentence.increase_score_by_owner');
        } else if ($type == 'decrease') {
            // check small score
            if ($score > $user->get_score['score']) {
                return redirect()->back()->withErrors([__('messages.score_is_small_than_current_score')]);
            }

            $type_translate = __('messages.word.decrease');
            $description = __('messages.sentence.decrease_score_by_owner');
        } else {
            return redirect()->back()->withErrors([__('messages.score_type_invalid')]);
        }

        $user->scores()->create([
            'score' => $score,
            'description' => $description,
            'type' => $type,
        ]);

        /** Report Action */
        $this->report(__("messages.report_update_score", ['score' => number_format($score), 'type' => $type_translate]), 'user', $user);

        // notification
        $user->notify(new Raw(__("messages.notification_admin_update_user_score", ['score' => number_format($score), 'type' => $type_translate]), true));

        return redirect()->back()->with('message', [__('messages.user_score_updated')]);
    }

    /**
     * return the specified user shortcut json
     */
    public function user_json(Request $request, $phone)
    {
        $user = User::where('phone', $phone)->first();

        if ($user == null) {
            return response()->json([
                'status' => 404,
            ]);
        }

        return response()->json([
            'status' => 200,
            'user' => [
                'id' => $user->id,
                'get_name' => $user->get_name,
                'username' => $user->username,
                'email' => $user->email,
                'get_avatar' => $user->get_avatar,
                'type' => $user->type,
            ]
        ]);
    }

    /**
     * return the specified product shortcut json
     */
    public function product_inventory(Request $request, $uuid)
    {
        $product = Product::where('uuid', $uuid)->first();

        if ($product == null) {
            return response()->json([
                'status' => 404,
            ]);
        }

        $inventories = $product->inventories()->with('store')
            ->where('status', 'publish')
            ->where('count', '>', 0)
            ->get()
            ->sortBy(function ($inventory) {
                return (float) $inventory->get_final_price;
            }, SORT_NUMERIC, true)
            ->values();

        return response()->json([
            'status' => 200,
            'product' => [
                'id' => $product->id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'inventory_status' => $product->inventory_status,
                'inventories' => $inventories->toArray()
            ]
        ]);
    }

    /**
     * add to cart
     */
    public function cart_add(User $user, AddToCartRequest $request, CartService $cartService)
    {
        if ($user == null) {
            return redirect()->back();
        }

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();

        // add new order
        if ($order == null) {
            $order = $user->orders()->create([
                'uuid' => rand(100000000, 999999999),
                'status' => 'awaiting',
            ]);
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get product
        $product = $inventory->product;
        if (!$product->in_stock_status) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get count
        $count = $request->count;

        // handle count
        if ($count > $inventory->count) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }
        if ($inventory->max_sale != null && $count > $inventory->max_sale) {
            return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
        }
        if ($inventory->min_sale != null && $count < $inventory->min_sale) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => $inventory->min_sale])]);
        }

        // get inventory props
        $props = $inventory->props->toArray();

        // get store
        $store = $inventory->store;

        /** @var \App\Models\Consignment $consignment */
        $consignment = $order->consignments()->where('store_id', $store != null ? $store->id : null)->first();

        // handle consignment
        if ($consignment == null) {
            $consignment = $order->consignments()->create([
                'user_id' => $user->id,
                'store_id' => $store != null ? $store->id : null,
                'uuid' => 'cn-' . rand(100000000, 999999999),
                'status' => 'created',
            ]);
        }

        // handle discount tree
        $price = $inventory->get_final_price;
        if ($inventory->is_discount_valid) {
            foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                if ($count >= $dt['count']) {
                    $price = str_replace(',', '', $dt['price']);
                    break;
                }
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // handle payment gateway price limit
        $pgPriceLimit = $cartService->validatePaymentGatewayPrice($order, $count, $price);
        if ($pgPriceLimit) {
            return redirect()->back()->withErrors([__('messages.pg_price_limit_error', ['price' => number_format($pgPriceLimit)])]);
        }

        // check for exist consignment items
        $consignmentItem = $consignment->consignmentItems()->where('inventory_id', $inventory->id)->first();

        // add affiliate
        $affiliatesCookie = request()->cookie('affiliates');
        $affiliates = $affiliatesCookie != null ? unserialize($affiliatesCookie) : [];
        $affiliate = null;
        if (array_key_exists($product->uuid, $affiliates)) {
            $targetUser = User::where('id', $affiliates[$product->uuid])->first();
            if ($targetUser != null) {
                $affiliate = $targetUser->id;
            }
        }

        // add inventory to consignment item
        if ($consignmentItem == null) {
            $consignmentItem = $consignment->consignmentItems()->create([
                'store_id' => $store != null ? $store->id : null,
                'inventory_id' => $inventory->id,
                'product_id' => $inventory->product->id,
                'storage_id' => $inventory->storage->id,
                'title' => $inventory->product->title,
                'props' => serialize($props),
                'count' => $count,
                'price' => $price,
                'discount' => $discount,
                'affiliate' => $affiliate,
                'status' => $store != null ? 'awaiting_request_from_seller' : 'ready_to_send',
            ]);
        } else {
            // get new count
            $newCount = $consignmentItem->count + $count;

            // validate again max_sale and total inventory count
            if ($inventory->max_sale != null && $newCount > $inventory->max_sale) {
                return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
            }
            if ($newCount > $inventory->count) {
                return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
            }

            // get new price
            if ($inventory->is_discount_valid) {
                foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                    if ($newCount >= $dt['count']) {
                        $price = str_replace(',', '', $dt['price']);
                        break;
                    }
                }
            }

            // get new discount
            $discount = $inventory->price - $price;

            // update count and price and discount
            $consignmentItem->update([
                'count' => $newCount,
                'price' => $price,
                'discount' => $discount,
                'affiliate' => $affiliate,
            ]);
        }

        // get final count
        $finalCount = $consignment->consignmentItems()->where('inventory_id', $inventory->id)->first()->count;

        return redirect()->back()->with('add_to_cart', [
            'type' => 'added_to_cart',
            'data' => [
                'props' => $props,
                'count' => $finalCount,
                'price' => $price,
                'discount' => $discount,
            ],
        ]);
    }

    /**
     * delete from cart
     */
    public function cart_delete(User $user, DeleteFromCartRequest $request)
    {
        if ($user == null) {
            return redirect()->back();
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // delete consignment item
        $consignmentItem->delete();

        return redirect()->back();
    }

    /**
     * increase 1 unit cart count
     */
    public function cart_increase(User $user, ChangeCartCountRequest $request, CartService $cartService)
    {
        if ($user == null) {
            return redirect()->back();
        }

        /** @var \App\Models\Order $order */
        $order = $user->orders()->where('status', 'awaiting')->first();

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // check for inventory count
        if ($inventory->count == 0) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }

        // get product
        $product = $inventory->product;
        if (!$product->in_stock_status) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get final count
        $finalCount = $consignmentItem->count + 1;

        // handle count
        if ($finalCount > $inventory->count) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }
        if ($inventory->max_sale != null && $finalCount > $inventory->max_sale) {
            return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
        }

        // handle discount
        $price = $inventory->get_final_price;
        if ($inventory->is_discount_valid) {
            foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                if ($finalCount >= $dt['count']) {
                    $price = str_replace(',', '', $dt['price']);
                    break;
                }
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // handle payment gateway price limit
        $pgPriceLimit = $cartService->validatePaymentGatewayPrice($order, 1, $price);
        if ($pgPriceLimit) {
            return redirect()->back()->withErrors([__('messages.pg_price_limit_error', ['price' => number_format($pgPriceLimit)])]);
        }

        // increase consignment item count
        $consignmentItem->update([
            'count' => $finalCount,
            'price' => $price,
            'discount' => $discount,
        ]);

        return redirect()->back();
    }

    /**
     * decrease 1 unit cart count
     */
    public function cart_decrease(User $user, ChangeCartCountRequest $request)
    {
        if ($user == null) {
            return redirect()->back();
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get final count
        $finalCount = $consignmentItem->count - 1;

        // handle count
        if ($finalCount < 1) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => 1])]);
        }
        if ($inventory->min_sale != null && $finalCount < $inventory->min_sale) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => $inventory->min_sale])]);
        }

        // handle discount
        $price = $inventory->get_final_price;
        foreach (array_reverse($inventory->get_discount_tree) as $dt) {
            if ($finalCount >= $dt['count']) {
                $price = str_replace(',', '', $dt['price']);
                break;
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // reduce consignment item count
        $consignmentItem->update([
            'count' => $finalCount,
            'price' => $price,
            'discount' => $discount,
        ]);

        return redirect()->back();
    }

    /**
     * export users
     */
    public function export(Request $request)
    {
        $request->validate([
            'type' => 'required'
        ]);

        $name = null;
        if ($request->type == 'all') {
            $name = 'همه کاربران';
        } else if ($request->type == 'with_order') {
            $name = 'کاربران دارای سفارش';
        } else {
            return redirect()->back()->withErrors([__('unknown_error')]);
        }

        /** Report Action */
        $this->report(__("messages.report_users_exported"));

        return Excel::download(new UsersExport($request->type), 'خروجی کاربران (' . $name . ').xlsx');
    }
}
