<?php

namespace App\Http\Controllers\Admin\Users;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\RolesRequest;
use App\Models\Role;
use App\Models\Permission;
use App\Models\User;
use App\Services\Admin\Users\RoleService;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class RolesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, RoleService $roleService)
    {
        return Inertia::render('Admin/Roles/List', [
            'roles' => $roleService->getRoles($request),
            'permissions' => Permission::get(['id', 'label'])
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(RolesRequest $request): RedirectResponse
    {
        $role = Role::create([
            'name' => $this->randStr(15),
            'label' => $request->label
        ]);

        $role->permissions()->sync($request->permissions);

        Cache::forget('permissions');

        return redirect()->back()->with('message', [__('messages.role_created', ['attribute' => $request->label])]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Role $role)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Role $role)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(RolesRequest $request, Role $role)
    {
        $role->update([
            'label' => $request->label
        ]);

        $role->permissions()->sync($request->permissions);

        Cache::forget('permissions');

        return redirect()->back()->with('message', [__('messages.role_updated', ['attribute' => $role->label])]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Role $role): RedirectResponse
    {
        $label = $role->label;
        $result = $role->delete();

        if ($result)
            return redirect()->back()->with('message', [__('messages.role_deleted', ['attribute' => $label])]);

        Cache::forget('permissions');

        return redirect()->back()->withErrors([__('messages.unknown_error')]);
    }


    /************* ADMIN SECTION *************/

    /**
     * admin list component
     */
    public function adminList(Request $request, RoleService $roleService)
    {
        return Inertia::render('Admin/Roles/Admin', [
            'admins' => $roleService->getAdmins($request),
            'roles' => Role::get(['id', 'label']),
            'keyword' => $request->only(['search'])
        ]);
    }

    /**
     * promote user
     */
    public function promote(Request $request, User $user): RedirectResponse
    {
        $request->validate([
            'role' => 'required',
        ]);

        if ($user->status == "banned") {
            return redirect()->back()->withErrors([__('messages.user_is_banned', ['attribute' => $user->name != null ? $user->name : $user->phone])]);
        }

        $user->update([
            'type' => 'admin',
        ]);

        $user->roles()->sync([$request->role]);

        /** Report Action */
        $this->report(__("messages.report_promote_user"), 'user', $user);

        return redirect()->back()->with('message', [__('messages.user_promoted', ['attribute' => $user->name != null ? $user->name : $user->phone])]);
    }

    /**
     * demote user
     */
    public function demote(User $user): RedirectResponse
    {
        $user->update([
            'type' => 'user',
        ]);

        $user->roles()->sync([]);

        /** Report Action */
        $this->report(__("messages.report_demote_user"), 'user', $user);

        return redirect()->back()->with('message', [__('messages.user_demoted', ['attribute' => $user->name != null ? $user->name : $user->phone])]);
    }
}
