<?php

namespace App\Http\Controllers\Admin\Tickets;

use App\Models\Ticket;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\TicketStoreRequest;
use App\Models\File;
use App\Models\User;
use App\Notifications\Tickets;
use App\Services\Admin\Tickets\TicketsService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class TicketsController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, TicketsService $ticketsService)
    {
        return Inertia::render('Admin/Tickets/List', [
            'tickets' => $ticketsService->getTickets($request)
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(TicketStoreRequest $request)
    {
        /** @var \App\Models\User $user */
        $user = User::find($request->user_id);

        // create ticket
        $ticket = $user->tickets()->create([
            'title' => $request->title,
            'priority' => $request->priority,
            'status' => 'waiting_user',
        ]);

        if ($ticket != null) {
            $ticket->ticketAnswers()->create([
                'user_id' => auth()->user()->id,
                'type' => 'text',
                'content' => $request->content,
                'status' => 'publish',
            ]);

            // notification
            $ticket->user->notify(new Tickets($ticket, 'from_admin'));

            /** Report Action */
            $this->report(__("messages.report_ticket_from_admin"), 'ticket', $ticket);

            return redirect()->route('admin.tickets.show', $ticket->id)->with('message', [__('messages.ticket_created')]);
        } else {
            return redirect()->back()->withErrors([__('messages.ticket_create_error')]);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Ticket $ticket)
    {
        return Inertia::render('Admin/Tickets/Show', [
            'ticket' => [
                'id' => $ticket->id,
                'user_id' => $ticket->user_id,
                'title' => $ticket->title,
                'status' => $ticket->status,

                'ticket_answers' => $ticket->ticketAnswers()->get()->map(fn($answer) => [
                    'id' => $answer->id,
                    'user_id' => $answer->user_id,
                    'content' => $answer->content,
                    'type' => $answer->type,
                    'get_file' => $answer->get_file,
                    'jalali_created_at' => $answer->jalali_created_at,

                    'user' => [
                        'id' => $answer->user->id,
                        'get_name' => $answer->user->get_name,
                        'get_avatar' => $answer->user->get_avatar,
                        'type' => $answer->user->type,
                        'seller' => $answer->user->seller,
                    ],
                ]),
            ]
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Ticket $ticket)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Ticket $ticket)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Ticket $ticket)
    {
        if ($ticket->status != "closed") {
            // close ticket

            $ticket->update([
                'status' => 'closed'
            ]);

            // noification
            $ticket->user->notify(new Tickets($ticket, 'closed'));

            /** Report Action */
            $this->report(__("messages.report_ticket_closed"), 'ticket', $ticket);

            return redirect()->back()->with('message', [__('messages.ticket_closed')]);
        } else {
            // open ticket

            $answers = $ticket->ticketAnswers()->orderBy('created_at', 'desc')->first();
            if ($answers->user->type == "admin" || $answers->user->type == "owner") {
                $ticket->update([
                    'status' => 'waiting_user'
                ]);
            } else {
                $ticket->update([
                    'status' => 'waiting_admin'
                ]);
            }

            // notification
            $ticket->user->notify(new Tickets($ticket, 'opened'));

            /** Report Action */
            $this->report(__("messages.report_ticket_opened"), 'ticket', $ticket);

            return redirect()->back()->with('message', [__('messages.ticket_opened')]);
        }
    }

    /**
     * send answer
     */
    public function answer(Request $request, Ticket $ticket)
    {
        if ($ticket->status == "closed") {
            return redirect()->route('admin.tickets.index')->withErrors([__('messages.ticket_is_closed')]);
        }

        /** @var \App\Models\User $user */
        $user = auth()->user();

        $request->validate([
            'content' => 'required',
        ]);

        // create ticket answer
        $answer = $ticket->ticketAnswers()->create([
            'user_id' => $user->id,
            'type' => 'text',
            'content' => $request->content,
            'status' => 'publish',
        ]);

        if ($answer != null) {
            $ticket->update([
                'status' => 'waiting_user'
            ]);
        } else {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        // noification
        $ticket->user->notify(new Tickets($ticket, 'answer'));

        return redirect()->back()->with('message', [__('messages.ticket_answer_created')]);
    }

    /** send attachment */
    public function attachment(Request $request, Ticket $ticket)
    {
        if ($ticket->status == "closed") {
            return redirect()->route('admin.tickets.index')->withErrors([__('messages.ticket_is_closed')]);
        }

        /** @var \App\Models\User $user */
        $user = auth()->user();

        $request->validate([
            'attachment' => 'required',
        ]);

        $file = File::find($request->attachment);

        if ($file == null) {
            return redirect()->back()->with('message', [__('messages.file_not_found')]);
        }

        $attachment = $ticket->ticketAnswers()->create([
            'user_id' => $user->id,
            'type' => 'file',
            'file' => $file->id,
            'status' => 'publish',
        ]);

        if ($attachment != null) {
            $ticket->update([
                'status' => 'waiting_user'
            ]);
        } else {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        // noification
        $ticket->user->notify(new Tickets($ticket, 'answer'));

        return redirect()->back()->with('message', [__('messages.ticket_attachment_created')]);
    }
}
