<?php

namespace App\Http\Controllers\Admin\Templates;

use App\Models\Template;
use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Services\Admin\Templates\TemplatesService;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File as FacadesFile;

class TemplatesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $templates = Template::orderBy('created_at', 'desc')->get()->map(fn($template) => [
            'id' => $template->id,
            'title' => $template->title,
            'image' => $template->image,
            'type' => $template->type,
            'description' => $template->description,
            'status' => $template->status,
        ]);

        return Inertia::render('Admin/Templates/List', [
            'templates' => $templates
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required'
        ]);

        Template::create([
            'title' => $request->title,
            'description' => $request->exists('description') ? $request->description : null,
            'image' => $request->exists('description') && $request->image != '' ? $request->image  : '/img/demo/templates/blank-template.jpg',
            'options' => serialize([
                'style' => [
                    'site_main_color' => '#0060ff',
                    'site_secondary_color' => '#f8a203',
                    'site_background_color' => '#f7f8fa',
                    'site_main_shadow' => '0',
                    'custom_css' => '',
                    'custom_js' => '',
                ]
            ]),
            'header' => '',
            'footer' => '',
            'data' => serialize([
                'index' => [],
                'product' => [
                    'settings' => [
                        'tabs_status' => true,
                    ]
                ],
            ]),
            'data_backup' => null,
            'blog_data' => serialize([
                'index' => [],
                'article' => [
                    'banners' => []
                ],
            ]),
            'blog_data_backup' => null,
            'type' => 'user',
            'status' => 'deactive',
        ]);

        /** Report Action */
        $this->report(__("messages.report_template_created", ['template' => $request->title]));

        return redirect()->back()->with('message', [__('messages.template_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Template $template)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Template $template, TemplatesService $templatesService)
    {
        // stories status
        $storiesStatus = Setting::where('key', 'stories_status')->first();
        $storiesStatus = $storiesStatus != null ? $storiesStatus->value : false;
        $storiesStatus = $storiesStatus == "1" ? true : false;

        return Inertia::render('Admin/Templates/Edit', [
            'template' => $template,
            'stories' => [
                'status' => $storiesStatus,
                'stories' => []
            ],
            'main_category_list' => $templatesService->getMainCategories(),
            'category_list' => $templatesService->getCategoriesBranch(),
            'product_list' => $templatesService->getProductsList(),
            'raw_product_list' => $templatesService->getRawProductsList(),
            'article_list' => $templatesService->getArticlesList(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Template $template)
    {
        $request->validate([
            'options' => 'required',
            'header' => 'required',
            'footer' => 'required',
            'main_data' => 'required',
            'blog_data' => 'required',
        ]);

        $template->update([
            'options' => serialize($request->options),
            'top_banner' => $request->top_banner != null ? serialize($request->top_banner) : null,
            'header' => $request->header,
            'footer' => $request->footer,
            'data' => serialize($request->main_data),
            'blog_data' => serialize($request->blog_data),
        ]);

        try {
            // delete cache
            FacadesFile::deleteDirectory(storage_path('framework/cache/data'));
            Cache::flush();

            // delete views
            $views = FacadesFile::files(storage_path('framework/views'));
            foreach ($views as $view) {
                FacadesFile::delete($view);
            }
        } catch (\Exception $e) {
        }

        /** Report Action */
        $this->report(__("messages.report_template_edited", ['template' => $template->title]));

        return redirect()->back()->with('message', [__('messages.template_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Template $template)
    {
        if ($template->type == 'user') {
            $template->delete();

            try {
                // delete cache
                FacadesFile::deleteDirectory(storage_path('framework/cache/data'));
                Cache::flush();

                // delete views
                $views = FacadesFile::files(storage_path('framework/views'));
                foreach ($views as $view) {
                    FacadesFile::delete($view);
                }
            } catch (\Exception $e) {
            }

            /** Report Action */
            $this->report(__("messages.report_template_removed", ['template' => $template->title]));

            return redirect()->back()->with('message', [__('messages.template_removed')]);
        } else if ($template->type == 'system') {
            $template->update([
                'options' => $template->options_backup,
                'header' => $template->header_backup,
                'footer' => $template->footer_backup,
                'data' => $template->data_backup,
                'blog_data' => $template->blog_data_backup,
            ]);

            try {
                // delete cache
                FacadesFile::deleteDirectory(storage_path('framework/cache/data'));
                Cache::flush();

                // delete views
                $views = FacadesFile::files(storage_path('framework/views'));
                foreach ($views as $view) {
                    FacadesFile::delete($view);
                }
            } catch (\Exception $e) {
            }

            /** Report Action */
            $this->report(__("messages.report_template_reset", ['template' => $template->title]));

            return redirect()->back()->with('message', [__('messages.template_reset')]);
        }
    }

    /**
     * switch template status
     */
    public function switch(Request $request, Template $template)
    {
        $templates = Template::all();
        foreach ($templates as $t) {
            if ($t->id == $template->id) {
                $t->update([
                    'status' => 'active'
                ]);
            } else {
                $t->update([
                    'status' => 'deactive'
                ]);
            }
        }

        try {
            // delete cache
            FacadesFile::deleteDirectory(storage_path('framework/cache/data'));
            Cache::flush();

            // delete views
            $views = FacadesFile::files(storage_path('framework/views'));
            foreach ($views as $view) {
                FacadesFile::delete($view);
            }
        } catch (\Exception $e) {
        }

        /** Report Action */
        $this->report(__("messages.report_template_activated", ['template' => $template->title]));

        return redirect()->back()->with('message', [__('messages.template_activated')]);
    }
}
