<?php

namespace App\Http\Controllers\Admin\Stores;

use App\Helpers\ClientHelper;
use App\Helpers\DropdownListHelper;
use App\Helpers\UUIDHelper;
use App\Models\Store;
use App\Http\Controllers\Controller;
use App\Models\Inventory;
use App\Models\User;
use App\Notifications\Raw;
use App\Notifications\StoreDirectPost;
use App\Notifications\StoreStatus;
use App\Rules\Phone;
use App\Services\Admin\Blog\ArticlesService;
use App\Services\Admin\Products\ProductService;
use App\Services\Admin\Stores\StoresService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Illuminate\Support\Str;

class StoresController extends Controller
{
    use DropdownListHelper, UUIDHelper, ClientHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, StoresService $storesService)
    {
        return Inertia::render('Admin/Stores/List', [
            'stores' => $storesService->getStores($request),
            'sellerTypeList' => $this->sellerTypeList()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Store $store, StoresService $storesService)
    {
        /** user */
        $user = $store->user()->with('wallet')->first();

        return Inertia::render('Admin/Stores/Show', [
            'store' => $store,
            'user' => [
                'id' => $user->id,
                'get_name' => $user->get_name,
                'articles_count' => $store->user->articles_count,
                'products_count' => $store->user->products_count,
                'type' => $user->type,
                'wallet' => [
                    'id' => $user->wallet->id,
                    'get_formatted_amount' => $user->wallet->get_formatted_amount,
                ],
            ],
            'transactions' => $storesService->getTransactions($user),
            'consignments' => $storesService->getConsignments($user),
            'tickets' => $storesService->getTickets($user),
            'priorityList' => $this->ticketPriorityList()
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Store $store)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Store $store)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Store $store)
    {
        if ($store->status == 'awaiting' || $store->status == 'reject') {
            return redirect()->back()->with('message', [__('messages.unknown_error')]);
        }

        if ($store->status == 'banned') {
            $store->update([
                'status' => 'active'
            ]);

            /** Report Action */
            $this->report(__("messages.report_store_unbanned"), 'store', $store);

            // notification
            $store->user->notify(new StoreStatus('unbanned'));

            return redirect()->back()->with('message', [__('messages.store_unbanned')]);
        } else if ($store->status == 'active') {
            $request->validate([
                'reason' => 'required'
            ]);

            $store->update([
                'status' => 'banned',
                'reason' => $request->reason
            ]);

            /** Report Action */
            $this->report(__("messages.report_store_banned"), 'store', $store);

            // notification
            $store->user->notify(new StoreStatus('banned'));

            return redirect()->back()->with('message', [__('messages.store_banned')]);
        }
    }

    /**
     * change store status (accept or reject)
     */
    public function change(Request $request, Store $store)
    {
        $request->validate([
            'status' => 'required'
        ]);

        if ($request->status == 'accept') {
            $store->update([
                'status' => 'active'
            ]);

            /** Report Action */
            $this->report(__("messages.report_store_accepted"), 'store', $store);

            // notification
            $store->user->notify(new StoreStatus('accepted'));

            return redirect()->back()->with('message', [__('messages.store_accepted')]);
        }

        if ($request->status == 'reject') {
            $request->validate([
                'reason' => 'required'
            ]);

            $store->update([
                'reason' => $request->reason,
                'status' => 'reject',
            ]);

            /** Report Action */
            $this->report(__("messages.report_store_rejected"), 'store', $store);

            // notification
            $store->user->notify(new StoreStatus('rejected'));

            return redirect()->back()->with('message', [__('messages.store_rejected')]);
        }
    }

    /**
     * awaiting stores
     */
    public function awaiting_stores(Request $request, StoresService $storesService)
    {
        return Inertia::render('Admin/Stores/AwaitingStores', [
            'stores' => $storesService->getAwaitingStores($request)
        ]);
    }

    /**
     * awaiting products list
     */
    public function awaiting_products(Request $request, ProductService $productService)
    {
        return Inertia::render('Admin/Stores/AwaitingProducts', [
            'products' => $productService->getProducts($request, "awaiting")
        ]);
    }

    /**
     * awaiting inventories list
     */
    public function awaiting_inventories()
    {
        $awaitingInventories = Inventory::where('status', 'awaiting')->paginate(10)->through(fn($inventory) => [
            'id' => $inventory->id,
            'store_id' => $inventory->store_id,
            'uuid' => $inventory->uuid,
            'props' => $inventory->props,
            'get_formatted_price' => $inventory->get_formatted_price,
            'discount_price' => $inventory->discount_price,
            'get_formatted_discount_price' => $inventory->get_formatted_discount_price,
            'jalali_discount_expire' => $inventory->jalali_discount_expire,
            'count' => $inventory->count,
            'min_sale' => $inventory->min_sale,
            'max_sale' => $inventory->max_sale,
            'get_discount_tree' => $inventory->get_discount_tree,
            'status' => $inventory->status,

            'storage' => $inventory->storage != null ? [
                'id' => $inventory->storage->id,
                'name' => $inventory->storage->name,
            ] : null,
            'product' => [
                'id' => $inventory->product->id,
                'title' => $inventory->product->title,
                'slug' => $inventory->product->slug,
                'status' => $inventory->product->status,
            ],
            'store' => $inventory->store != null ? [
                'id' => $inventory->store->id,
                'uuid' => $inventory->store->uuid,
                'name' => $inventory->store->name,
                'get_logo' => $inventory->store->get_logo,
            ] : null,
        ]);
        $awaitingInventories->withQueryString();

        return Inertia::render('Admin/Stores/AwaitingInventories', [
            'inventories' => $awaitingInventories,
        ]);
    }

    /**
     * awaiting inventories list
     */
    public function awaiting_articles(Request $request, ArticlesService $articlesService)
    {
        return Inertia::render('Admin/Stores/AwaitingArticles', [
            'articles' => $articlesService->getArticles($request, "awaiting")
        ]);
    }

    /**
     * awaiting inventories list
     */
    public function direct_post_status(Store $store)
    {
        if ($store->direct_post) {
            $store->update([
                'direct_post' => false
            ]);

            // notification
            $store->user->notify(new StoreDirectPost($store, 'deactivated'));

            /** Report Action */
            $this->report(__("messages.report_seller_direct_post_deactivated"), 'store',  $store);

            return redirect()->route('admin.stores.show', $store->uuid)->with('message', [__('messages.direct_post_deactivated')]);
        } else {
            $store->update([
                'direct_post' => true
            ]);

            // notification
            $store->user->notify(new StoreDirectPost($store, 'activated'));

            /** Report Action */
            $this->report(__("messages.report_seller_direct_post_activated"), 'store',  $store);

            return redirect()->route('admin.stores.show', $store->uuid)->with('message', [__('messages.direct_post_activated')]);
        }
    }

    /**
     * add seller
     */
    public function add_seller(Request $request)
    {
        $request->validate([
            'phone' => ['required', 'string', new Phone()],
            'type' => ['required', 'string', 'in:legal,genuine'],
            'store_name' => 'required'
        ]);

        // find user
        $user = User::where('phone', $request->phone)->first();

        if ($user == null) {
            // create user
            $user = User::create([
                'name' => $request->exists('user_name') && $request->user_name != null ? $request->user_name : null,
                'username' => $this->generateUsername(),
                'phone' => $request->phone,
                'phone_verified_at' => Carbon::now(),
                'password' => Hash::make(Str::password(50))
            ]);

            // create user wallet
            $user->wallet()->create();
        }

        if ($user->store != null) {
            return redirect()->back()->withErrors([__('messages.seller_already_exists')]);
        }

        if ($user->type == 'owner') {
            return redirect()->back()->withErrors([__('messages.for_owner_can_not_create_store')]);
        }

        if ($request->type == 'genuine') {
            $information = [
                'national_number' => null,
                'national_card_image' => null,
            ];
        } else {
            $information = [
                'company_name' => null,
                'company_type' => null,
                'company_national_number' => null,
                'company_economic_number' => null,
                'company_signatory' => null,
                'legal_identity_image' => null,
            ];
        }

        $store = $user->store()->create([
            'uuid' => $this->generateUniqueRandomNumber(6, \App\Models\Product::class, 'uuid', 'store-'),
            'name' => $request->store_name,
            'meta' => serialize([]),
            'information' => serialize($information),
            'type' => $request->type,
            'status' => 'awaiting',
        ]);

        $user->update([
            'seller' => true,
        ]);

        return redirect()->back()->with('message', [__('messages.seller_added')]);
    }
}
