<?php

namespace App\Http\Controllers\Admin\Storage;

use App\Exports\StorageExport;
use App\Models\Storage;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\StorageRequest;
use App\Imports\StorageImport;
use App\Models\Brand;
use App\Models\Color;
use App\Models\PaymentMeta;
use App\Models\Product;
use App\Services\Admin\Products\ProductService;
use App\Services\Admin\Storage\StorageService;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Maatwebsite\Excel\Facades\Excel;

class StorageController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, StorageService $storageService)
    {
        /** check payment meta */
        if (!$storageService->checkPaymentMeta())
            return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);

        return Inertia::render('Admin/Storage/List', [
            'storages' => $storageService->getStorages($request)
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorageRequest $request, StorageService $storageService)
    {
        /** check payment meta */
        if (!$storageService->checkPaymentMeta())
            return redirect()->back()->withErrors([__('messages.payment_meta_incomplete')]);

        // create storage
        Storage::create([
            'uuid' => 'storage-' . rand(1000000, 9999999),
            'name' => $request->name,
            'address' => $request->address,
        ]);

        /** Report Action */
        $this->report(__("messages.report_storage_created", ['storage' => $request->name]));

        return redirect()->route('admin.storages.index')->with('message', [__('messages.storage_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Storage $storage, Request $request, StorageService $storageService, ProductService $productService)
    {
        return Inertia::render('Admin/Storage/Show', [
            'storage' => $storage,
            'products' => $storageService->getStorageProducts($request, $storage),
            'categories' => $productService->getCategoriesBranch(),
            'brands' => Brand::where('status', 'publish')->get()->map(fn($brand) => [
                'id' => $brand->id,
                'title' => $brand->title,
            ]),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Storage $storage)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(StorageRequest $request, Storage $storage)
    {
        // update storage
        $storage->update([
            'name' => $request->name,
            'address' => $request->address,
        ]);

        /** Report Action */
        $this->report(__("messages.report_storage_updated", ['storage' => $storage->name]));

        return redirect()->route('admin.storages.index')->with('message', [__('messages.storage_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Storage $storage)
    {
        $storageName = $storage->name;

        if ($storage->product_count > 0)
            return redirect()->route('admin.storages.index')->withErrors([__('messages.storage_has_product_can_not_delete')]);

        // delete storage
        $storage->delete();

        /** Report Action */
        $this->report(__("messages.report_storage_deleted", ['storage' => $storageName]));

        return redirect()->route('admin.storages.index')->with('message', [__('messages.storage_deleted')]);
    }

    /**
     * storage inventories list
     */
    public function storageList(Storage $storage, Product $product)
    {
        // inventories list
        $inventories = $product->inventories()->where('storage_id', $storage->id)->orderBy('created_at', 'desc')->get();

        // other inventories count
        $inventories_count = [
            'you' => $product->inventories()->where('store_id', null)->get()->count() - count($inventories),
            'other' => $product->inventories()->get()->count() - ($product->inventories()->where('store_id', null)->get()->count()),
        ];

        return Inertia::render('Admin/Storage/Inventory/StorageList', [
            'storage' => $storage,
            'product' => [
                'id' => $product->id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'price_model' => $product->price_model,
                'is_vip' => $product->is_vip,
                'commission' => $product->commission,
            ],
            'inventories' => $inventories,
            'inventories_count' => $inventories_count
        ]);
    }

    /**
     * export storage
     */
    public function export(Storage $storage, Request $request)
    {
        $request->validate([
            'type' => 'required',
            'category' => 'required',
            'brand' => 'required',
        ]);

        /** Report Action */
        $this->report(__("messages.report_storage_exported", ['storage' => $storage->name]));

        return Excel::download(new StorageExport($storage, $request->type, $request->category, $request->brand), 'خروجی انبار ' . $storage->name . '.xlsx');
    }

    /**
     * import storage
     */
    public function import(Storage $storage, Request $request)
    {
        $request->validate([
            'file' => 'required',
        ]);

        /** Report Action */
        $this->report(__("messages.report_storage_imported", ['storage' => $storage->name]));

        Excel::import(new StorageImport($storage), $request->file('file'));

        return redirect()->back()->with('message', [__('messages.excel_imported_to_storage')]);
    }
}
