<?php

namespace App\Http\Controllers\Admin\Storage;

use App\Helpers\UUIDHelper;
use App\Models\Inventory;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\InventoryRequest;
use App\Models\Color;
use App\Models\Product;
use App\Models\Storage;
use App\Notifications\InventoryStatus;
use App\Services\Admin\Storage\InventoriesService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Morilog\Jalali\Jalalian;

class InventoryController extends Controller
{
    use UUIDHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Product $product, InventoriesService $inventoriesService)
    {
        return Inertia::render('Admin/Storage/Inventory/List', [
            'product' => [
                'id' => $product->id,
                'uuid' => $product->uuid,
                'title' => $product->title,
                'slug' => $product->slug,
                'get_images' => $product->get_images,
                'price_model' => $product->price_model,
                'is_vip' => $product->is_vip,
                'commission' => $product->commission,
            ],
            'inventories' => $inventoriesService->getInventories($product),
            'colors' => Color::all(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(Request $request, InventoriesService $inventoriesService)
    {
        $request->validate([
            'uuid' => 'required',
        ]);

        // product
        $product = Product::where('uuid', $request->uuid)->first();
        if ($product == null)
            return redirect()->route('admin.products.index')->withErrors([__('messages.product_not_found')]);

        // validate single price
        if ($inventoriesService->validateSinglePrice($product))
            return redirect()->back()->withErrors([__('messages.just_can_add_one_price')]);

        // storage list
        $storages = Storage::where('store_id', null)->get()->map(fn($storage) => [
            'id' => $storage->id,
            'name' => $storage->name,
        ]);

        // get storage
        $storage = Storage::where('uuid', $request->stg)->first();
        if ($storage != null) {
            $storage = $storage->id;
        }

        return Inertia::render('Admin/Storage/Inventory/Create', [
            'product' => [
                'id' => $product->id,
                'uuid' => $product->uuid,
                'price_model' => $product->price_model,
            ],
            'storages' => $storages,
            'colors' => Color::all(),
            'originalList' => [
                ["id" => true, "label" => __('messages.sentence.original_product')],
                ["id" => false, "label" => __('messages.sentence.unoriginal_product')],
            ],
            'usedList' => [
                ["id" => false, "label" => __('messages.sentence.unused_product')],
                ["id" => true, "label" => __('messages.sentence.used_product')],
            ],
            'storage' => $storage,
            'storage_uuid' => $request->stg,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(InventoryRequest $request, InventoriesService $inventoriesService)
    {
        // product
        $product = Product::where('uuid', $request->uuid)->first();
        if ($product == null)
            return redirect()->route('admin.products.index')->withErrors([__('messages.product_not_found')]);

        // init fields
        $props = $inventoriesService->getProps($request, $product);
        $price = (int)str_replace(',', '', $request->price);
        $purchasePrice = $request->has('purchase_price') ? (int)str_replace(',', '', $request->purchase_price) : null;
        $discount_price = null;
        $discount_expire = null;
        $count = (int)str_replace(',', '', $request->count);
        $minSale = $request->exists('min_sale') && $request->min_sale != null && $request->min_sale > 0 ? (int)str_replace(',', '', $request->min_sale) : 1;
        $maxSale = $request->exists('max_sale') && $request->max_sale != null && $request->max_sale != 0 ? (int)str_replace(',', '', $request->max_sale) : null;
        $discountTree = [];

        // validate single price
        if ($inventoriesService->validateSinglePrice($product))
            return redirect()->back()->withErrors([__('messages.just_can_add_one_price')]);

        if (count($props) > 0) {
            // validate props
            if ($inventoriesService->validateProps($props, $product) !== false)
                $props = $inventoriesService->validateProps($props, $product);
            else
                return redirect()->back()->withErrors([__('messages.product_props_required')]);

            // validate duplicated props
            if (!$inventoriesService->validateDuplicateProps($props, $product))
                return redirect()->back()->withErrors([__('messages.price_is_duplicated')]);
        }

        // validate discount fields
        if ($request->discount_price != null && $request->discount_price != 0) {
            // compare discount_price and main price
            $discount_price = (int)str_replace(',', '', $request->discount_price);
            if ($discount_price > $price) {
                return redirect()->back()->withErrors([__('messages.discount_price_larger_than_price')]);
            }

            // check discount_expire required field
            if ($request->discount_expire != null) {
                $discount_expire = Jalalian::fromFormat('Y/m/d', $request->discount_expire)->getTimestamp();
                $discount_expire = Jalalian::forge($discount_expire)
                    ->toCarbon()
                    ->setTimezone('Asia/Tehran')
                    ->setTime(23, 59, 59); // تنظیم ساعت به 23:59:00
                $discount_expire = $discount_expire->format('Y-m-d H:i:s');
            }
        }

        // validate discount tree
        if ($request->exists('discount_tree') && is_array($request->discount_tree) && count($request->discount_tree) > 0) {
            if ($request->discount_price == null) {
                return redirect()->back()->withErrors([__('messages.to_add_discount_tree_first_fill_main_discount')]);
            }

            foreach ($request->discount_tree as $dt) {
                // check for null
                if ($dt['count'] != null && $dt['count'] != '' && $dt['price'] != null && $dt['price'] != '') {

                    // discount_tree fields
                    $dtCount = (int)str_replace(',', '', $dt['count']);
                    $dtPrice = (int)str_replace(',', '', $dt['price']);

                    // compare discount_tree price and main price
                    if ($dtPrice > $price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_price')]);
                    }
                    if ($discount_price != null && $dtPrice > $discount_price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_discount_price')]);
                    }

                    $discountTree[] = [
                        'count' => $dtCount,
                        'price' => $dtPrice,
                    ];
                }
            }

            // sort discount tree
            sort($discountTree);

            // validate discount tree counts
            $discountTreeCounts = [];
            foreach ($discountTree as $dt) {
                $discountTreeCounts[$dt['count']][] = $dt['count'];
            }
            if (count($discountTreeCounts) != count($discountTree)) {
                return redirect()->back()->withErrors([__('messages.discount_tree_count_wrong')]);
            }

            // validate discount tree prices
            $dtPrevPrice = null;
            foreach ($discountTree as $dt) {
                if ($dtPrevPrice != null) {
                    if ($dtPrevPrice <= $dt['price']) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_wrong')]);
                    }
                }
                $dtPrevPrice = $dt['price'];
            }
        }

        // check min and max sale value
        if ($maxSale != null && $maxSale != '' && $minSale > $maxSale) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_max_sale')]);
        }
        if ($minSale != null && $count != 0 && $minSale > $count) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_count')]);
        }

        // validate weight
        $weight = null;
        if ($request->exists('weight') && $request->weight != null && is_numeric(str_replace(',', '', $request->weight))) {
            $weight = str_replace(',', '', $request->weight);
        }

        // price changes
        $priceChanges[Carbon::now()->timestamp] = $discount_price != null ? $discount_price : $price;

        // add inventory
        $inventory = $product->inventories()->create([
            'storage_id' => $request->storage,
            'uuid' => $this->generateUniqueRandomNumber(8, \App\Models\Product::class, 'uuid', 'inv-'),
            'price' => $price,
            'discount_price' => $discount_price,
            'discount_expire' => $discount_expire,
            'count' => $count,
            'min_sale' => $minSale,
            'max_sale' => $maxSale,
            'original' => $request->original,
            'used' => $request->used,
            'weight' => $weight,
            'purchase_price' => $purchasePrice,
            'discount_tree' => is_array($discountTree) && count($discountTree) > 0 ? serialize($discountTree) : null,
            'price_changes' => $priceChanges != null ? serialize($priceChanges) : null,
            'image' => $request->exists('image') ? $request->image : null,
            'description' => $request->exists('description') ? $request->description : null,
            'send_time' => $request->exists('send_time') ? $request->send_time : null,
            'status' => 'publish',
        ]);

        // add inventory props
        $pmID = 0;
        foreach ($product->price_model as $key => $pm) {
            if (array_key_exists($key, $props)) {
                if (array_key_exists('value', $props[$key])) {
                    if ($props[$key]['value'] == null || $props[$key]['value'] == '') {
                        return redirect()->back()->withErrors([__('messages.product_props_required')]);
                    }

                    $pmModel = $inventory->inventoryProps()->create([
                        'child' => $pmID,
                        'name' => $pm['name'],
                        'type' => $pm['type'],
                        'value' => $pm['type'] == 'color' ? serialize($props[$key]['value']) : $props[$key]['value'],
                    ]);
                    $pmID = $pmModel->id;
                } else {
                    return redirect()->back()->withErrors([__('messages.product_props_required')]);
                }
            } else {
                return redirect()->back()->withErrors([__('messages.product_props_required')]);
            }
        }

        /** product notifications */
        $inventoriesService->handleProductNotifications($product);

        /** Report Action */
        $this->report(__("messages.report_inventory_added"), 'product', $product);

        if ($request->exists('stg') && $request->stg != null && $request->stg != '')
            return redirect()->route('admin.storages.storageList', ['storage' => $request->stg, 'product' => $product->slug])->with('message', [__('messages.inventory_added')]);

        return redirect()->route('admin.inventories.index', $product->slug)->with('message', [__('messages.inventory_added')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Inventory $inventory)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Inventory $inventory, Request $request, InventoriesService $inventoriesService)
    {
        // check for main store inventory
        if ($inventory->store_id != null)
            return redirect()->back()->withErrors([__('messages.unauthorized')]);

        $request->validate([
            'uuid' => 'required'
        ]);

        // product
        $product = Product::where('uuid', $request->uuid)->first();
        if ($product == null)
            return redirect()->route('admin.products.index')->withErrors([__('messages.product_not_found')]);

        // storage list
        $storages = Storage::where('store_id', null)->get()->map(fn($storage) => [
            'id' => $storage->id,
            'name' => $storage->name,
        ]);

        return Inertia::render('Admin/Storage/Inventory/Edit', [
            'inventory' => $inventory,
            'product' => [
                'id' => $product->id,
                'uuid' => $product->uuid,
            ],
            'priceModel' => $inventoriesService->getPriceModel($inventory),
            'storages' => $storages,
            'colors' => Color::all(),
            'originalList' => [
                ["id" => true, "label" => __('messages.sentence.original_product')],
                ["id" => false, "label" => __('messages.sentence.unoriginal_product')],
            ],
            'usedList' => [
                ["id" => false, "label" => __('messages.sentence.unused_product')],
                ["id" => true, "label" => __('messages.sentence.used_product')],
            ],
            'storage_uuid' => $request->stg,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(InventoryRequest $request, Inventory $inventory, InventoriesService $inventoriesService)
    {
        // check for main store inventory
        if ($inventory->store_id != null)
            return redirect()->back()->withErrors([__('messages.unauthorized')]);

        // product
        $product = Product::where('uuid', $request->uuid)->first();
        if ($product == null)
            return redirect()->route('admin.products.index')->withErrors([__('messages.product_not_found')]);

        // init fields
        $props = $inventoriesService->getProps($request, $product);
        $price = (int)str_replace(',', '', $request->price);
        $purchasePrice = $request->has('purchase_price') ? (int)str_replace(',', '', $request->purchase_price) : null;
        $discount_price = null;
        $discount_expire = null;
        $count = (int)str_replace(',', '', $request->count);
        $minSale = $request->exists('min_sale') && $request->min_sale != null && $request->min_sale > 0 ? (int)str_replace(',', '', $request->min_sale) : 1;
        $maxSale = $request->exists('max_sale') && $request->max_sale != null && $request->max_sale != 0 ? (int)str_replace(',', '', $request->max_sale) : null;
        $discountTree = [];

        if (count($props) > 0) {
            // validate props
            if ($inventoriesService->validateProps($props, $product, $inventory) != false)
                $props = $inventoriesService->validateProps($props, $product, $inventory);
            else
                return redirect()->back()->withErrors([__('messages.product_props_required')]);

            // validate duplicated props
            if (!$inventoriesService->validateDuplicateProps($props, $product, $inventory))
                return redirect()->back()->withErrors([__('messages.price_is_duplicated')]);
        }

        // validate discount fields
        if ($request->discount_price != null && $request->discount_price != 0) {
            // compare discount_price and main price
            $discount_price = (int)str_replace(',', '', $request->discount_price);
            if ($discount_price > $price) {
                return redirect()->back()->withErrors([__('messages.discount_price_larger_than_price')]);
            }

            // check discount_expire required field
            if ($request->discount_expire != null) {
                $discount_expire = Jalalian::fromFormat('Y/m/d', $request->discount_expire)->getTimestamp();
                $discount_expire = Jalalian::forge($discount_expire)
                    ->toCarbon()
                    ->setTimezone('Asia/Tehran')
                    ->setTime(23, 59, 59); // تنظیم ساعت به 23:59:00
                $discount_expire = $discount_expire->format('Y-m-d H:i:s');
            }
        }

        // validate discount tree
        if ($request->exists('discount_tree') && is_array($request->discount_tree) && count($request->discount_tree) > 0) {
            if ($request->discount_price == null) {
                return redirect()->back()->withErrors([__('messages.to_add_discount_tree_first_fill_main_discount')]);
            }

            foreach ($request->discount_tree as $dt) {
                // check for null
                if ($dt['count'] != null && $dt['count'] != '' && $dt['price'] != null && $dt['price'] != '') {
                    // discount_tree fields
                    $dtCount = (int)str_replace(',', '', $dt['count']);
                    $dtPrice = (int)str_replace(',', '', $dt['price']);
                    // compare discount_tree price and main price
                    if ($dtPrice > $price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_price')]);
                    }
                    if ($discount_price != null && $dtPrice > $discount_price) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_larger_than_discount_price')]);
                    }

                    $discountTree[] = [
                        'count' => $dtCount,
                        'price' => $dtPrice,
                    ];
                }
            }

            // sort discount tree
            sort($discountTree);

            // validate discount tree counts
            $discountTreeCounts = [];
            foreach ($discountTree as $dt) {
                $discountTreeCounts[$dt['count']][] = $dt['count'];
            }
            if (count($discountTreeCounts) != count($discountTree)) {
                return redirect()->back()->withErrors([__('messages.discount_tree_count_wrong')]);
            }

            // validate discount tree prices
            $dtPrevPrice = null;
            foreach ($discountTree as $dt) {
                if ($dtPrevPrice != null) {
                    if ($dtPrevPrice <= $dt['price']) {
                        return redirect()->back()->withErrors([__('messages.discount_tree_price_wrong')]);
                    }
                }
                $dtPrevPrice = $dt['price'];
            }
        }

        // check min and max sale value
        if ($maxSale != null && $maxSale != '' && $minSale > $maxSale) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_max_sale')]);
        }
        if ($minSale != null && $count != 0 && $minSale > $count) {
            return redirect()->back()->withErrors([__('messages.min_sale_larger_than_count')]);
        }

        // validate weight
        $weight = $inventory->weight;
        if ($request->exists('weight') && $request->weight != null && is_numeric(str_replace(',', '', $request->weight))) {
            $weight = str_replace(',', '', $request->weight);
        }

        // update inventory
        $inventory->update([
            'storage_id' => $request->storage,
            'price' => $price,
            'discount_price' => $discount_price,
            'discount_expire' => $discount_expire,
            'count' => $count,
            'min_sale' => $minSale,
            'max_sale' => $maxSale,
            'original' => $request->original,
            'used' => $request->used,
            'weight' => $weight,
            'purchase_price' => $purchasePrice,
            'discount_tree' => is_array($discountTree) && count($discountTree) > 0 ? serialize($discountTree) : null,
            'price_changes' => serialize($inventoriesService->handlePriceChanges($inventory->get_price_changes, $discount_price != null ? $discount_price : $price)),
            'image' => $request->exists('image') && $request->image != null ? $request->image : $inventory->image,
            'description' => $request->exists('description') && $request->description != null ? $request->description : $inventory->description,
            'send_time' => $request->exists('send_time') ? $request->send_time : $inventory->send_time,
            'status' => 'publish',
        ]);

        // update inventory props
        $inventoryProps = $inventory->inventoryProps()->get();
        foreach ($inventoryProps as $inventoryProp) {
            if (array_key_exists($inventoryProp->id, $props)) {
                if (array_key_exists('value', $props[$inventoryProp->id])) {
                    if ($props[$inventoryProp->id]['value'] == null || $props[$inventoryProp->id]['value'] == '') {
                        return redirect()->back()->withErrors([__('messages.product_props_required')]);
                    }

                    $inventoryProp->update([
                        'value' => $inventoryProp->type == 'color' ? serialize($props[$inventoryProp->id]['value']) : $props[$inventoryProp->id]['value'],
                    ]);
                } else {
                    return redirect()->back()->withErrors([__('messages.product_props_required')]);
                }
            } else {
                return redirect()->back()->withErrors([__('messages.product_props_required')]);
            }
        }

        /** product notifications */
        $inventoriesService->handleProductNotifications($product);

        /** Report Action */
        $this->report(__("messages.report_inventory_updated"), 'product', $product);

        if ($request->exists('stg') && $request->stg != null && $request->stg != '')
            return redirect()->route('admin.storages.storageList', ['storage' => $request->stg, 'product' => $product->slug])->with('message', [__('messages.inventory_updated')]);

        return redirect()->route('admin.inventories.index', $product->slug)->with('message', [__('messages.inventory_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Inventory $inventory)
    {
        $product = $inventory->product;
        $store = $inventory->store;

        // delete inventory from users cart
        $inventory->consignmentItems()->whereHas('consignment', function ($query) {
            $query->whereHas('order', function ($query) {
                $query->whereDoesntHave('orderGroup');
            });
        })->delete();

        // delete
        $inventory->delete();

        /** Report Action */
        $this->report(__("messages.report_inventory_deleted"), 'product', $product);

        if ($store != null) {
            // notification
            $store->user->notify(new InventoryStatus($inventory, 'delete'));
        }

        return redirect()->back()->with('message', [__('messages.inventory_deleted')]);
    }

    /**
     * decrease inventory
     */
    public function decrease(Inventory $inventory)
    {
        if ($inventory->store_id != null)
            return redirect()->back()->withErrors([__('messages.unauthorized')]);

        if ($inventory->count == 0)
            return redirect()->back()->withErrors([__('messages.inventory_count_is_zero')]);

        // decrease inventory
        $inventory->update([
            'count' => $inventory->count - 1
        ]);

        return redirect()->back()->with('message', [__('messages.inventory_count_decrease')]);
    }

    /**
     * change inventory status
     */
    public function status(Request $request, Inventory $inventory)
    {
        // check force seller inventories
        if ($inventory->store == null) {
            return redirect()->back();
        }

        /** @var App\Models\User $user */
        $store = $inventory->store;

        /** inventory product */
        $product = $inventory->product;

        $request->validate([
            'status' => 'required'
        ]);

        if ($request->status == 'activate') {
            $inventory->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_inventory_activated"), 'product', $product);

            // notification
            $store->user->notify(new InventoryStatus($inventory, 'active'));

            return redirect()->back()->with('message', [__('messages.inventory_activated')]);
        } else if ($request->status == 'confirmation') {
            $inventory->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_inventory_confirmed"), 'product', $product);

            // notification
            $store->user->notify(new InventoryStatus($inventory, 'confirmation'));

            return redirect()->back()->with('message', [__('messages.inventory_confirmed')]);
        } else if ($request->status == 'deactivate') {
            $inventory->update([
                'status' => 'deactive'
            ]);

            /** Report Action */
            $this->report(__("messages.report_inventory_deactivated"), 'product', $product);

            // notification
            $store->user->notify(new InventoryStatus($inventory, 'deactive'));

            return redirect()->back()->with('message', [__('messages.inventory_deactivated')]);
        }

        return redirect()->back();
    }
}
