<?php

namespace App\Http\Controllers\Admin\Products;

use App\Models\Color;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ColorsRequest;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ColorController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        /** @var LengthAwarePaginator */
        $colors = Color::query()->when($request->input('search'), function ($query, $search) {
            $query->where('label', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->paginate(40)->through(fn($brand) => [
            'id' => $brand->id,
            'label' => $brand->label,
            'color' => $brand->color,
            'image' => $brand->image,
        ]);
        $colors->withQueryString();

        return Inertia::render('Admin/Products/Colors/List', [
            'colors' => $colors
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Products/Colors/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ColorsRequest $request)
    {
        // check unique color
        if (Color::where('color', $request->color)->first() != null)
            return redirect()->back()->withErrors([__('messages.color_exists')]);

        // add color to DB
        Color::create([
            'color' => $request->color,
            'label' => $request->label,
            'image' => $request->image,
        ]);

        /** Report Action */
        $this->report(__("messages.report_color_created"));

        return redirect()->route('admin.product.colors.index')->with('message', [__('messages.color_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Color $color)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Color $color)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Color $color)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Color $color)
    {
        $color->delete();

        /** Report Action */
        $this->report(__("messages.report_color_deleted"));

        return redirect()->back()->with('message', [__('messages.color_deleted')]);
    }
}
