<?php

namespace App\Http\Controllers\Admin\Pages;

use App\Helpers\DropdownListHelper;
use App\Helpers\UUIDHelper;
use App\Models\Page;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\PagesRequest;
use App\Services\Admin\Pages\PagesService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class PagesController extends Controller
{
    use DropdownListHelper, UUIDHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, PagesService $pagesService)
    {
        // create system pages
        $pagesService->createSystemPages();

        return Inertia::render('Admin/Pages/List', [
            'pages' => $pagesService->getPages($request)
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Pages/Create', [
            'statusList' => $this->contentStatusList()
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(PagesRequest $request)
    {
        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null) {
            $pageWithSlug = Page::where('slug', $request->slug)->first();
            if ($pageWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        $page = Page::create([
            'uuid' => $this->generateUniqueRandomNumber(6, \App\Models\Page::class, 'uuid', 'page-'),
            'title' => $request->title,
            'slug' => $request->exists('slug') && $request->slug != null ? $request->slug : null,
            'content' => $request->content,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : null,

            'flag' => 'normal',
            'status' => $request->status,
        ]);

        /** Report Action */
        $this->report(__("messages.report_page_created"), 'page', $page);

        return redirect()->route('admin.pages.index')->with('message', [__('messages.page_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Page $page)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $page)
    {
        return Inertia::render('Admin/Pages/Edit', [
            'page' => $page,
            'statusList' => $this->contentStatusList()
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(PagesRequest $request, Page $page)
    {
        /** validate duplicated slug */
        if ($request->exists('slug') && $request->slug != null && $request->slug != $page->slug) {
            $pageWithSlug = Page::where('slug', $request->slug)->first();
            if ($pageWithSlug != null) {
                return redirect()->back()->withErrors([__('messages.slug_is_exist')]);
            }
        }

        $page->update([
            'title' => $request->title,
            'slug' => $request->exists('slug') && $page->flag != 'system' ? $request->slug : $page->slug,
            'content' => $request->content,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $page->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $page->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $page->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : $page->seo_canonical,

            'status' => $request->status,
        ]);

        /** Report Action */
        $this->report(__("messages.report_page_updated"), 'page', $page);

        return redirect()->back()->with('message', [__('messages.page_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Page $page)
    {
        if ($page->flag == 'system') {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        if ($request->status == 'deleted') {
            $page->delete();

            /** Report Action */
            $this->report(__("messages.report_page_deleted"), 'page', $page);

            return redirect()->back()->with('message', [__('messages.page_deleted')]);
        } else if ($page->status != 'publish' && $request->status == 'publish') {
            $page->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_page_publish"), 'page', $page);

            return redirect()->back()->with('message', [__('messages.page_publish')]);
        } else if ($page->status != 'draft' && $request->status == 'draft') {
            $page->update([
                'status' => 'draft'
            ]);

            /** Report Action */
            $this->report(__("messages.report_page_draft"), 'page', $page);

            return redirect()->back()->with('message', [__('messages.page_draft')]);
        }

        return redirect()->back()->withErrors([__('messages.unknown_error')]);
    }
}
