<?php

namespace App\Http\Controllers\Admin\Menu;

use App\Helpers\DropdownListHelper;
use App\Models\Menu;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\MenuRequest;
use App\Models\Form;
use App\Models\Page;
use App\Models\ProductCategory;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Inertia\Inertia;

class MenusController extends Controller
{
    use DropdownListHelper;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        return Inertia::render('Admin/Menu/Manage', [
            'menus' => Menu::all(),
            'menuPlaceList' => $this->menuPlaceList()
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(MenuRequest $request): RedirectResponse
    {
        $menu = Menu::create([
            'name' => $request->name,
            'place' => $request->place,
        ]);

        /** Report Action */
        $this->report(__("messages.report_menus_created"));

        return redirect()->route('admin.menus.edit', $menu->id)->with('message', [__('messages.menu_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Menu $menu)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Menu $menu)
    {
        // all menus
        $menus = Menu::all();

        $staticLinks = [
            [
                "id" => 1,
                "name" => __('messages.sentence.home_page'),
                "link" => '/',
                'type' => 'internal'
            ],
            [
                "id" => 1,
                "name" => __('messages.word.blog'),
                "link" => '/blog',
                'type' => 'internal'
            ],
            [
                "id" => 1,
                "name" => __('messages.word.faqs'),
                "link" => '/faq',
                'type' => 'internal'
            ],
            [
                "id" => 2,
                "name" => __('messages.sentence.become_seller'),
                "link" => '/auth/seller/register',
                'type' => 'internal'
            ],
            [
                "id" => 2,
                "name" => __('messages.sentence.sellers_panel'),
                "link" => '/seller/dashboard',
                'type' => 'internal'
            ]
        ];

        $categories = ProductCategory::where('status', 'publish')->get()->map(fn($page) => [
            'id' => $page->id,
            'name' => $page->title,
            'link' => str_replace(rtrim(asset(''), '/'), '', route('main.products.index', ['cat' => $page->slug])),
            'type' => 'internal'
        ])->toArray();

        $pages = Page::where('status', 'publish')->get()->map(fn($page) => [
            'id' => $page->id,
            'name' => $page->title,
            'link' => str_replace(rtrim(asset(''), '/'), '', route('main.pages.show', $page->slug)),
            'type' => 'internal'
        ])->toArray();

        $forms = Form::where('status', 'publish')->get()->map(fn($page) => [
            'id' => $page->id,
            'name' => $page->title,
            'link' => str_replace(rtrim(asset(''), '/'), '', route('main.forms.show', $page->slug)),
            'type' => 'internal'
        ])->toArray();

        return Inertia::render('Admin/Menu/Edit', [
            'menus' => $menus,
            'menu' => $menu->safe,
            'staticLinks' => $staticLinks,
            'categories' => $categories,
            'pages' => $pages,
            'forms' => $forms,
            'menuPlaceList' => $this->menuPlaceList(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(MenuRequest $request, Menu $menu)
    {
        $menu->update([
            'name' => $request->name,
            'content' => $request->items != null ? serialize($request->items) : serialize([]),
            'place' => $request->place,
        ]);

        /** Report Action */
        $this->report(__("messages.report_menus_changed"));

        return redirect()->back()->with('message', [__('messages.menu_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Menu $menu): RedirectResponse
    {
        $menu->delete();

        /** Report Action */
        $this->report(__("messages.report_menus_removed"));

        return redirect()->route('admin.menus.index')->with('message', [__('messages.menu_deleted')]);
    }
}
