<?php

namespace App\Http\Controllers\Admin\Financial;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ChangeWithdrawalStatusRequest;
use App\Http\Requests\Admin\DeleteWithdrawalRequest;
use App\Models\Transaction;
use App\Models\Withdrawal;
use App\Notifications\WithdrawalDeleted;
use App\Notifications\WithdrawalDeposited;
use App\Services\Admin\Financial\WithdrawalsService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class WithdrawalsController extends Controller
{
    /**
     * withdrawals page
     */
    public function index(Request $request, WithdrawalsService $withdrawalsService)
    {
        return Inertia::render('Admin/Financial/Withdrawals/List', [
            'withdrawals' => $withdrawalsService->getWithdrawals($request),
        ]);
    }

    /**
     * deposited withdrawal status
     */
    public function deposited(ChangeWithdrawalStatusRequest $request, Withdrawal $withdrawal)
    {
        if ($request->status == 'deposited') {
            // update transaction
            $transaction = Transaction::where('transaction_id', $withdrawal->transaction_id)->first();
            $transaction->update([
                'status' => 'accepted',
            ]);

            // update withdrawal
            $withdrawal->update([
                'tracking_code' => $request->tracking_code,
                'status' => 'deposited',
            ]);
            $withdrawal = Withdrawal::find($withdrawal->id);

            /** Report Action */
            $this->report(__("messages.report_withdrawal_deposited", ['transaction_id' => $withdrawal->transaction_id]), 'user', $withdrawal->user);

            // notification
            $withdrawal->user->notify(new WithdrawalDeposited($withdrawal));

            return redirect()->back()->with('message', [__('messages.withdrawal_deposited')]);
        }

        return redirect()->back();
    }

    /**
     * delete withdrawal
     */
    public function delete(DeleteWithdrawalRequest $request, Withdrawal $withdrawal)
    {
        /** @var \App\Models\User $user */
        $user = $withdrawal->user;
        $wallet = $user->wallet;

        // delete from transaction list
        $transaction = Transaction::where('transaction_id', $withdrawal->transaction_id)->first();
        if ($transaction != null) {
            $transaction->update([
                'status' => 'rejected'
            ]);
        }

        // increase the amount in the wallet
        $user->wallet->update([
            'amount' => $wallet->amount + $withdrawal->amount,
        ]);

        // change status to deleted
        $withdrawal->update([
            'reason' => $request->reason,
            'status' => 'deleted',
        ]);

        // get withdrawal with new value
        $withdrawal = Withdrawal::find($withdrawal->id);

        /** Report Action */
        $this->report(__("messages.report_withdrawal_deleted", ['transaction_id' => $withdrawal->transaction_id]), 'user', $withdrawal->user);

        // notification
        $withdrawal->user->notify(new WithdrawalDeleted($withdrawal));

        return redirect()->back()->with('message', [__('messages.withdrawal_request_deleted')]);
    }
}
