<?php

namespace App\Http\Controllers\Admin\Financial;

use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Http\Requests\Main\AddToCartRequest;
use App\Http\Requests\Main\ChangeCartCountRequest;
use App\Http\Requests\Main\DeleteFromCartRequest;
use App\Models\ConsignmentItem;
use App\Models\Discount;
use App\Models\Inventory;
use App\Models\Order;
use App\Models\Prefactor;
use App\Models\User;
use App\Notifications\PrefactorIssued;
use App\Services\Admin\Financial\PrefactorsService;
use App\Services\Cart\CartService;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Morilog\Jalali\Jalalian;

class PrefactorsController extends Controller
{
    use UUIDHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, PrefactorsService $prefactorsService)
    {
        return Inertia::render('Admin/Financial/Prefactors/List', [
            'prefactors' => $prefactorsService->getPrefactors($request),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required',
            'title' => 'required',
        ]);

        $user = User::find($request->user_id);
        if ($user == null) {
            return redirect()->back()->withErrors([__('messages.user_not_found')]);
        }

        $order = $user->orders()->create([
            'uuid' => $this->generateUniqueRandomNumber(9, \App\Models\Order::class, 'uuid'),
            'status' => 'prefactor',
        ]);

        $prefactor = $user->prefactors()->create([
            'order_id' => $order->id,
            'by' => auth()->user()->id,
            'uuid' => $order->uuid,
            'title' => $request->title,
            'status' => 'awaiting_issuance',
        ]);

        return redirect()->route('admin.prefactors.edit', $prefactor->id)->with('message', [__('messages.prefactor_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Prefactor $prefactor, PrefactorsService $prefactorsService)
    {
        if ($prefactor == null || $prefactor->status != 'paid') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        return Inertia::render('Admin/Financial/Prefactors/Show', [
            'prefactor' => $prefactor,
            'user' => $prefactor->user->safe,
            'orderGroup' => $prefactorsService->getPrefactorOrderGroup($prefactor),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Prefactor $prefactor, PrefactorsService $prefactorsService)
    {
        if ($prefactor == null || $prefactor->status == 'paid') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        return Inertia::render('Admin/Financial/Prefactors/Edit', [
            'prefactor' => $prefactor,
            'user' => $prefactor->user->safe,
            'order' => $prefactorsService->getPrefactorOrder($prefactor),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Prefactor $prefactor)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Prefactor $prefactor)
    {
        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        if ($prefactor->status != 'paid') {
            /** Report Action */
            $this->report(__("messages.report_prefactor_deleted", ['factor_id' => $prefactor->uuid]), 'user', $user);

            $prefactor->delete();
            $prefactor->order->delete();

            return redirect()->back()->with([__('messages.prefactor_deleted')]);
        }

        return redirect()->back()->withErrors([__('messages.unknown_error')]);
    }

    /**
     * add to prefactor
     */
    public function prefactor_add(Prefactor $prefactor, AddToCartRequest $request, CartService $cartService)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        if ($order->status != 'prefactor') {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get product
        $product = $inventory->product;

        // get count
        $count = $request->count;

        // handle count
        if ($count > $inventory->count) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }
        if ($inventory->max_sale != null && $count > $inventory->max_sale) {
            return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
        }
        if ($inventory->min_sale != null && $count < $inventory->min_sale) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => $inventory->min_sale])]);
        }

        // get inventory props
        $props = $inventory->props->toArray();

        // get store
        $store = $inventory->store;

        /** @var \App\Models\Consignment $consignment */
        $consignment = $order->consignments()->where('store_id', $store != null ? $store->id : null)->first();

        // handle consignment
        if ($consignment == null) {
            $consignment = $order->consignments()->create([
                'user_id' => $user->id,
                'store_id' => $store != null ? $store->id : null,
                'uuid' => 'cn-' . rand(100000000, 999999999),
                'status' => 'created',
            ]);
        }

        // handle discount tree
        $price = $inventory->get_final_price;
        if ($inventory->is_discount_valid) {
            foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                if ($count >= $dt['count']) {
                    $price = str_replace(',', '', $dt['price']);
                    break;
                }
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // handle payment gateway price limit
        $pgPriceLimit = $cartService->validatePaymentGatewayPrice($order, $count, $price);
        if ($pgPriceLimit) {
            return redirect()->back()->withErrors([__('messages.pg_price_limit_error', ['price' => number_format($pgPriceLimit)])]);
        }

        // check for exist consignment items
        $consignmentItem = $consignment->consignmentItems()->where('inventory_id', $inventory->id)->first();

        // add affiliate
        $affiliatesCookie = request()->cookie('affiliates');
        $affiliates = $affiliatesCookie != null ? unserialize($affiliatesCookie) : [];
        $affiliate = null;
        if (array_key_exists($product->uuid, $affiliates)) {
            $targetUser = User::where('id', $affiliates[$product->uuid])->first();
            if ($targetUser != null) {
                $affiliate = $targetUser->id;
            }
        }

        // add inventory to consignment item
        if ($consignmentItem == null) {
            $consignmentItem = $consignment->consignmentItems()->create([
                'store_id' => $store != null ? $store->id : null,
                'inventory_id' => $inventory->id,
                'product_id' => $inventory->product->id,
                'storage_id' => $inventory->storage->id,
                'title' => $inventory->product->title,
                'props' => serialize($props),
                'count' => $count,
                'price' => $price,
                'discount' => $discount,
                'affiliate' => $affiliate,
                'status' => $store != null ? 'awaiting_request_from_seller' : 'ready_to_send',
            ]);
        } else {
            // get new count
            $newCount = $consignmentItem->count + $count;

            // validate again max_sale and total inventory count
            if ($inventory->max_sale != null && $newCount > $inventory->max_sale) {
                return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
            }
            if ($newCount > $inventory->count) {
                return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
            }

            // get new price
            if ($inventory->is_discount_valid) {
                foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                    if ($newCount >= $dt['count']) {
                        $price = str_replace(',', '', $dt['price']);
                        break;
                    }
                }
            }

            // get new discount
            $discount = $inventory->price - $price;

            // update count and price and discount
            $consignmentItem->update([
                'count' => $newCount,
                'price' => $price,
                'discount' => $discount,
                'affiliate' => $affiliate,
            ]);
        }

        // get final count
        $finalCount = $consignment->consignmentItems()->where('inventory_id', $inventory->id)->first()->count;

        return redirect()->back()->with('add_to_cart', [
            'type' => 'added_to_cart',
            'data' => [
                'props' => $props,
                'count' => $finalCount,
                'price' => $price,
                'discount' => $discount,
            ],
        ]);
    }

    /**
     * delete from prefactor
     */
    public function prefactor_delete(Prefactor $prefactor, DeleteFromCartRequest $request)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        if ($order->status != 'prefactor') {
            return redirect()->back();
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // delete consignment item
        $consignmentItem->delete();

        return redirect()->back();
    }

    /**
     * increase 1 unit prefactor count
     */
    public function prefactor_increase(Prefactor $prefactor, ChangeCartCountRequest $request, CartService $cartService)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        if ($order->status != 'prefactor') {
            return redirect()->back();
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // check for inventory count
        if ($inventory->count == 0) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }

        // get product
        $product = $inventory->product;

        // get final count
        $finalCount = $consignmentItem->count + 1;

        // handle count
        if ($finalCount > $inventory->count) {
            return redirect()->back()->withErrors([__('messages.inventory_count_not_enough')]);
        }
        if ($inventory->max_sale != null && $finalCount > $inventory->max_sale) {
            return redirect()->back()->withErrors([__('messages.max_number_of_orders_must_be', ['count' => $inventory->max_sale])]);
        }

        // handle discount
        $price = $inventory->get_final_price;
        if ($inventory->is_discount_valid) {
            foreach (array_reverse($inventory->get_discount_tree) as $dt) {
                if ($finalCount >= $dt['count']) {
                    $price = str_replace(',', '', $dt['price']);
                    break;
                }
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // handle payment gateway price limit
        $pgPriceLimit = $cartService->validatePaymentGatewayPrice($order, 1, $price);
        if ($pgPriceLimit) {
            return redirect()->back()->withErrors([__('messages.pg_price_limit_error', ['price' => number_format($pgPriceLimit)])]);
        }

        // increase consignment item count
        $consignmentItem->update([
            'count' => $finalCount,
            'price' => $price,
            'discount' => $discount,
        ]);

        return redirect()->back();
    }

    /**
     * decrease 1 unit prefactor count
     */
    public function prefactor_decrease(Prefactor $prefactor, ChangeCartCountRequest $request)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        if ($order->status != 'prefactor') {
            return redirect()->back();
        }

        // get consignment item
        $consignmentItem = ConsignmentItem::where('id', $request->consignment_item_id)->first();
        if ($consignmentItem == null) {
            return redirect()->back();
        }

        // authorize user
        if ($consignmentItem->consignment->user_id != $user->id) {
            return redirect()->back();
        }

        // get inventory
        $inventory = Inventory::where('id', $request->inventory_id)->where('status', 'publish')->first();
        if ($inventory == null) {
            return redirect()->back()->withErrors([__('messages.inventory_not_exist')]);
        }

        // get final count
        $finalCount = $consignmentItem->count - 1;

        // handle count
        if ($finalCount < 1) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => 1])]);
        }
        if ($inventory->min_sale != null && $finalCount < $inventory->min_sale) {
            return redirect()->back()->withErrors([__('messages.min_number_of_orders_must_be', ['count' => $inventory->min_sale])]);
        }

        // handle discount
        $price = $inventory->get_final_price;
        foreach (array_reverse($inventory->get_discount_tree) as $dt) {
            if ($finalCount >= $dt['count']) {
                $price = str_replace(',', '', $dt['price']);
                break;
            }
        }

        // handle discount amount
        $discount = $inventory->price - $price;

        // reduce consignment item count
        $consignmentItem->update([
            'count' => $finalCount,
            'price' => $price,
            'discount' => $discount,
        ]);

        return redirect()->back();
    }

    /**
     * check discount
     */
    public function add_discount(Request $request, Prefactor $prefactor, CartService $cartService)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        $request->validate([
            'code' => 'required',
        ]);

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        if ($order == null) {
            return redirect()->back()->withErrors([__('messages.prefactor_not_found')]);
        }

        // get order consignment
        $consignmentItems = [];
        $consignments = [];
        $consignments = $order->consignments()->with('consignmentItems')->get();

        // add consignment items
        $consignmentItems = new Collection();
        foreach ($consignments as $consignment) {
            $consignmentItems = $consignmentItems->merge($consignment->consignmentItems()->get());
        }

        // validate empty cart
        if ($consignmentItems->count() == 0) {
            return redirect()->back()->withErrors([__('messages.prefactor_items_empty')]);
        }

        // clear prev discount
        $order->update([
            'discount' => null
        ]);

        // selected address
        $selectedAddress = $order->address != null ? unserialize($order->address) : null;

        // consignments information array
        $consignmentsInformation = $cartService->sepratedConsignmentInformation($order, $selectedAddress);

        // validate discount code
        $discount = Discount::where('code', $request->code)->where(function ($query) use ($user) {
            $query->where('for', $user->id)->orWhere('for', 'all');
        })->where('expire_at', '>=', Carbon::now()->toDateTimeString())->first();
        if ($discount == null) {
            return redirect()->back()->withErrors([__('messages.discount_code_is_invalid')]);
        }
        if ($discount->users()->where('user_id', $user->id)->count() >= $discount->max_use) {
            return redirect()->back()->withErrors([__('messages.discount_code_has_already_been_used')]);
        }

        // calculate discount amount
        $amount = 0;
        $discountPrice = ($consignmentsInformation['financial']['totalFinalPrice'] * $discount->percent) / 100;
        if ($discountPrice <= $discount->max_amount) {
            $amount = $discountPrice;
        } else {
            $amount = $discount->max_amount;
        }

        // add discount to order
        $discount = [
            'information' => [
                'id' => $discount->id,
                'title' => $discount->title,
                'code' => $discount->code,
            ],
            'amount' => $amount,
        ];
        $order->update([
            'discount' => serialize($discount)
        ]);

        return redirect()->back()->with('message', [__('messages.discount_applied')]);
    }

    /**
     * delete discount
     */
    public function delete_discount(Prefactor $prefactor)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        if ($order == null) {
            return redirect()->back()->withErrors([__('messages.prefactor_not_found')]);
        }

        $order->update([
            'discount' => null
        ]);

        return redirect()->back()->with('message', [__('messages.discount_deleted')]);
    }

    /**
     * issuance prefactor
     */
    public function issuance(Request $request, Prefactor $prefactor)
    {
        if ($prefactor == null || $prefactor->status != 'awaiting_issuance') {
            return redirect()->back()->withErrors([__('messages.unauthorized')]);
        }

        $request->validate([
            'expired_at' => 'required'
        ]);

        /** @var \App\Models\User $user */
        $user = $prefactor->user;

        /** @var \App\Models\Order $order */
        $order = $prefactor->order;

        $order = Order::whereHas('consignments.consignmentItems')->find($order->id);
        if (!$order) {
            return redirect()->back()->withErrors([__('messages.prefactor_items_empty')]);
        }

        $expiredAtTimestamp = null;
        if ($request->exists('expired_at') && $request->expired_at != '') {
            $expiredAtTimestamp = Jalalian::fromFormat('Y/m/d', $request->expired_at)->addDay()->getTimestamp();
            $expiredAtTimestamp = Jalalian::forge($expiredAtTimestamp)->toCarbon()->setTimezone('Asia/Tehran');
            $expiredAtTimestamp = $expiredAtTimestamp->format('Y-m-d H:i:s');
        }

        if (!Carbon::parse($expiredAtTimestamp)->greaterThan(Carbon::now())) {
            return redirect()->back()->withErrors([__('messages.expiration_date_must_be_greater_than_current_moment')]);
        }

        $prefactor->update([
            'status' => 'waiting_payment',
            'expired_at' => $expiredAtTimestamp
        ]);

        /** Report Action */
        $this->report(__("messages.report_prefactor_issued", ['factor_id' => $prefactor->uuid]), 'user', $user);

        // notification
        $user->notify(new PrefactorIssued($prefactor));

        return redirect()->back()->with('message', [__('messages.prefactor_issued')]);
    }
}
