<?php

namespace App\Http\Controllers\Admin\Financial;

use App\Helpers\UUIDHelper;
use App\Http\Controllers\Controller;
use App\Models\PendingPayment;
use App\Notifications\AffiliateRevenue;
use App\Notifications\SellerRevenue;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class PendingPaymentsController extends Controller
{
    use UUIDHelper;

    /**
     * pending payments list
     */
    public function index(Request $request)
    {
        /** @var LengthAwarePaginator */
        $payments = PendingPayment::query()
            ->when($request->input('search'), function ($query, $search) {
                $query->whereHas('order', function ($q2) use ($search) {
                    $q2->where('uuid', 'like', "%{$search}%");
                });
            })
            ->when($request->input('awaiting'), function ($query, $awaiting) {
                if ($awaiting) {
                    $query->where('status', 'awaiting');
                }
            })
            ->when($request->input('paid'), function ($query, $paid) {
                if ($paid) {
                    $query->where('status', 'paid');
                }
            })
            ->with(['order', 'user'])
            ->orderByRaw("
                CASE
                    WHEN status = 'awaiting' AND EXISTS (
                        SELECT 1 FROM orders
                        WHERE orders.id = pending_payments.order_id
                        AND orders.status = 'delivered'
                    ) THEN 0
                    WHEN status = 'awaiting' THEN 1
                    WHEN status = 'paid' THEN 2
                    ELSE 3
                END
            ")
            ->orderBy('paid_at', 'desc')
            ->paginate(20)
            ->through(fn($payment) => [
                'id' => $payment->id,
                'amount' => $payment->amount,
                'type' => $payment->type,
                'status' => $payment->status,
                'jalali_paid_at' => $payment->jalali_paid_at,
                'jalali_created_at' => $payment->jalali_created_at,

                'order' => [
                    'id' => $payment->order->id,
                    'uuid' => $payment->order->uuid,
                    'status' => $payment->order->status,
                ],
                'user' => [
                    'id' => $payment->user->id,
                    'get_name' => $payment->user->get_name,
                    'type' => $payment->user->type,
                ],
            ]);

        $payments->withQueryString();

        // statistics
        $totalAwaiting = PendingPayment::where('status', 'awaiting')->sum('amount');
        $totalPaid = PendingPayment::where('status', 'paid')->sum('amount');

        return Inertia::render('Admin/Financial/PendingPayments/List', [
            'payments' => $payments,
            'totalAwaiting' => $totalAwaiting,
            'totalPaid' => $totalPaid,
        ]);
    }

    /**
     * pay pending payment
     */
    public function pay(Request $request, PendingPayment $pendingPayment)
    {
        try {
            DB::transaction(function () use ($pendingPayment) {
                $pendingPayment->refresh();
                if ($pendingPayment->status !== 'awaiting' || $pendingPayment->paid_at !== null) {
                    throw new \Exception('payment already paid');
                }

                // check order status
                if ($pendingPayment->order->status != 'delivered') {
                    throw new \Exception('The date of the decree has not yet come');
                }

                // target user
                $user = $pendingPayment->user;

                // check target user wallet
                if (!$user->wallet) {
                    throw new \Exception(__('messages.wallet_not_found'));
                }

                // increase target user wallet amount
                $user->wallet->increment('amount', $pendingPayment->amount);

                // wallet chare transaction
                $transactionData = [
                    'amount' => $pendingPayment->amount,
                    'transaction_id' => $this->generateUniqueRandomNumber(9, \App\Models\Transaction::class, 'transaction_id'),
                    'status' => 'accepted',
                ];

                if ($pendingPayment->type === 'seller') {
                    $transactionData += [
                        'description' => __('messages.transaction_seller_revenue', [
                            'consignment' => $pendingPayment->get_data['consignment_uuid'] ?? '-'
                        ]),
                        'type' => 'seller_revenue',
                    ];

                    $user->transactions()->create($transactionData);

                    // send notification
                    $user->notify(new SellerRevenue(
                        $pendingPayment->get_data['consignment_uuid'] ?? null,
                        $pendingPayment->amount
                    ));
                } elseif ($pendingPayment->type === 'affiliate') {
                    $product = $pendingPayment->get_data['product'] ?? null;
                    $transactionData += [
                        'description' => __('messages.transaction_affiliate_revenue', [
                            'url' => $product ? route('main.products.show', $product['slug']) : '-',
                            'title' => $product['title'] ?? '-',
                        ]),
                        'type' => 'affiliate_revenue',
                    ];

                    $user->transactions()->create($transactionData);

                    // send notification
                    $user->notify(new AffiliateRevenue(
                        $pendingPayment->amount,
                        $product ? route('main.products.show', $product['slug']) : '#',
                        $product['title'] ?? '-'
                    ));
                }

                // update payment status to paid
                $pendingPayment->update([
                    'status' => 'paid',
                    'paid_at' => now(),
                ]);
            });
        } catch (\Exception $e) {
            return redirect()->back()->withErrors([__('messages.unknown_error')]);
        }

        return redirect()->back()->with('message', [__('messages.pending_payment_paid')]);
    }
}
