<?php

namespace App\Http\Controllers\Admin\Financial;

use App\Models\Discount;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\DiscountRequest;
use App\Models\User;
use App\Notifications\UserSingleDiscount;
use App\Services\Admin\Financial\DiscountService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class DiscountController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, DiscountService $discountService)
    {
        return Inertia::render('Admin/Financial/Discounts/List', [
            'discounts' => $discountService->getDiscounts($request),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(DiscountRequest $request, DiscountService $discountService)
    {
        if ($discountService->getFor($request) == 'redirect_back')
            return redirect()->back()->withErrors([__('messages.user_not_found')]);

        if ($discountService->getPercent($request) == 'redirect_back')
            return redirect()->back()->withErrors([__('messages.percent_must_between_0_to_100')]);

        if ($discountService->getExpireAt($request) == null) {
            return redirect()->back()->withErrors([__('messages.expire_at_incorrect')]);
        }

        $discount = Discount::create([
            'for' => $discountService->getFor($request),
            'title' => $request->title,
            'code' => $request->code,
            'expire_at' => $discountService->getExpireAt($request),
            'max_use' => $request->max_use,
            'percent' => $discountService->getPercent($request),
            'max_amount' => str_replace(',', '', $request->max_amount),
            'status' => 'active',
        ]);

        // get discount user
        $user = null;
        if ($discount != 'all') {
            $user = User::find($discount->for);
        }

        // notification
        if ($user != null) {
            $user->notify(new UserSingleDiscount($discount));
        }

        /** Report Action */
        $this->report(__("messages.report_discount_created", ['discount' => $request->code]));

        return redirect()->back()->with('message', [__('messages.discount_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Discount $discount)
    {
        $users = $discount->users()->orderBy('created_at', 'desc')->distinct()->paginate(20)->through(fn($user) => [
            'user' => $user,
            'count' => $discount->users()->where('user_id', $user->id)->count()
        ]);

        return Inertia::render('Admin/Financial/Discounts/Usage', [
            'discount' => $discount,
            'users' => $users,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Discount $discount)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(DiscountRequest $request, Discount $discount, DiscountService $discountService)
    {
        if ($discountService->getFor($request, $discount) == 'redirect_back')
            return redirect()->back()->withErrors([__('messages.user_not_found')]);

        if ($discountService->getPercent($request, $discount) == 'redirect_back')
            return redirect()->back()->withErrors([__('messages.percent_must_between_0_to_100')]);

        if ($discountService->getExpireAt($request) == null) {
            return redirect()->back()->withErrors([__('messages.expire_at_incorrect')]);
        }

        $discount->update([
            'for' => $discountService->getFor($request, $discount),
            'title' => $request->title,
            'code' => $request->code,
            'max_use' => $request->max_use,
            'percent' => $discountService->getPercent($request, $discount),
            'max_amount' => str_replace(',', '', $request->max_amount),
        ]);

        Discount::where('id', $discount->id)->update([
            'expire_at' => $discountService->getExpireAt($request, $discount),
        ]);

        /** Report Action */
        $this->report(__("messages.report_discount_updated", ['discount' => $request->code]));

        return redirect()->back()->with('message', [__('messages.discount_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Discount $discount)
    {
        $discount->delete();

        /** Report Action */
        $this->report(__("messages.report_discount_deleted", ['discount' => $discount->code]));

        return redirect()->back()->with('message', [__('messages.discount_deleted')]);
    }
}
