<?php

namespace App\Http\Controllers\Admin\Blog;

use App\Helpers\DropdownListHelper;
use App\Helpers\UUIDHelper;
use App\Models\Article;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ArticleCheckRequest;
use App\Http\Requests\Admin\ArticlesRequest;
use App\Models\ArticleCategory;
use App\Models\ArticleTag;
use App\Models\Brand;
use App\Notifications\ArticleStatus;
use App\Services\Admin\Blog\ArticlesService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ArticlesController extends Controller
{
    use DropdownListHelper, UUIDHelper;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request, ArticlesService $articlesService)
    {
        return Inertia::render('Admin/Blog/Articles/List', [
            'articles' => $articlesService->getArticles($request, "all")
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = ArticleCategory::where('status', 'publish')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
        ]);

        return Inertia::render('Admin/Blog/Articles/Create', [
            'categories' => $categories,
            'statusList' => $this->contentStatusList(),
            'articleTypeList' => $this->articleTypeList(),
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(ArticlesRequest $request, ArticlesService $articlesService)
    {
        /** current user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        /** validate duplicated slug */
        if (!$articlesService->validateSlug($request))
            return redirect()->back()->withErrors([__('messages.slug_is_exist')]);

        /** validate source */
        $source = $articlesService->validateSource($request);

        /** validate type */
        $type = $articlesService->validateType($request);

        /** validate brand */
        $brand = Brand::where('title', $request->brand)->first();

        $article = $user->articles()->create([
            'edited_by' => $user->id,
            'brand_id' => $brand != null ? $brand->id : null,
            'category_id' => $request->category['id'],
            'uuid' => $this->generateUniqueRandomNumber(8, \App\Models\Article::class, 'uuid', 'a-'),
            'title' => $request->title,
            'slug' => $request->has('slug') ? str_replace('/', '-', str_replace('\\', '-', str_replace('%', '', $request->slug))) : null,
            'content' => $request->exists('content') ? $request->content : null,
            'source' => $source,
            'read_time' => $request->exists('read_time') ? $request->read_time : null,
            'type' => $type,

            'image' => $request->exists('image') ? $request->image : null,
            'video' => $request->exists('video') ? $request->video : null,
            'podcast' => $request->exists('podcast') ? $request->podcast : null,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : null,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : null,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : null,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : null,

            'editor_choice' => $request->editor_choice,
            'comment_status' => $request->comment_status,

            'status' => $request->status,
        ]);

        /** sync tags */
        $articlesService->syncTags($request, $article);

        /** Report Action */
        $this->report(__("messages.report_article_created"), 'article', $article);

        return redirect()->route('admin.articles.index')->with('message', [__('messages.article_created')]);
    }

    /**
     * Display the specified resource.
     */
    public function show(Article $article)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Article $article)
    {
        if ($article->status == 'rejected')
            return redirect()->route('admin.articles.index')->withErrors([__('messages.unauthorized')]);

        $categories = ArticleCategory::where('status', 'publish')->get()->map(fn($category) => [
            'id' => $category->id,
            'title' => $category->title,
            'slug' => $category->slug,
        ]);

        return Inertia::render('Admin/Blog/Articles/Edit', [
            'article' => [
                'id' => $article->id,
                'user_id' => $article->user_id,
                'category_id' => $article->category_id,
                'title' => $article->title,
                'slug' => $article->slug,
                'uuid' => $article->uuid,
                'get_image' => $article->get_image,
                'get_tags_string' => $article->get_tags_string,
                'content' => $article->content,
                'source' => $article->source,
                'read_time' => $article->read_time,
                'type' => $article->type,
                'image' => $article->image,
                'video' => $article->video,
                'podcast' => $article->podcast,
                'seo_title' => $article->seo_title,
                'seo_description' => $article->seo_description,
                'seo_keywords' => $article->seo_keywords,
                'seo_canonical' => $article->seo_canonical,
                'comment_status' => $article->comment_status,
                'editor_choice' => $article->editor_choice,
                'reason' => $article->reason,
                'status' => $article->status,
                'brand' => $article->brand != null ? [
                    'id' => $article->brand->id,
                    'title' => $article->brand->title,
                ] : null,
            ],
            'categories' => $categories,
            'statusList' => $this->contentStatusList(),
            'articleTypeList' => $this->articleTypeList(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(ArticlesRequest $request, Article $article, ArticlesService $articlesService)
    {
        /** current user
         * @var \App\Models\User $user
         */
        $user = auth()->user();

        /** article initial status */
        $articleStatus = $article->status;

        /** validate duplicated slug */
        if (!$articlesService->validateSlug($request, $article))
            return redirect()->back()->withErrors([__('messages.slug_is_exist')]);

        /** validate source */
        $source = $articlesService->validateSource($request, $article);

        /** validate type */
        $type = $articlesService->validateType($request, $article);

        /** validate brand */
        $brand = Brand::where('title', $request->brand)->first();

        $article->update([
            'edited_by' => $user->id,
            'brand_id' => $brand != null ? $brand->id : $article->brand_id,
            'category_id' => $request->category['id'],
            'title' => $request->title,
            'slug' => $request->has('slug') ? str_replace('/', '-', str_replace('\\', '-', str_replace('%', '', $request->slug))) : $article->slug,
            'content' => $request->content,
            'source' => $source,
            'read_time' => $request->exists('read_time') ? $request->read_time : $article->read_time,
            'type' => $type,

            'image' => $request->exists('image') ? $request->image : $article->image,
            'video' => $request->exists('video') ? $request->video : $article->video,
            'podcast' => $request->exists('podcast') ? $request->podcast : $article->podcast,

            'seo_title' => $request->exists('seo_title') ? $request->seo_title : $article->seo_title,
            'seo_description' => $request->exists('seo_description') ? $request->seo_description : $article->seo_description,
            'seo_keywords' => $request->exists('seo_keywords') ? $request->seo_keywords : $article->seo_keywords,
            'seo_canonical' => $request->exists('seo_canonical') ? urldecode($request->seo_canonical) : $article->seo_canonical,

            'editor_choice' => $request->editor_choice,
            'comment_status' => $request->comment_status,

            'status' => $request->status,
        ]);

        /** sync tags */
        $articlesService->syncTags($request, $article);

        if ($articleStatus == 'awaiting') {
            /** Report Action */
            $this->report(__("messages.report_article_accepted"), 'article', $article);
        } else {
            /** Report Action */
            $this->report(__("messages.report_article_updated"), 'article', $article);
        }

        return redirect()->route('admin.articles.edit', $article->slug)->with('message', [__('messages.article_updated')]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Request $request, Article $article)
    {
        if ($article->status == 'deleted' && $request->status == 'deep_deleted') {
            $article->delete();

            /** Report Action */
            $this->report(__("messages.report_article_deleted"), 'article', $article);

            return redirect()->back()->with('message', [__('messages.article_deleted')]);
        } else if ($article->status != 'deleted' && $request->status == 'deleted') {
            $article->update([
                'status' => 'deleted'
            ]);

            /** Report Action */
            $this->report(__("messages.report_article_to_trash"), 'article', $article);

            return redirect()->back()->with('message', [__('messages.article_to_trash')]);
        } else if ($article->status != 'deleted' && $request->status == 'draft') {
            $article->update([
                'status' => 'draft'
            ]);

            /** Report Action */
            $this->report(__("messages.report_article_draft"), 'article', $article);

            return redirect()->back()->with('message', [__('messages.article_draft')]);
        } else if ($request->status == 'publish') {
            $article->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_article_publish"), 'article', $article);

            return redirect()->back()->with('message', [__('messages.article_publish')]);
        }

        return redirect()->route('admin.articles.index')->withErrors([__('messages.unknown_error')]);
    }

    /**
     * articles trash list
     */
    public function trash(Request $request)
    {
        /** @var LengthAwarePaginator */
        $articles = Article::query()->where('status', 'deleted')->when($request->input('search'), function ($query, $search) {
            $query->where('title', 'like', "%{$search}%");
        })->orderBy('created_at', 'desc')->paginate(20)->through(fn($article) => [
            'id' => $article->id,
            'user_id' => $article->user_id,
            'uuid' => $article->uuid,
            'title' => $article->title,
            'slug' => $article->slug,
            'get_image' => $article->get_image,
            'editor_choice' => $article->editor_choice,
            'get_formatted_view_count' => $article->get_formatted_view_count,
            'comments_info' => $article->comments_info,
            'vote' => $article->vote,
            'jalali_updated_at' => $article->jalali_updated_at,
            'status' => $article->status,
            'user' => [
                'id' => $article->user->id,
                'get_name' => $article->user->get_name,
                'type' => $article->user->type,
                'seller' => $article->user->seller,
                'store' => $article->user->store != null ? [
                    'id' => $article->user->store->id,
                    'uuid' => $article->user->store->uuid,
                    'name' => $article->user->store->name,
                ] : null,
            ],
        ]);
        $articles->withQueryString();

        return Inertia::render('Admin/Blog/Articles/Trash', [
            'articles' => $articles
        ]);
    }

    /**
     * accept or reject awaiting products
     */
    public function check(ArticleCheckRequest $request, Article $article)
    {
        if ($article->status != 'awaiting')
            return redirect()->route('admin.articles.index')->withErrors([__('messages.unauthorized')]);

        if ($request->action == 'accept') {
            $article->update([
                'status' => 'publish'
            ]);

            /** Report Action */
            $this->report(__("messages.report_article_accepted"), 'article', $article);

            // notification
            $article->user->notify(new ArticleStatus($article, 'accept'), true);

            // update article
            $articlesRequest = ArticlesRequest::createFrom($request);
            $articlesRequest->setMethod('POST');
            $articlesService = app(ArticlesService::class);
            $this->update($articlesRequest, $article, $articlesService);

            return redirect()->back()->with('message', [__('messages.article_accepted')]);
        } else if ($request->action == 'reject') {
            $request->validate([
                'reason' => 'required'
            ]);

            $article->update([
                'reason' => $request->reason,
                'status' => 'rejected'
            ]);

            /** Report Action */
            $this->report(__("messages.report_article_rejected"), 'article', $article);

            // notification
            $article->user->notify(new ArticleStatus($article, 'reject'), true);

            return redirect()->back()->with('message', [__('messages.article_rejected')]);
        }

        return redirect()->back()->withErrors([__('messages.unknown_error')]);
    }
}
