<?php

namespace App\Http\Controllers\Admin\Blog;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\ArticleCommentsRequest;
use App\Models\Article;
use App\Models\ArticleComment;
use App\Services\Admin\Blog\ArticleCommentsService;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ArticleCommentsController extends Controller
{
    /**
     * article comment list
     */
    public function index(Request $request, ArticleCommentsService $articleCommentsService)
    {
        return Inertia::render('Admin/Blog/Comments/List', [
            'comments' => $articleCommentsService->getComments($request)
        ]);
    }

    /**
     * accept comment
     */
    public function accept(ArticleComment $comment)
    {
        if ($comment->status == 'awaiting') {
            $comment->update([
                'status' => 'publish',
            ]);

            /** Report Action */
            $this->report(__("messages.report_article_comment_accepted"), 'article', $comment->article);

            return redirect()->back()->with('message', [__('messages.comment_accepted')]);
        }

        return redirect()->back();
    }

    /**
     * edit comment
     */
    public function edit(ArticleCommentsRequest $request, ArticleComment $comment)
    {
        $comment->update([
            'content' => $request->validated('content'),
            'vote_up' => $request->exists('vote_up') && $request->vote_up != null ? $request->vote_up : $comment->vote_up,
            'vote_down' => $request->exists('vote_down') && $request->vote_down != null ? $request->vote_down : $comment->vote_down,
            'status' => 'publish',
        ]);

        /** Report Action */
        $this->report(__("messages.report_article_comment_edited"), 'article', $comment->article);

        return redirect()->back()->with('message', [__('messages.comment_edited')]);
    }

    /**
     * send comment answer
     */
    public function send_answer(ArticleCommentsRequest $request, Article $article, ArticleComment $comment)
    {
        /** @var \App\Models\User $user */
        $user = auth()->user();

        $user->articleComments()->create([
            'article_id' => $article->id,
            'content' => $request->validated('content'),
            'reply_to' => $comment->id,
            'status' => 'publish'
        ]);

        $comment->update([
            'status' => 'publish'
        ]);

        /** Report Action */
        $this->report(__("messages.report_article_comment_send_answer"), 'article', $comment->article);

        return redirect()->back()->with('message', [__('messages.comment_answer_send')]);
    }

    /**
     * delete comment
     */
    public function delete(ArticleComment $comment)
    {
        $comment->delete();

        /** Report Action */
        $this->report(__("messages.report_article_comment_deleted"), 'article', $comment->article);

        return redirect()->back()->with('message', [__('messages.comment_deleted')]);
    }
}
