<?php

namespace App\Exports;

use App\Models\User;
use Illuminate\Database\Eloquent\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class UsersExport implements FromCollection, WithHeadings, WithStyles
{
    protected $type;

    public function __construct($type)
    {
        $this->type = $type;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $users = User::with('orders')->with('wallet')->with('scores')->with('addresses');

        if ($this->type == 'with_order') {
            $users = $users->whereHas('orders', function ($query) {
                $query->where('status', 'registered')->orWhere('status', 'sent')->orWhere('status', 'delivered');
            });
        }

        $users = $users->orderBy('created_at', 'desc')->get();

        $result = new Collection();
        foreach ($users as $user) {
            $address = $user->addresses()->where('active', true)->first();

            $result->add([
                'name' => $user->get_name,
                'phone' => $user->phone,
                'email' => $user->email != null ? $user->email : 'خالی',
                'username' => $user->username,
                'national_number' => $user->national_number != null ? $user->national_number : 'خالی',
                'birthday' => $user->birthday != null ? $user->jalali_birthday[1] : 'خالی',
                'gender' => $user->gender == 'man' ? 'مرد' : ($user->gender == 'woman' ? 'زن' : 'مشخص نشده'),
                'wallet' => number_format($user->wallet->amount),
                'type' => $user->type == 'owner' ? 'مالک' : ($user->type == 'admin' ? $user->role['label'] : ($user->seller == true ? 'فروشنده' : 'کاربر')),
                'order_count' => $user->orders()->count(),
                'address' => $address != null ? implode(', ', [
                    'state' => $address->state,
                    'city' => $address->city,
                    'full_address' => $address->full_address,
                ]) : 'خالی',
                'score' => $user->get_score['formatted_score'],
                'level' => $user->get_level == 'bronze' ? 'برنزی' : ($user->get_level == 'silver' ? 'نقره ای' : ($user->get_level == 'gold' ? 'طلایی' : '')),
                'status' => $user->status == 'active' ? 'فعال' : ($user->status == 'banned' ? 'مسدود شده' : ''),
                'created_at' => $user->jalali_created_at[1],
            ]);
        }

        return $result;
    }

    public function headings(): array
    {
        return [
            'نام',
            'شماره همراه',
            'ایمیل',
            'نام کاربری',
            'کد ملی',
            'تاریخ تولد',
            'جنسیت',
            'کیف پول (تومان)',
            'سطح کاربری',
            'تعداد سفارش',
            'آدرس',
            'امتیاز باشگاه مشتریان',
            'سطح باشگاه مشتریان',
            'وضعیت',
            'تاریخ ثبت نام',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        $sheet->setRightToLeft(true);

        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col);
            $sheet->getColumnDimension($columnLetter)->setAutoSize(true);
        }

        $highestRow = $sheet->getHighestRow();
        $sheet->getStyle("A1:{$highestColumn}{$highestRow}")->getFont()->setName('Tahoma')->setSize(11);
    }
}
