<?php

namespace App\Exports;

use App\Models\Product;
use App\Models\Storage;
use Illuminate\Database\Eloquent\Collection;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithStyles;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;

class StorageExport implements FromCollection, WithHeadings, WithStyles
{
    protected Storage $storage;
    protected $type;
    protected $category;
    protected $brand;

    public function __construct(Storage $storage, $type, $category, $brand)
    {
        $this->storage = $storage;
        $this->type = $type;
        $this->category = $category;
        $this->brand = $brand;
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        $storage = $this->storage;
        $type = $this->type;
        $category = $this->category;
        $brand = $this->brand;

        $products = Product::with('inventories');

        // inventory type filter
        if ($type == 'instock') {
            $products = $products->whereHas('inventories', function ($query) use ($storage) {
                $query->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0);
            });
        } else if ($type == 'outofstock') {
            $products = $products->whereHas('inventories', function ($query) use ($storage) {
                $query->where('storage_id', $storage->id)->where('status', 'publish');
            })->whereDoesntHave('inventories', function ($query) use ($storage) {
                $query->where('storage_id', $storage->id)->where('status', 'publish')->where('count', '>', 0);
            });
        }

        // category filter
        if ($category != 'all' && $category != null && $category != '') {
            $products = $products->where('category_id', $category);
        }

        // brand filter
        if ($brand != 'all' && $brand != null && $brand != '') {
            $products = $products->where('brand_id', $brand);
        }

        // order by desc
        $products = $products->orderBy('created_at', 'desc')->get();

        // add inventories to final list
        $result = new Collection();
        foreach ($products as $product) {
            $inventories = $product->inventories()->where('storage_id', $storage->id)->get();
            foreach ($inventories as $inventory) {
                $result->add([
                    'product_uuid' => $inventory->product->uuid,
                    'title' => $inventory->product->title,
                    'inventory_uuid' => $inventory->uuid,
                    'props_key' => collect($inventory->props)->pluck('name')->join(', '),
                    'props_value' => collect($inventory->props)->map(function ($item) {
                        return $item['type'] === 'color' ? $item['value']['label'] : $item['value'];
                    })->join(', '),
                    'price' => $inventory->price,
                    'discount' => $inventory->discount_price,
                    'discount_expire' => $inventory->discount_price != null ? ($inventory->discount_expire != null ? $inventory->jalali_discount_expire[1] : null) : null,
                    'count' => $inventory->count > 0 ? $inventory->count : 0,
                    'purchase_price' => $inventory->purchase_price > 0 ? $inventory->purchase_price : null,
                    'sales_method' => $inventory->product->in_stock_status ? 'فروش اینترنتی' : 'استعلام موجودی',
                    'last_update' => $inventory->jalali_updated_at[2],
                ]);
            }
        }

        return $result;
    }

    public function headings(): array
    {
        return [
            'شناسه محصول',
            'نام محصول',
            'شناسه تنوع',
            'نوع متغیر ها',
            'مقدار متغیر ها',
            'قیمت (تومان)',
            'قیمت بعد تخفیف (تومان)',
            'انقضای تخفیف',
            'تعداد موجودی',
            'قیمت خرید',
            'نوع فروش',
            'آخرین ویرایش',
        ];
    }

    public function styles(Worksheet $sheet)
    {
        $sheet->setRightToLeft(true);

        $highestColumn = $sheet->getHighestColumn();
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col);
            $sheet->getColumnDimension($columnLetter)->setAutoSize(true);
        }

        $highestRow = $sheet->getHighestRow();
        $sheet->getStyle("A1:{$highestColumn}{$highestRow}")->getFont()->setName('Tahoma')->setSize(11);
    }
}
